function	[V,F] = make_cylinder(X,dN,dr)
% Make cylinder around set of points which represent sequential line
%  [V,F] = make_cylinder(X,dN,dr)
% X  : set of points 3D coordinates [N x 3]
% dN : number of points consist cylinder
%      = 6 means hexagonal cylinder will be made
% dr : radius of cylinder (same unit with X)
%
% 2014-11-10 Masa-aki Sato
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%ʬ˱ä dN ꡢpatch

[N, d] = size(X);
if d ~= 3, error('data dimension error'); end

% dx : line segment direction [N x 3]
dx = diff(X);
dx = [dx; dx(end,:)];

[dY,dZ] = make_orthogonal_vec_old(dx);

% incremental angle of cylindrical points
ang = 2*pi/dN;
da  = ang*(0:dN-1);
% [Y,Z] coordinate of cylindrical points
y = cos(da);
z = sin(da);

% make dN points for each N points
V   = zeros(dN*N,3);
xyz = zeros(N,3,dN);

% make cylindrical points around X with radius dr
for m=1:dN
	xyz(:,:,m) = X + dr*(dY*y(m) + dZ*z(m));
end

% find nearest point from xyz(n,:,1) to xyz(n+1,:,:)
% which is cylindrical points in next X(n+1,:)
% and reorder the next cylindrical points xyz(n+1,:,:)
for n=1:N-1
	xn = xyz(n,:,1);  % current 1st point
	xp = xyz(n+1,:,:);% next cylindrical points
	xp = reshape(xp,[3,dN]);
	
	% distance from xyz(n,:,1) to xyz(n+1,:,:)
	dp = vb_repadd(xp, -xn(:));
	dd = sum(dp.^2,1);
	% nearest point index
	[tmp, imin] = min(dd);
	
	% reorder xyz(n+1,:,:)
	ind = imin + (0:dN-1);
	ind = mod(ind-1,dN) + 1;
	xyz(n+1,:,:) = xyz(n+1,:,ind);
end

% cylindrical points of X(n,:)
% : xyz(n,:,m) 
% = V( dN*(n-1) + m, :)
nj = 0:dN:(dN*(N-1));

for m=1:dN
	V(nj + m,:) = xyz(:,:,m);
end

%--- Example of vertex & patch
%  1  2  3  1
%  |/ |/ |/ |
%  4  5  6  4
%---  n=1, dN=3
%  F = [n n+1 n+dN; 
%         n+1 n+dN n+1+dN] 
% There are 2*dN patches for pair of consecutive cylindrical points
F  = zeros(2*dN*(N-1),3);

% make patch index
for i=1:N-1
	np = dN*(i-1);  % start point index
	nf = 2*dN*(i-1);% start patch index
	
	for n=1:dN
		if n==dN, 
			m = 1;
		else
			m = n+1;
		end
		
		%  F = [n n+1 n+dN; 
		%         n+1 n+dN n+1+dN] 
		F(nf+n,:)    = [n   m    n+dN] + np;
		F(nf+dN+n,:) = [m   m+dN n+dN] + np;
	end
end
