function dmri_fiber_tck_file_for_display_worker_start(parm_mat_file, worker_number)
% Process tck files.
%
% [Usage]
%    dmri_fiber_tck_file_for_display_worker_start(parm_mat_file, worker_number);
%
% [Input]
%    parm_mat_file : tck file processing parameter.
%          (variables in the file)
%                    work_dir : directory of processing files.
%                  fa_gz_file : FA nifti-gz file(.nii.gz) Fractional Anisotropy image.
%            fa_brain_gz_file : Freesurfer nifti-gz file(.nii.gz) Cortex image.
%              trans_info_dir : transform information directory.
%    worker_number : This is a number given by manager program
%
% [Output]
%    none
%
% [Note]
%    Many workers(MATLAB) concurrently launched to process tck files.
%    So, synchronous processing is required.
%    To realize it, directory is used as a exclusion access control.
%
%    1.Fiber tracking program creates directory.
%         work_dir/finished/parcelA-B.tck
%    2.Worker program try to delete the directory.
%      if success, it means that the worker got a authority to 
%      process the tck file.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Worker create worker directory to notify a manager to ready to process.
% Note: The directory will be deleted by manager.

load(parm_mat_file);
worker_dir = fullfile(work_dir, ['worker', num2str(worker_number)]);
mkdir(worker_dir);
log_file = fullfile(worker_dir, 'log.txt');
diary_flush(log_file);

try

% track file 
finished_dir   = fullfile(work_dir, filesep, 'finished');
processed_dir  = fullfile(work_dir, filesep, 'processed');
warp_file = fullfile(trans_info_dir, 'fa2freesurfer_warp.nii.gz');

% start processing
while(exist(worker_dir, 'dir') == 7)

    d = dir(finished_dir);

    for k=1:length(d)
        name = d(k).name;
        if strcmp(name, '.') || strcmp(name, '..')
            continue;
        end
        % To obtain an authority to process tck file, try to make a directory.
        % if the return value is success, the worker got an authority.
        [ret, warn] = mkdir(fullfile(processed_dir, filesep, name));
        % (mkdir always retuns 1.... second ret value check)

        % Get process authority
        if isempty(warn)
            % Success
            log_write(sprintf('%s.tck processing start\n', name));
            % Process tck file.
            base_name = name;
            tck_filename  = fullfile(work_dir, [base_name, '.tck']);

            % FA coord file
            coord_file    = fullfile(worker_dir, ['coord_', base_name, '.txt']);
            % freesurfer coordinate file
            fs_coord_file = fullfile(worker_dir, ['fs_coord_', base_name, '.txt']);

            if exist(tck_filename, 'file') ~= 2
                error(sprintf('%s not found.\n', tck_filename));
            end
            % processing
            vtrack  = [];
            vtracks = [];
            try
            vtrack = read_mrtrix_tracks(tck_filename);
            catch
            end
            if ~isempty(vtrack) && ~isempty(vtrack.data)
                ave_data = calc_mean_streamlines({vtrack.data});
                dlmwrite(coord_file, ave_data, 'delimiter', ' ', 'precision', 8);

                % Non-linear registration of stream lines
                cmd = ['img2imgcoord -mm', ...
                       ' -src ', fa_gz_file, ...
                       ' -dest ', fs_brain_gz_file, ...
                       ' -warp ', warp_file, ...
                       ' ', coord_file, ...
                       '| sed ''$d'' > ' fs_coord_file];
                dmri_system(cmd);

                % Read coordinate values in .txt file
                vtracks = dlmread(fs_coord_file, '', 1, 0);
            end

            % Write mat file
            fs_matfile = fullfile(work_dir, ['fs_', base_name, '.mat']);
            save(fs_matfile, 'vtracks');

            log_write(sprintf('%s.tck processing end\n', name));

            % delete processed name from tck list
            ret = rmdir(fullfile(finished_dir, filesep, name));
            diary_flush(log_file);
        else
            % Failure
            %log_write(sprintf('Failed to get a process authority %s.tck\n', name));
        end
        % Terminate
        if exist(worker_dir, 'dir') ~= 7, break; end
        pause(0.01);
    end
    pause(1);
end
catch
    errinfo = lasterror;
    log_write(errinfo.message);
    diary_flush;
    rethrow(errinfo);
end
diary_flush;
exit;

function log_write(message)
   fprintf('%s:%s', datestr(now, 31), message);
   
function diary_flush(diary_file)
    diary off;
    if exist('diary_file', 'var') && ~isempty(diary_file)
        diary(diary_file);
    end
 