function dmri_parcel_view(freesurfer_dir, parcel_dir, hemi)
% Launch parcellation viewer.
%
% [Usage]
%    dmri_parcel_view(freesurfer_dir, parcel_dir, hemi);
%
% [Input]
%    freesurfer_dir : freesurfer directory.
%        parcel_dir : parcel directory.
%              hemi : 'l' or 'r'
%                    = 'l' : left hemisphere. [default]
%                    = 'r' : right hemisphere.
%
% [Output]
%    none
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin < 2
    error('Please check input arguments.');
end
if exist(freesurfer_dir, 'dir') ~= 7
    error('Specified freesurfer_dir not found.');
end
if exist(parcel_dir, 'dir') ~= 7
    error('Specified parcel_dir not found.');
end

if ~exist('hemi', 'var')
    hemi = 'l';
end

%
% --- Main Procedure
%
disp('Reading Left/Right parcellation area...');
% Create color table file of parcels for left/right hemisphere.
[lh_clut_file, rh_clut_file, Nlabel_l, Nlabel_r] = ...
    dmri_color_table_create_for_parcels(parcel_dir, parcel_dir);

% Create annot file
[FS_SUBJECTS_ROOT, SUBJ] = fileparts(vb_rm_trailing_slash(freesurfer_dir));

switch(hemi)
    case 'l'
        clut_file = lh_clut_file;
        hemi      = 'lh';
        angle     = [-90 0];
        script_file = fullfile(parcel_dir, filesep, 'view_left_parcel.sh');
    case 'r'
        clut_file = rh_clut_file;
        hemi      = 'rh';
        angle     = [90, 0];
        script_file = fullfile(parcel_dir, filesep, 'view_right_parcel.sh');
    otherwise
        error('Unknown hemisphere type was specified. Please specify ''l'' or ''r''.');
end

label_dir      = fullfile(parcel_dir, filesep, 'parcels_label');
tmp_label_dir  = tempname(label_dir);

% Create symbolic link
if exist('/usr/bin/rename.ul', 'file')
    rename = 'rename.ul'; % Ubuntu Debian
else
    rename = 'rename';    % other distribution
end


cmd0_1 = ['ln -s ', label_dir, '/*.label', ...
        ' ', tmp_label_dir];
cmd0_2 = [rename ' parcel lh.parcel parcel{1..' num2str(Nlabel_l), '}.label'];
cmd0_3 = [rename ' parcel rh.parcel parcel{' num2str(Nlabel_l+1), '..', num2str(Nlabel_l+Nlabel_r) '}.label'];

cmd1 = ['mris_label2annot', ...
       ' --ctab ', clut_file, ...
       ' --ldir ', tmp_label_dir, ...
       ' --s ', SUBJ, ...
       ' --a parcel2000', ...
       ' --h ', hemi, ...
       ' --no-unknown'];

dmri_script_file_create(script_file);
fid = fopen(script_file, 'a');
if fid == -1
    error('Cannot create view_parcel script file.');
end
fprintf(fid, '\n%s\n', ['SUBJECTS_DIR=' FS_SUBJECTS_ROOT]);
fprintf(fid, '\n## PREPROCESS\n');
fprintf(fid, '# Create symbolic link to label files\n');
fprintf(fid, '%s\n', ['mkdir ', tmp_label_dir]);
fprintf(fid, '%s\n', cmd0_1);
fprintf(fid, '\n## Add prefix lh/rh to parcelN.label\n');
fprintf(fid, '%s\n', ['cd ', tmp_label_dir]);
fprintf(fid, '%s\n', cmd0_2);
fprintf(fid, '%s\n', cmd0_3);

fprintf(fid, '\n## Create annotation file\n');
fprintf(fid, 'rm -rf %s\n', fullfile(freesurfer_dir, filesep, 'label/lh.parcel2000.annot'));
fprintf(fid, 'rm -rf %s\n', fullfile(freesurfer_dir, filesep, 'label/rh.parcel2000.annot'));
fprintf(fid, '%s\n', cmd1);
fprintf(fid, '%s\n', ['rm -rf ', tmp_label_dir]);
fclose(fid);


%
% --- Create annotation file and confirm display
%
[status, msg] = system(script_file);
if status ~= 0
    error(msg);
end

% Load inflate data
h = figure;
inflated_file = fullfile(freesurfer_dir, 'surf',  [hemi, '.inflated']);
aparc_file    = fullfile(freesurfer_dir, 'label', [hemi, '.parcel2000.annot']);

[V, F]     = read_surf(inflated_file);
[tmp, col] = read_annotation(aparc_file);

msh.vertices = V;
msh.faces    = F + 1; % MATLAB patch requires 1 start
msh.FaceVertexCdata = fscolor2rgb(col);

p = patch(msh);
set(p,'FaceColor','flat','EdgeColor','none');
set(h, 'Color', 'k');
set(gca, 'Color', 'k');

view(angle);
axis tight;
axis equal;
rotate3d;
