function dmri_vbmeg_vertex_select(brain_file, fs_info_file, num_vertex, parcel_file)
% Choose verticies of VBMEG brain model by specifing num_vertex.
%
% [Usage]
%    dmri_vbmeg_vertex_select(brain_file, fs_info_file, num_vertex, parcel_file);
%
% [Input]
%      brain_file : VBMEG BRAIN-MAT file(.brain.mat)
%    fs_info_file : Freesurfer information file.
%      num_vertex : The number of verticies.
%     parcel_file : Parcel file.
%
% [Output]
%    none
%
% [Example]
%    
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin ~= 4
    error('Please check input arguments');
end
if exist(brain_file, 'file') ~= 2
    error('brain_file not found.');
end
if exist(fs_info_file, 'file') ~= 2
    error('fs_info_file not found.');
end
if num_vertex <= 0
    error('Invalid num_vertex was specified.');
end
if ~ischar(parcel_file)
    error('parcel_file is not a filename.');
end

%
% --- Main Procedure
%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Select vertices on standard brain
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp(['Select ' num2str(num_vertex) ' vertices on standard brain.']);
std_brain_file = which('mni_icbm152_t1_tal_nlin_asym_09c_20000.brain.mat');

% Select specified number of verticies on the standard brain file.
load(std_brain_file, 'V', 'Vmni');
std_area_ix = vb_reduce_cortex(std_brain_file, 1:size(V,1), num_vertex);
Vmni_std = Vmni(std_area_ix, :);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Map selected vertices onto a subject brain
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Mapped selected vertices onto a single subject brain.');
Vmni_sbj = vb_load_cortex(brain_file, 'MNI');
ix_area = vb_find_nearest_point(Vmni_sbj, Vmni_std);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Remove non cortical area index 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Remove non cortical area index from verticies.');
load(fs_info_file);
load(brain_file);

% Devide left/right
vbmeg_lh_area_ix = ix_area(ix_area<=Vinfo.NdipoleL);
vbmeg_rh_area_ix = ix_area(ix_area>Vinfo.NdipoleL)- Vinfo.NdipoleL;

% FreeSurfer vertex index of vbmeg_(lh/rh)_ix
FS_lh_area_ix    = BV_index.Left(vbmeg_lh_area_ix);
FS_rh_area_ix    = BV_index.Right(vbmeg_rh_area_ix) - Vinfo.NdipoleL0;

% Remove non-cortical index
if vb_matlab_version('>=', '8.1')
    [tmp, ix_l] = intersect(FS_lh_area_ix, FS_wm.lh_cortex_index, 'legacy');
    [tmp, ix_r] = intersect(FS_rh_area_ix, FS_wm.rh_cortex_index, 'legacy');
else
    [tmp, ix_l] = intersect(FS_lh_area_ix, FS_wm.lh_cortex_index);
    [tmp, ix_r] = intersect(FS_rh_area_ix, FS_wm.rh_cortex_index);
end
ix_l = sort(ix_l);
ix_r = sort(ix_r);

vbmeg_lh_area_ix = vbmeg_lh_area_ix(ix_l);
vbmeg_rh_area_ix = vbmeg_rh_area_ix(ix_r);
FS_lh_area_ix    = FS_lh_area_ix(ix_l);
FS_rh_area_ix    = FS_rh_area_ix(ix_r);

vbmeg_area_ix = [vbmeg_lh_area_ix' vbmeg_rh_area_ix'+Vinfo.NdipoleL]';
vb_fsave(parcel_file, 'vbmeg_area_ix');
disp(['Saved:' parcel_file]);
