function dmri_vbmeg_vertex_select(brain_file, fs_info_file, num_vertex, parcel_file)
% Choose verticies of VBMEG brain model by specifing num_vertex.
%
% [Usage]
%    dmri_vbmeg_vertex_select(brain_file, fs_info_file, num_vertex, parcel_file);
%
% [Input]
%      brain_file : VBMEG BRAIN-MAT file(.brain.mat)
%    fs_info_file : Freesurfer information file.
%      num_vertex : The number of verticies.
%     parcel_file : Parcel file.
%
% [Output]
%    none
%
% [Example]
%    
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin ~= 4
    error('Please check input arguments');
end
if exist(brain_file, 'file') ~= 2
    error('brain_file not found.');
end
if exist(fs_info_file, 'file') ~= 2
    error('fs_info_file not found.');
end
if num_vertex <= 0
    error('Invalid num_vertex was specified.');
end
if ~ischar(parcel_file)
    error('parcel_file is not a filename.');
end

%
% --- Main Procedure
%
load(brain_file, 'brain_parm', 'Vinfo');
if ~isfield(Vinfo, 'cortexL')
    warning('Specified BRAIN-MAT(.brain.mat) is created by VBMEG 1.0. The number of parcels will be different between subjects.');
    dmri_vbmeg_vertex_select_old(brain_file, fs_info_file, num_vertex, parcel_file);
    return;
end

% Reference standard brain
[std_parm, std_brain_dir] = vb_set_icbm152(brain_parm.Nvertex);
std_brain_file = fullfile(std_brain_dir, std_parm.brain_file);
if exist(std_brain_file, 'file') ~= 2
    % error case
    fprintf('To make parcellation information about this brain model, \n');
    fprintf('it needs to create a custom resolution standard brain model, please type as below.\n');
    fprintf('==========\n');
    fprintf('[parm, proj_root] = vb_set_icbm152(%d);\n', brain_parm.Nvertex);
    fprintf('vb_job_brain_std(proj_root, parm);\n');
    fprintf('==========\n');
    fprintf('Then re-run the operation.\n');
    return;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Select vertices on standard brain
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp(['Select ' num2str(num_vertex) ' vertices on standard brain.']);

% Select specified number of verticies on the standard brain file.
s = load(std_brain_file, 'Vmni', 'Vinfo');
s_cortex_ix = [s.Vinfo.cortexL; s.Vinfo.cortexR];
reduce_ratio = num_vertex/length(s_cortex_ix);
vbmeg_area_ix = vb_reduce_cortex(std_brain_file, s_cortex_ix, reduce_ratio);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Check and save if selected verticies are subcortex.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
p = struct;
[tmp1, tmp2, p.Vinfo] = vb_load_cortex_info(brain_file, 'subj');
p_cortex_ix = [p.Vinfo.cortexL; p.Vinfo.cortexR];
sub_cortex_ix = setdiff(vbmeg_area_ix, p_cortex_ix);

vb_fsave(parcel_file, 'vbmeg_area_ix', 'sub_cortex_ix');
disp(['Saved:' parcel_file]);
