function dmri_fiber_track_prob_specific_pair(...
                              parcel_dir, parcel_from_list, parcel_to_list, ...
                              mif_file, mask_file, ...
                              fa_gz_file, fs_brain_gz_file, ...
                              trans_info_dir, dmri_file, ...
                              output_dir, process_host, Nworker)
% Stream lines tracking and save connection.
% (parcelN.nii.gz --> parcel_exN.nii.gz)
%
% [Usage]
%     dmri_fiber_track_specific_pair(...
%                             parcel_dir, parcel_from_list, parcel_to_list, ...
%                             mif_file, mask_file, ...
%                             fa_gz_file, trans_info_dir, ...
%                             output_dir, [,process_host][,Nworker]);
%
% [Input]
%           parcel_dir : parcel files in the directory are used.
%                        parcel_dir/parcelN.nii.gz
%     parcel_from_list : calculate path from parcel_from_list(n) to parcel_to_list(n).
%       parcel_to_list : calculate path from parcel_from_list(n) to parcel_to_list(n).
%             mif_file : Fiber orientation density function file(.mif)
%            mask_file : a mask region of interest.
%           fa_gz_file : FA nifti-gz file(.nii.gz) Fractional Anisotropy image.
%     fs_brain_gz_file : Freesurfer nifti-gz file(.nii.gz) Cortex image.
%       trans_info_dir : transform information directory.
%            dmri_file : difusion mri file(.dmri.mat)
%           output_dir : The output directory for fiber tracking result.
%         process_host : [Optional] process hosts name(default: {'localhost'})
%                         stream lines tracking will be executed on specified hosts.
%                        caution: It requires SSH login to the hosts without password.
%                         e.g. {'cbi-node01g', 'cbi-node02g', 'cbi-node03g'};
%              Nworker : [Optional] The number of matlab process which is used for 
%                                   tck file processing. (default:8)
%
% [Output]
%    none
%
% [Output files]
%    output_dir/parcelA-B.tck : fiber tracking result from parcel A to parcel B.
%                               A : parcel_from_list(n)
%                               B : parcel_to_list(n)
%
%
%    output_dir/Vmni_connect.mat : Vmni_connection file. 
%                                  This is used for visualization brain activity.
%                                  http://www.cns.atr.jp/cbi/?p=596
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin < 10
    error('Please check input arguments.');
end
if exist(parcel_dir, 'dir') ~= 7
    error('Specified parcel directory not found.');
end
if exist(mif_file, 'file') ~= 2
    error('Specified mif_file not found.');
end
if exist(mask_file, 'file') ~= 2
    error('Specified mask_file not found.');
end
if exist(output_dir, 'dir') ~= 7
    mkdir(output_dir);
end
if length(parcel_from_list) ~= length(parcel_to_list)
    error('The length of parcel_from_list and parcel_to_list must be same.');
end
if exist(fa_gz_file, 'file') ~= 2
    error('Specified fa_gz_file not found.');
end
if exist(fs_brain_gz_file, 'file') ~= 2
    error('Specified fs_brain_gz_file not found.');
end
if exist(trans_info_dir, 'dir') ~= 7
    error('Specified trans_info_dir not found.');
end
if exist(dmri_file, 'file') ~= 2
    error('Specified dmri_file not found.');
end
if ~exist('process_host', 'var') || isempty(process_host)
    process_host = {'localhost'};
end
if ischar(process_host)
    process_host = {process_host};
end
if ~exist('Nworker', 'var')
    Nworker = 8;
end
load(dmri_file, 'brain_file', 'vbmeg_area_ix');
if exist(brain_file, 'file') ~= 2
    error('Cannot load brain_file : %s', brain_file);
end


dmri_setenv;
check_cmd = which('is_cpu_host_available.sh');
disp('Connecting hosts without password ...');
connection = true;
for k=1:length(process_host)
    if strcmpi(process_host{k}, 'localhost')
        [a, real_host] = system('hostname');
        real_host = real_host(1:end-1); % remove last \n
        process_host{k} = real_host;
    end
    fprintf('%s ...', process_host{k});
    ret_msg = '';
    [a, ret_msg] = system(['timeout 5 ' check_cmd ' ', process_host{k}]);
    
    is_ok = true;
    if a == 0 || a == 1
        cmd_track = ['timeout 5 ssh ' process_host{k} ' "', 'bash -c ''source ' getenv('BASH_PROF') ';streamtrack --help''"'];
        [ret_st, ret_msg] = system(cmd_track);
        if ret_st ~= 0
            is_ok = false;
        end
    else
        is_ok = false;
    end
    if is_ok
        disp('OK.');
    else
        connection = false;
        disp(sprintf('Failed.\n   %s', ret_msg));
    end
end
if connection == false
    error('Please check the setting of ssh connection.');
end

%
% --- Main Procedure
%
disp('Now preparing for execution...');

s = dbstack;
if length(s) > 1 && isequal(s(2).name, mfilename)
else
    fprintf('Start Fiber tracking to create Vmni_connect.mat file.\n');
end

% get absolute output directory path.
pushd = pwd;
cd(output_dir);
output_dir = pwd;
cd(pushd);

% Working directory
work_dir = tempname(output_dir);
mkdir(work_dir);
process_fiber   = cell(0);
kill_remote_process = true;

% Define terminate function
function terminate_fcn

    if kill_remote_process
        % Kill fiber tracking process on specified hosts
        for k=1:length(process_fiber)
            process_fiber{k}.destroy();
        end
    end
    % Kill the workers to process fiber tracking result files.
    dmri_fiber_tck_file_worker_stop(work_dir);
end

fprintf('working directory : %s\n', work_dir);
fprintf('(caution: Do not press the ctrl+c when copying the path string.)\n');

% copy all parcel files into work_dir
parcel_list = unique([parcel_from_list; parcel_to_list]);
for k=1:length(parcel_list)
    copyfile(fullfile(parcel_dir, filesep, ['parcel', num2str(parcel_list(k)), '.nii.gz']), work_dir);
end
copyfile(mask_file, work_dir);

% Unzip mask_file
[p, f, e] = fileparts(mask_file);
mask_nii_gz_file = fullfile(work_dir, [f, e]);
mask_nii_file    = strrep(mask_nii_gz_file, '.nii.gz', '.nii');
cmd = ['gunzip ' mask_nii_gz_file];
system(cmd);

% Parallel processing
Nhost = length(process_host);

Npair = length(parcel_from_list);

script_files    = cell(0, 1);
fids            = cell(0, 1);
Npair_per_host = ceil(Npair / Nhost);

finished_list_dir = fullfile(work_dir, filesep, 'finished');
mkdir(finished_list_dir);
timeout_sh = which('automatic_shutoff_short.sh');


% Create scripts as many as the number of host.
for k=1:Npair
    if mod(k, Npair_per_host) == 1 || Npair == 1 || Npair_per_host == 1
        Nth_script_file = length(script_files)+1;
        script_file     = fullfile(work_dir, filesep, [process_host{Nth_script_file}, '.sh']);

        % create template script
        isok = dmri_script_file_create(script_file);
        if isok == false, error('Failed to create script.'); end

        % open append mode
        fid = fopen(script_file, 'a');
        if fid == -1
            error('File open failed. processing terminated.');
        end

        fids{Nth_script_file}         = fid;
        script_files{Nth_script_file} = script_file;
    end


    % Unzip .nii.gz
    from = parcel_from_list(k);
    to   = parcel_to_list(k);
    parcel_a_to_b_str = [num2str(from) '-' num2str(to)];
    parcel_n_gz_file1 = fullfile(work_dir, filesep, ['parcel', num2str(from), '.nii.gz']);
    parcel_n_gz_file2 = fullfile(work_dir, filesep, ['parcel', num2str(to),   '.nii.gz']);


    fprintf(fid, 'gunzip %s\n', parcel_n_gz_file1);
    fprintf(fid, 'gunzip %s\n', parcel_n_gz_file2);

    % Fiber tracking
    tck_file = fullfile(work_dir, filesep, ...
                        ['parcel', parcel_a_to_b_str, '.tck']);
    parcel_n_nii_file1    = strrep(parcel_n_gz_file1,    '.nii.gz', '.nii');
    parcel_n_nii_file2    = strrep(parcel_n_gz_file2,    '.nii.gz', '.nii');
    
    fprintf(fid, ['echo "Fiber tracking - parcel', ...
                  parcel_a_to_b_str, '"\n']);
    fprintf(fid, '%s\n', [timeout_sh, ' ', ...
                        'streamtrack SD_PROB', ...
                        ' '          ,mif_file, ...
                        ' -seed '    ,parcel_n_nii_file1, ...
                        ' -include ' ,parcel_n_nii_file2, ...
                        ' -mask '    ,mask_nii_file, ...
                        ' '          ,tck_file, ...
                        ' -maxnum 50000000', ...
                        ' -num 10', ...
                        ' -stop' , ...
                        ' -unidirectional', ...
                        ' -length 300', ...
                        ' -debug']);
    fprintf(fid, '%s\n', ['if [ ! -e ' tck_file, ' ]; then']); 
    fprintf(fid, '%s\n', ['    echo -n > ' tck_file]);
    fprintf(fid, '%s\n', ['fi']);
    fprintf(fid, '%s\n', ['mkdir ', fullfile(finished_list_dir, filesep, ['parcel' parcel_a_to_b_str])]);
    fprintf(fid, '%s\n', ['if [ ! -d ' work_dir ' ]; then']);
    fprintf(fid, 'exit $?;\nfi\n');
end


% close script files
for k=1:length(fids), fclose(fids{k}); end

%
% --- Execution
%
cpu_job_throw = which('cpu_job_throw.sh');
runtime = java.lang.Runtime.getRuntime();

for k=1:length(script_files)
    % Create JOB FILE(one host processes only one job).
    JOBS_FILE  = fullfile(work_dir, filesep, [process_host{k}, '_job.txt']);
    fid = fopen(JOBS_FILE, 'wt');
    if fid == -1, error('Job file write error.'); end
    fprintf(fid, '%s\n', script_files{k});
    fclose(fid);

    % Create HOSTS FILE
    HOSTS_FILE = fullfile(work_dir, filesep, [process_host{k}, '.txt']);
    fid = fopen(HOSTS_FILE, 'wt');
    if fid == -1, error('hosts file write error.'); end
    fprintf(fid, '%s\n', process_host{k});
    fclose(fid);

    % Execute
    cmd = [cpu_job_throw, ' ', JOBS_FILE, ' ', HOSTS_FILE];
    process_fiber{length(process_fiber)+1} = runtime.exec(cmd);
    pause(5);
    % Wait for creating log directory.
    while(1)
        Nlog_dir = length(dir(fullfile(work_dir, 'log*')));
        if Nlog_dir == k
            break;
        else
            pause(1);
        end
    end
end
fprintf('Running %d process.\n', Nlog_dir);

%
% --- Convert TCK files into MAT files.
%
parm_mat_file = fullfile(work_dir, 'parm.mat');
matlab_exe = fullfile(matlabroot, 'bin', 'matlab');
vb_save(parm_mat_file, 'work_dir', 'fa_gz_file', ...
                       'fs_brain_gz_file', 'trans_info_dir');

prog_dir   = fileparts(which('vbmeg.m'));
for n=1:Nworker
    worker_script   = fullfile(work_dir, filesep, ['worker', num2str(n), '.m']);
    worker_err_file = fullfile(work_dir, filesep, ['worker', num2str(n), '_error.txt']);
    fid = fopen(worker_script, 'wt');
    if fid == -1, error('Worker file create error.'); end
    fprintf(fid, '%s\n', 'try');
    fprintf(fid, '%s\n', ['cd(''', prog_dir, ''');']);
    fprintf(fid, '%s\n', 'vbmeg;');
    fprintf(fid, '%s\n', ['dmri_fiber_tck_file_for_display_worker_start(''', ...
                           parm_mat_file, ''', ', num2str(n), ');']);
    fprintf(fid, '%s\n', 'catch');
    fprintf(fid, '%s\n', ['fid = fopen(''' worker_err_file, ''', ''wt'');']);
    fprintf(fid, '%s\n', 'err = lasterror;');
    fprintf(fid, '%s\n', 'fprintf(fid, err.message)');
    fprintf(fid, '%s\n', 'fclose(fid);');
    fprintf(fid, '%s\n', 'end');
    fprintf(fid, '%s\n', 'exit;');
    
    fclose(fid);
  
    cmd = [matlab_exe, ...
           ' -nodisplay -nojvm -nosplash ', ...
           ' -singleCompThread ', ...
           '-r "cd ', work_dir, '; ' 'worker', num2str(n) ';"'];
    [status, result] = system([cmd, '&']);
end

%
% --- Check the number of created files
%

% Register clean up function(onCleanup is MATLAB builtin function)
onCleanup(@terminate_fcn);

mat_files_pre = cell(0);
tic;
err = false;
while(1)
    log_info_list = dmri_fiber_log_info_get(work_dir);
    
    % Check result file and Exit code
    for l=1:length(log_info_list)
        result_file = log_info_list(l).result_file;
        err_file    = log_info_list(l).err_file;
        if ~isempty(result_file)
            [tmp, msg] = system(['cat ' result_file, '| grep ExitCode=']);
            if isempty(msg), continue; end
            [parse] = textscan(msg, '%s', 'delimiter', '=,');
            host      = parse{1}{4};
            exit_code = str2double(parse{1}{6});
            if exit_code ~= 0
                fprintf('errorlog(host:%s):\n%s\n', host, err_file);
                err = true;
            end
        end
        if l == length(log_info_list) && err
            return;
        end
    end
    d = dir(fullfile(work_dir, filesep, 'fs_parcel*.mat'));
    mat_files_now = {d.name}';

    % Display progress
    diff = setdiff(mat_files_now, mat_files_pre);
    if ~isempty(diff)
        for j=1:length(diff)
            fprintf('Created (%d/%d): %s\n', length(mat_files_pre)+j, Npair, diff{j});
        end
        mat_files_pre = mat_files_now;
    end

    if length(mat_files_now) == Npair && length(dir(finished_list_dir)) == 2 % 2:'.' and '..'
        fprintf('Processing (%d/%d)\nDone.\n', Npair, Npair)
        break;
    end
    pause(5);
end

% Wait for delete of MATLAB workers.
kill_remote_process = false;
pause(2);
dmri_fiber_tck_file_worker_stop(work_dir);
pause(2);
rmdir(finished_list_dir);

Vtracks = cell(Npair, 1);
retry_parcel_from_list = [];
retry_parcel_to_list   = [];
empty_path_ix          = [];
for k=1:Npair
    from = parcel_from_list(k);
    to   = parcel_to_list(k);
    path_file = fullfile(work_dir, ['fs_parcel' num2str(from) '-', num2str(to), '.mat']);
    load(path_file, 'vtracks');
    Vtracks{k} = vtracks;
    if isempty(vtracks)
        % reverse tracking
        retry_parcel_from_list = [retry_parcel_from_list; to];
        retry_parcel_to_list   = [retry_parcel_to_list; from];
        empty_path_ix = [empty_path_ix; k];
    end
end

if ~isempty(empty_path_ix)
    % remove empty path
    parcel_from_list(empty_path_ix) = [];
    parcel_to_list(empty_path_ix)   = [];
    Vtracks(empty_path_ix)          = [];

%     % retry B-A fiber tracking
%     disp('reverse fiber tracking...');
%     for k=1:length(retry_parcel_from_list)
%         disp(['parcel' num2str(retry_parcel_from_list(k)), '-' ...
%               num2str(retry_parcel_to_list(k))]);
%     end
%     output_dir2 = tempname(output_dir);
%     dmri_fiber_track_prob_specific_pair(...
%                               parcel_dir, retry_parcel_from_list, retry_parcel_to_list, ...
%                               mif_file, mask_file, ...
%                               fa_gz_file, fs_brain_gz_file, ...
%                               trans_info_dir, dmri_file, ...
%                               output_dir2, process_host, Nworker);
    retry_parcel_from_list = [];
    retry_parcel_to_list   = [];
    empty_path_ix = [];
end


%
% --- Save Vmni_connect.mat
%
V    = vb_load_cortex(brain_file, 'subj');
Vmni = vb_load_cortex(brain_file, 'MNI');

Index_from = vbmeg_area_ix(parcel_from_list);
Index_to   = vbmeg_area_ix(parcel_to_list);
V_from    = V(Index_from, :);
V_to      = V(Index_to, :);
Vmni_from = Vmni(Index_from, :);
Vmni_to   = Vmni(Index_to, :);

if ~isempty(retry_parcel_from_list)
  rVmni_connect_file = fullfile(output_dir2, 'Vmni_connect.mat');
  r = load(rVmni_connect_file);

  for k=1:length(r.Vtracks)
      if ~isempty(r.Vtracks{k})
          % save B-A result as A-B
          Index_from = [Index_from; r.Index_to(k)];
          Index_to   = [Index_to; r.Index_from(k)];
          V_from     = [V_from; r.V_to(k)];
          V_to       = [V_to;   r.V_from(k)];
          Vmni_from  = [Vmni_from; r.Vmni_to(k)];
          Vmni_to    = [Vmni_to;   r.Vmni_from(k)];
          Vtracks    = [Vtracks, r.Vtracks(k)];
      else
          disp(['No path : parcel', num2str(r.Index_to(k)) '-' num2str(r.Index_from(k)) '.']);
      end
  end
  rmdir(output_dir2, 's');
end
  

% convert Vtracks to Vtracks_mni
parm.brainfile  = brain_file;

[Vc, Nc]     = cell_to_mat(Vtracks);
Vtracks_mni  = trans_subj_mni_coord(Vc,parm);
Vtracks_mni  = mat_to_cell(Vtracks_mni, Nc);

% Save Vmni_connect.mat
Vmni_connect_file = fullfile(output_dir, 'Vmni_connect.mat');
vb_save(Vmni_connect_file, 'Index_from', 'Index_to', ...
                           'V_from', 'V_to', ...
                           'Vmni_from', 'Vmni_to', ...
                           'Vtracks', 'Vtracks_mni');
s = dbstack;
if length(s) > 1 && isequal(s(2).name, mfilename)
else
    fprintf('Created Vmni_connect file : %s\n', Vmni_connect_file);
end

% clean up working directory.
rmdir(work_dir, 's');

toc;
end
