function dmri_fiber_track_prob(mif_file, parcel_dir, mask_file, anatmat_file, process_host, Nworker)
% Stream lines tracking and save connection.
% (parcelN.nii.gz --> parcel_exN.nii.gz)
%
% [Usage]
%    dmri_fiber_track_prob(mif_file, parcel_dir, mask_file, anatmat_file [,process_host][,Nworker]);
%
% [Input]
%       mif_file : Fiber orientation density function file(.mif)
%     parcel_dir : parcel files in the directory are used.
%                   parcel_dir/parcelN.nii.gz      : The seed region of interest.
%                             /parcel_exN.nii.gz   : an inclusion region of interest.
%                             /parcel_mm_coord.mat : Parcels vertices in mm coordinate.
%                        
%      mask_file : a mask region of interest.
%   anatmat_file : anatomical matrix file.
%   process_host : [Optional] process hosts name(default: {'localhost'})
%                    stream lines tracking will be executed on specified hosts.
%                    caution: It requires SSH login to the hosts without password.
%                  e.g. {'cbi-node01g', 'cbi-node02g', 'cbi-node03g'};
%        Nworker : [Optional] The number of matlab process which is used for 
%                             tck file processing. (default:8)
%
% [Output]
%    none
%
% [Output file]
%    anatmat_file : This file contain these variables.
%                     Ntrack  <Nparcel x 1 double>
%                     c_count <Nparcel x Nparcel double>
%                     f_leng  <Nparcel x Nparcel double>
%                     t_count <Nparcel x 1 double>
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin < 4
    error('Please check input arguments.');
end
if exist(mif_file, 'file') ~= 2
    error('Specified mif_file not found.');
end
if exist(parcel_dir, 'dir') ~= 7
    error('Specified parcel directory not found.');
end
if exist(mask_file, 'file') ~= 2
    error('Specified mask_file not found.');
end

output_dir = fileparts(anatmat_file);
if exist(output_dir, 'dir') ~= 7
    mkdir(output_dir);
end
if ~exist('process_host', 'var') || isempty(process_host)
    process_host = {'localhost'};
end
if ischar(process_host)
    process_host = {process_host};
end
if ~exist('Nworker', 'var')
    Nworker = 8;
end

dmri_setenv;
check_cmd = which('is_cpu_host_available.sh');
disp('Connecting hosts without password ...');
connection = true;
for k=1:length(process_host)
    if strcmpi(process_host{k}, 'localhost')
        [a, real_host] = system('hostname');
        real_host = real_host(1:end-1); % remove last \n
        process_host{k} = real_host;
    end
    fprintf('%s ...', process_host{k});
    ret_msg = '';
    [a, ret_msg] = system(['timeout 5 ' check_cmd ' ', process_host{k}]);
    
    is_ok = true;
    if a == 0 || a == 1
        cmd_track = ['timeout 5 ssh ' process_host{k} ' "', 'bash -c ''source ' getenv('BASH_PROF') ';streamtrack --help''"'];
        [ret_st, ret_msg] = system(cmd_track);
        if ret_st ~= 0
            is_ok = false;
        end
    else
        is_ok = false;
    end
    if is_ok
        disp('OK.');
    else
        connection = false;
        disp(sprintf('Failed.\n   %s', ret_msg));
    end
end
if connection == false
    error('Please check the setting of ssh connection.');
end

%
% --- Main Procedure
%


% get absolute output directory path.
pushd = pwd;
cd(output_dir);
output_dir = pwd;
cd(pushd);

% Working directory
work_dir = tempname(output_dir);
mkdir(work_dir);
process_fiber   = cell(0);
kill_remote_process = true;

% Define terminate function
function terminate_fcn
    if kill_remote_process
        % Kill fiber tracking process on specified hosts
        for k=1:length(process_fiber)
            process_fiber{k}.destroy();
        end
    end
    % Kill the workers to process fiber tracking result files.
    dmri_fiber_tck_file_worker_stop(work_dir);
end

fprintf('working directory : %s\n', work_dir);
disp('Now preparing for execution...');


% copy all parcel files into work_dir
copyfile(fullfile(parcel_dir, filesep, 'parcel*.nii.gz'), work_dir);
copyfile(mask_file, work_dir);

% Unzip mask_file
[p, f, e] = fileparts(mask_file);
mask_nii_gz_file = fullfile(work_dir, [f, e]);
mask_nii_file    = strrep(mask_nii_gz_file, '.nii.gz', '.nii');
cmd = ['gunzip ' mask_nii_gz_file];
system(cmd);

% Parallel processing
Nhost = length(process_host);

parcel_mm_file = fullfile(parcel_dir, filesep, 'parcel_mm_coord.mat');
copyfile(parcel_mm_file, work_dir);
load(parcel_mm_file, 'Nlabel');

script_files    = cell(0, 1);
fids            = cell(0, 1);
Nlabel_per_host = ceil(Nlabel / Nhost);

finished_list_dir = fullfile(work_dir, filesep, 'finished');
mkdir(finished_list_dir);
timeout_sh = which('automatic_shutoff.sh');


% Create scripts as many as the number of host.
FT_JOB_FILE     = fullfile(work_dir, filesep, ['jobs.txt']);

% make main script.
fid = fopen(FT_JOB_FILE, 'wt');
if fid == -1, error('Failed to create script.'); end

fiber_tracking_sh_files = cell(0);
for k=1:Nlabel
    fiber_tracking_sh_files{k} = fullfile(work_dir, ['fiber_track' num2str(k), '.sh']);
    fprintf(fid, '%s\n', fiber_tracking_sh_files{k});
end
fclose(fid);

ix = randperm(Nlabel); % randomize fiber-tracking processing order not to 
                       % stop calculation because there is an possitibility 
                       % that parcel which causes local-minumum calculation
                       % exists on specific parcel.

for k=1:Nlabel
    ft_file = fiber_tracking_sh_files{k};
    % make script for fiber tracking
    isok = dmri_script_file_create(ft_file);
    if isok == false, error('Failed to create script.'); end

    % open append mode
    ft_fid = fopen(ft_file, 'a');
    if ft_fid == -1
        error('File open failed. processing terminated.');
    end
    
    parcel_n = ix(k);
    parcel_n_str = num2str(parcel_n);

    % Unzip .nii.gz
    parcel_n_gz_file    = fullfile(work_dir, filesep, ['parcel',    parcel_n_str, '.nii.gz']);
    parcel_n_ex_gz_file = fullfile(work_dir, filesep, ['parcel_ex', parcel_n_str, '.nii.gz']);
    fprintf(ft_fid, 'gunzip %s\n', parcel_n_gz_file);
    fprintf(ft_fid, 'gunzip %s\n', parcel_n_ex_gz_file);

    % Fiber tracking
    tck_file = fullfile(work_dir, filesep, ['parcel', parcel_n_str, '.tck']);
    parcel_n_nii_file    = strrep(parcel_n_gz_file,    '.nii.gz', '.nii');
    parcel_n_ex_nii_file = strrep(parcel_n_ex_gz_file, '.nii.gz', '.nii');
    
    fprintf(ft_fid, ['echo "Fiber tracking - parcel', parcel_n_str, '"\n']);
    fprintf(ft_fid, '%s\n', [timeout_sh, ' ', ...
                        'streamtrack SD_PROB', ...
                        ' '          ,mif_file, ...
                        ' -seed '    ,parcel_n_nii_file, ...
                        ' -include ' ,parcel_n_ex_nii_file, ...
                        ' -mask '    ,mask_nii_file, ...
                        ' '          ,tck_file, ...
                        ' -num 100000 -stop -unidirectional -maxnum 100000 -length 300 -debug']);
    fprintf(ft_fid, '%s\n', ['if [ ! -e ' tck_file, ' ]; then']); 
    fprintf(ft_fid, '%s\n', ['    echo -n > ' tck_file ]);
    fprintf(ft_fid, '%s\n', ['fi']);
    fprintf(ft_fid, '%s\n', ['mkdir ', fullfile(finished_list_dir, filesep, parcel_n_str)]);
    fprintf(ft_fid, '%s\n', ['if [ ! -d ' work_dir ' ]; then']);
    fprintf(ft_fid, 'exit $?;\nfi\n');
    
    fclose(ft_fid);
end


% Create HOSTS FILE
HOSTS_FILE = fullfile(work_dir, filesep, 'hosts.txt');
fid = fopen(HOSTS_FILE, 'wt');
if fid == -1, error('hosts file write error.'); end
for k=1:length(process_host)
    fprintf(fid, '%s\n', process_host{k});
end
fclose(fid);

%
% --- Execution
%
cpu_job_throw = which('cpu_job_throw.sh');
%runtime = java.lang.Runtime.getRuntime();

% Execute
% cmd = ['gnome-terminal -e "' cpu_job_throw, ' ', FT_JOB_FILE, ' ', HOSTS_FILE '" &'];
% system(cmd); % debug
cmd     = [cpu_job_throw, ' ', FT_JOB_FILE, ' ', HOSTS_FILE];
cmd_exe = [cmd ' > ' fullfile(work_dir, 'main_log.txt &')];
system(cmd_exe);

%process_fiber{1} = runtime.exec(cmd);

% Wait for creating log directory.
while(1)
    log_dir = length(dir(fullfile(work_dir, 'log*')));
    if log_dir
        break;
    else
        pause(1);
    end
end
fprintf('Started processing.\n');

%
% --- Convert TCK files into MAT files.
%
matlab_exe = fullfile(matlabroot, 'bin', 'matlab');
prog_dir   = fileparts(which('vbmeg.m'));
for n=1:Nworker
    worker_script   = fullfile(work_dir, filesep, ['worker', num2str(n), '.m']);
    worker_err_file = fullfile(work_dir, filesep, ['worker', num2str(n), '_error.txt']);
    fid = fopen(worker_script, 'wt');
    if fid == -1, error('Worker file create error.'); end
    fprintf(fid, '%s\n', 'try');
    fprintf(fid, '%s\n', ['cd(''', prog_dir, ''');']);
    fprintf(fid, '%s\n', 'vbmeg;');
    fprintf(fid, '%s\n', ['dmri_fiber_tck_file_worker_start(''', ...
                           work_dir, ''', ' num2str(n), ');']);
    fprintf(fid, '%s\n', 'catch');
    fprintf(fid, '%s\n', ['fid = fopen(''' worker_err_file, ''', ''wt'');']);
    fprintf(fid, '%s\n', 'err = lasterror;');
    fprintf(fid, '%s\n', 'fprintf(fid, err.message)');
    fprintf(fid, '%s\n', 'fclose(fid);');
    fprintf(fid, '%s\n', 'end');
    fprintf(fid, '%s\n', 'exit;');
    
    fclose(fid);
  
    cmd = [matlab_exe, ...
           ' -nodisplay -nojvm -nosplash ', ...
           ' -singleCompThread ', ...
           '-r "cd ', work_dir, '; ' 'worker', num2str(n) ';"'];
    [status, result] = system([cmd, '&']);
end

%
% --- Check the number of created files
%

% Register clean up function(onCleanup is MATLAB builtin function)
onCleanup(@terminate_fcn);

mat_files_pre = cell(0);
tic;
err = false;
while(1)
    log_info_list = dmri_fiber_log_info_get(work_dir);
    
    % Check result file and Exit code
    for l=1:length(log_info_list)
        result_file = log_info_list(l).result_file;
        err_file    = log_info_list(l).err_file;
        if ~isempty(result_file)
            [tmp, msg] = system(['cat ' result_file, '| grep ExitCode=']);
            if isempty(msg), continue; end
            [parse] = textscan(msg, '%s', 'delimiter', '=,');
            host      = parse{1}{4};
            exit_code = str2double(parse{1}{6});
            if exit_code ~= 0
                fprintf('errorlog(host:%s):\n%s\n', host, err_file);
                err = true;
            end
        end
        if l == length(log_info_list) && err
            return;
        end
    end
    d = dir(fullfile(work_dir, filesep, 'parcel_fiber*.mat'));
    mat_files_now = {d.name}';

    % Display progress
    diff = setdiff(mat_files_now, mat_files_pre);
    if ~isempty(diff)
        for j=1:length(diff)
            fprintf('Created (%d/%d): %s\n', length(mat_files_pre)+j, Nlabel, diff{j});
        end
        mat_files_pre = mat_files_now;
    end

    if length(mat_files_now) == Nlabel && length(dir(finished_list_dir)) == 2 % 2: '.' and '..'
        fprintf('Processing (%d/%d)\nDone.\n', Nlabel, Nlabel)
        break;
    end
    pause(5);
end

% Wait for delete of MATLAB workers.
kill_remote_process = false;
pause(2);
dmri_fiber_tck_file_worker_stop(work_dir);
pause(2);
rmdir(finished_list_dir);

%
% --- Create Anatomical matrix file
%
save_anat_matrix(work_dir, Nlabel, 'parcel_fiber', anatmat_file);
fprintf('Created Anatomical matrix file : %s\n', anatmat_file);

% filber tracking mat files zipped and put into the same directory.
zip(fullfile(output_dir, filesep, 'parcel_fiber.zip'), fullfile(work_dir, 'parcel_fiber*.mat'));

% clean up working directory.
rmdir(work_dir, 's');

toc;

end
