function h = vb_plot_cortex(plot_parm,V,F,inf_C,J,Jmin,Jmax,ix_area)
% Plot cortical surface model. 
% (VBMEG public function)
%
% [syntax]
% vb_plot_cortex(plot_parm,V,F)
% vb_plot_cortex(plot_parm,V,F,inf_C)
% vb_plot_cortex(plot_parm,V,F,inf_C,J)
% vb_plot_cortex(plot_parm,V,F,inf_C,J,Jmin)
% vb_plot_cortex(plot_parm,V,F,inf_C,J,Jmin,Jmax)
% vb_plot_cortex(plot_parm,V,F,inf_C,J,Jmin,Jmax,ix_area)
%
% [input]
% plot_parm: <<struct>> Plotting parameters.
% --- fields of plot_parm
%  LRflag: <<string>> Specifying hemisphere for plotting. 
%   'LR' for both hemishepre
%   'L' for left hemisphere
%   'R' for both hemisphere
%  wmode : <<bool>> If true, absolute value of 'J' is plotted. Otherwise,
%          raw value of 'J' is plotted. 
%  paint : <<bool>> If true, 
%  cmap  : <<int>> Colormap (1: hot, otherwise: jet). 
%  isact : <optional> <<bool>> If true, activity map is superimposed on
%         cortical surface.
%  color : <optional> <<vector>> 
%  index : <optional> <<vector>>
%  area  : <optional> <<struct>>
%  edgecolor : <optional> <<vector>> Color of edges (default: 'none')
%  --- fields of plot_parm.area
%   area_file          : <<string>> cortical area file(.area.mat)
%   key                : <<cell string>> area key name list.
%   transparent        : = true  : area tranceparent plot ON
%                      : = false : area tranceparent plot OFF <<default>>
%   tranceparent_value : from 0 to 1.0 (0.5 is default).
%                        If you set 0, area will disappear.
%   clr                : <<cell string/3d vector>>
%   str                : <optional> <<cell string>>
%   Rlabel             : <optional> <<double>> Distance between string and area.
%  ---
% ---
% V        : <<matrix>> Vertices of cortical surface model.
% F        : <<matrix>> Faces of cortical surface model.
% inf_C    : <<vector>> Color of sulcus/gyrus. It should be empty when
%                       plotting folded surface model. 
%
% [output]
% h: Handle of light.
%
% [example]
% % Load inflated standard brain model
% [V,F,xx,inf_C] = vb_load_cortex(brainfile,'Inflate');
% 
% % Load current and current information
% [Jinfo, J] = vb_load_current(currentfile);
% 
% % get loaded data index between 80[msec] to 120[msec]
% ix = intersect(find(Jinfo.Tmsec>80), find(Jinfo.Tmsec<120));
% 
% % Average current between 80msec to 120msec
% Jplot = mean(J(:, ix), 2);
% 
% % set viewpoint
% view([10, -2]);
% 
% % plot current on cortical inflate model
% plot_parm = vb_set_plot_parm;
% vb_plot_cortex(plot_parm,V,F,inf_C,Jplot);
% 
% axis equal;
% axis off;
% 
% [history]
% 2009-01-26 taku-y
% 2010-12-27 taku-y
%  [minor] Not superimpose activity (not invoke patch fnc) if all values
%  do not exceed threshold. 
% 2012-02-09 taku-y
%  [enhancement] If isfield(area,'clr'), plor_parm.area{i}.clr is
%  replaced with a.clr, where a=vb_get_area(area_file,area_key{i}). 
% 2012-03-14 taku-y
%  [minor] Check color value in a.clr, where
%  a=vb_get_area(area_file,area_key). 
% 2022-03-14 rhayashi
%  [enhancement] area transparent valueis  added.
%  - plot_parm.area.transparent_value
% 2022-04-26 Y. Takeda
%  [minor] Added option to set edge color
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- Check input arguments
if nargin<8, ix_area = []; end
if nargin<5, 
  J = [];
  Jmin = [];
  Jmax = [];
else
  if nargin<7, Jmax = max(abs(J)); end
  if nargin<6, Jmin = max(abs(J))*0.1; end
end
if nargin<4, inf_C = []; end; 

% color of cortical surface
if isfield(plot_parm,'color') & ~isempty(plot_parm.color)
  fclr = plot_parm.color;
else
  fclr = [0.8 0.8 0.8];
end

if ~isfield(plot_parm,'isact'), 
  plot_parm.isact = true;
end

% Color of edges
if isfield(plot_parm, 'edgecolor') & ~isempty(plot_parm.edgecolor)
    edgecolor = plot_parm.edgecolor;
else
    edgecolor = 'none';
end

% Left/Right hemisphere
switch plot_parm.LRflag
 case 'L'
  F0 = F.F3L;
 case 'R'
  F0 = F.F3R;
 case 'LR'
  F0 = F.F3;
end

if isempty(inf_C)
    inflate = false;
else
    inflate = true;
end
Vshrink = vb_shrink_cortex_for_plot(V, F0, inflate);

% --- Select visible patches
if ~isfield(plot_parm,'index'), index = [];
else index = plot_parm.index; end
if length(index) == 0, FF = F0;
else FF = vb_patch_select2(index,F0,size(V,1)); end

% --- Plot cortical surface
h = gcf; 
set(h, 'renderer', 'zbuffer');
if isempty(inf_C)
  % Display patches
  patch('Faces',FF,'Vertices',Vshrink,'FaceColor',fclr, ...
	'FaceLighting','phong','EdgeColor',edgecolor, ...
        'Tag', 'vb_plot_cortex_brain');
  
  % Light
  h = camlight('headlight');
else
  c = repmat(fclr,[length(inf_C) 1]) ...
      -0.4*[fclr(1)*inf_C fclr(2)*inf_C fclr(3)*inf_C];
  patch('Faces',FF,'Vertices',Vshrink,'FaceColor','interp',...
	'EdgeColor',edgecolor,'FaceVertexCData',c, ...
        'Tag', 'vb_plot_cortex_brain');
  h = [];
end

material dull;
hold on;

% --- Superimpose activity
if plot_parm.isact & ~isempty(J), 
  plot_value(plot_parm,V,F0,J,index,Jmin,Jmax);
end

% --- Superimpose area
if ~isempty(ix_area),
  if ~isempty(index), ix_area = intersect(index,ix_area); end
  FF = vb_patch_select2(ix_area,F0,size(V,1));
  patch('Faces',FF,'Vertices',V,'FaceColor','b',...
	'EdgeColor','none','EraseMode','xor', ...
        'FaceAlpha',1);
end

plot_area(plot_parm,V,F0);


% To avoid MATLAB alpha bug(R2014b to R2015b)
if vb_matlab_version('>=', '8.4') && vb_matlab_version('<=', '8.6')
    % reset alpha value of brain to 0.99
    t = timer('Name', 'vb_plot_cortex', 'StartDelay', 1, ...
          'TimerFcn', 'alpha(findobj(gca, ''Tag'', ''vb_plot_cortex_brain''), 0.99);', ...
          'ExecutionMode', 'singleshot');
    start(t);
end

return; 

% --- Inner function: plot cortical activity
function plot_value(plot_parm,V,F0,J,index,wmin,wmax);

if length(J) == 0
  return;
end

% Select vertices
if length(index) == 0
  tmp = ones(length(J),1);
else  
  tmp = zeros(length(J),1);
  tmp(index) = 1;
end

% Left/Right hemisphere
tmp2 = zeros(length(J),1);
tmp2(unique(F0(:))) = 1;

tmp = J.*tmp.*tmp2;
ix = find(abs(tmp) >= wmin);
if plot_parm.wmode==1
  J = abs(J);
end

% Superimpose activity on cortical surface
if plot_parm.paint==1
  h = scatter3(V(ix,1),V(ix,2),V(ix,3),1,J(ix),'.','filled');
  set(h,'MarkerSize',plot_parm.ms);
else  
  FF = vb_patch_select2(ix,F0,size(V,1));
  if ~isempty(FF), 
    patch('Faces',FF,'Vertices',V,'FaceColor','interp',...
          'FaceVertexCData',J,'EdgeColor','none',...
          'FaceLighting','none','FaceAlpha',1);
    material dull;
  end
end

if plot_parm.wmode==1
  caxis([0 wmax]);
else
  caxis([-wmax wmax]);
end
  
if plot_parm.cmap==1
  colormap(hot);
else
  colormap(jet);
end

return; 

% --- Inner function: plot cortical area
function plot_area(plot_parm,V,F0);

if isfield(plot_parm,'area'), 
  if ~isfield(plot_parm,'index') | isempty(plot_parm.index), 
    tmp = ones(size(V,1),1);
  else
    tmp = zeros(size(V,1),1);
    tmp(plot_parm.index) = 1;
  end
  
  % For compatibility with old format
  if isfield(plot_parm,'Rlabel'), 
    plot_parm.area.Rlabel = plot_parm.Rlabel;
  elseif ~isfield(plot_parm.area,'Rlabel'), 
    plot_parm.area.Rlabel = 1;
  end
  
  if isfield(plot_parm,'areafile'), 
    plot_parm.area.area_file = plot_parm.areafile;
  end

  % Transparency
  if isfield(plot_parm.area,'transparent') & plot_parm.area.transparent, 
    if isfield(plot_parm.area, 'tranceparent_value')
        alpha = plot_parm.area.tranceparent_value;
    else
        alpha = 0.5;
    end
  else
    alpha = 1.0;
  end

  % Plot area
  area_file = plot_parm.area.area_file;
  for i = 1:length(plot_parm.area.key)
    Area = vb_get_area(area_file,plot_parm.area.key{i});
    if isfield(Area,'clr'), 
      clr = Area.clr;
      if max(clr)>1 | min(clr)<0, 
        vb_disp(['Invalid color value of cortical region. area.clr must ' ...
                 'be in range [0 1], where area = vb_get_area(area_file,' ...
                 'area_key). Normalized in ' ...
                 'vb_plot_cortex.']);        
        clr = abs(clr)./max(abs(clr));
      end
    else
      Nclr = length(plot_parm.area.clr);
      clr = plot_parm.area.clr{1+mod(i-1,Nclr)};
    end
    ix = Area.Iextract;
    ix = ix(:);
    ix = ix(find(ix.*tmp(ix))); % Union

    % Dot/Face plot    
    if plot_parm.paint==1
      ix2 = intersect(ix,unique(F0(:)));
      plot3(V(ix2,1),V(ix2,2),V(ix2,3),'.',...
        'MarkerFaceColor',clr,'MarkerEdgeColor',clr, ...
        'MarkerSize',plot_parm.ms);
    else
      FF = vb_patch_select2(ix,F0,size(V,1));
      patch('Faces',FF,'Vertices',V,...
            'FaceColor',clr,'EdgeColor','none', ...
            'FaceLighting','gouraud','FaceAlpha',alpha);
    end

    % Display name of cortical area
    if isfield(plot_parm.area,'str') & ...
          ~isempty(plot_parm.area.str) & length(FF) > 0, 
      V0 = mean(V); % Center of cortical model
      pos = median(V(ix,:))-V0;
      [theta,phi,r] = cart2sph(pos(1),pos(2),pos(3));
      r = plot_parm.area.Rlabel*r;

      [x,y,z] = sph2cart(theta,phi,r);
      x = x+V0(1);
      y = y+V0(2);
      z = z+V0(3);
      pos = pos+V0;
      plot3([pos(1) x],[pos(2) y],[pos(3) z]);
      h = text(x,y,z,plot_parm.area.str{i},...
           'Interpreter','none','FontSize',plot_parm.fs,...
           'HorizontalAlignment','center');
    end
  end
end

return;