function [Y ,Itbl, Dim, Rmax] = vb_spherical_harmo_F(X,N,Rmax)
% calculate spherical harmonics
% [usage]
%   [Y ,Itbl, Dim, Rmax] = vb_spherical_harmo_F( X, N, Rmax )
% [input]
%   X : (NP*3) three-D coordinates of NP points
%   N : maximum order of spherical harmonics (generally 10~20)
%   Rmax : reference radius for normalization [m]
% [output]
%   Y   : Spherical harmonics (NP x D),   D = N*(N+2)
%  Itbl : Index table to map (n,m,0/1) to 'k' of Y(:,k)
%  Dim  : Degree of Legendre function
%  Rmax : reference radius for normalization [m]
%
% --- Example to get Spherical harmonics for (X ,n, m, c/s)
% k = Itbl(n+1,m+1,i+1)    for ( n = 0:N ; m = 0:N ; i=0,1:(c,s) )
% 
% Y(:,k) = Sharmonics(X ,n, m, i) for ( n = 0:N ; m = 0:n ; i=0,1:(c,s) )
%  if m > n, k = Itbl(n+1,m+1,i+1) = 0, which means Y(:,k) = 0
%
% Y(:,1) = Legendre(z ,n=0 ,m=0 ,i=0 ) = 1 
% Y(:,2) = Legendre(z ,n=1 ,m=0 ,i=0 )
% Y(:,3) = Legendre(z ,n=1 ,m=1 ,i=0 )
%
% Dim(1,k) = n : degree of polynomial r^n
% Dim(2,k) = m : degree of azimuthal angle rotation
% Dim(3,k) = 0 (:cos) or 1 (:sin)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if ~exist('N','var'), N=10; end;

NP = size(X,1);
D  = (N + 1)^2 ; % !)(2n+1) : n=0:N
Y  = zeros(NP,D);

Dim  = zeros(3,D);
Itbl = zeros(N+1,N+1,2);

% convert coordinates from cartisian to spherical
% theta  : $B6D3Q(B  (elevation angle)  : [-pi/2  pi/2 ] (0:$B?eJ?(B pi/2:$B??>e(B)
% phi    : $BJ}0L3Q(B(azimuthal angle)  : [   0   2*pi ] 
[phi, theta, r] = cart2sph(X(:,1),X(:,2),X(:,3));

% --- Legendre basis function 
% k = ix_tbl(n+1,m+1)        for ( n = 0:N ; m = 0:N )
% P(:,k) = Legendre(z ,n, m) 
%    If m > n,  P(:,k) = 0

z  = X(:,3)./r; % = cos(pi/2 - theta) (NP * 1)

[P ,ix_tbl] = vb_legendre_basis(z,N);

% --- Sin/Cos function
% cos( m * phi ) = cos_Phi( m+1 )  for ( m = 0:N )
% sin( m * phi ) = sin_Phi( m+1 )  for ( m = 0:N )

Phi = phi * (0:N);  % (NP * (N+1))
cos_Phi = cos(Phi); % (NP * (N+1))
sin_Phi = sin(Phi); % (NP * (N+1))


if ~exist('Rmax','var'), Rmax=sum(r)/NP; end;

r  = r / Rmax; % (NP * 1)
Rn = r;;

% Sharmonics(X ,n=0 ,m=0 , i=0:(cos) ) = 1 
Itbl(1,1,1) = 1;
Dim(:,1)    = [0; 0; 0];

nn = 1;
Y(:,1) = 1;

for n = 1:N

	% Cos-part
	for m = 0:n
	    % Y(X,{n,m,cos})
	    nn = nn + 1;
	    k  = ix_tbl(n+1,m+1);
	    Y(:,nn) = Rn .* cos_Phi(:,m+1) .* P(:,k); 
	    Dim(:,nn) = [n; m; 0];
	    Itbl(n+1,m+1,1) = nn;
	end
	
	% Sin-part
	for m = 1:n
	    % Y(X,{n,m,sin})
	    nn = nn + 1;
	    k  = ix_tbl(n+1,m+1);
	    Y(:,nn) = Rn .* sin_Phi(:,m+1) .* P(:,k); 
	    Dim(:,nn) = [n; m; 1];
	    Itbl(n+1,m+1,2) = nn;
	end
	
    Rn = Rn .* r;
end

if nn ~= D, fprintf('Number of function = %d\n',nn); end;
