function [P ,Itbl, Dim] = vb_legendre_basis( z, N ,mode)
% Calculate Legendre basis function (Normalized)
% [usage]
%   [P ,Itbl, Dim] = vb_legendre_basis( z, N )
% [input]
%   z : (NP x 1) normalized z-coordinates of NP points
%   N : maximum order of Legendre function
% mode : normalization mode , 
%      = 'norm' : normalized (default), '' : unnormalized
% [output]
%   P    : Legendre basis function 
%  Itbl  : Index table to map (n,m) to 'k' of P(:,k)
%  Dim   : Degree of Legendre function
%
% --- Example to get Legendre(z ,n, m)
% k = Itbl(n+1,m+1)        for ( n = 0:N ; m = 0:N )
% 
% P(:,k) = Legendre(z ,n, m) for ( n = 0:N ; m = 0:n )
%  if m > n, k = Itbl(n+1,m+1) = 0, which means P(:,k) = 0
%
% P(:,1) = Legendre(z ,n=0 ,m=0 ) = 1 
% P(:,2) = Legendre(z ,n=1 ,m=0 )
% P(:,3) = Legendre(z ,n=1 ,m=1 )
%     ...
% P(:,D) = Legendre(z ,n=N ,m=N )
%     D  = (N+1) * (N+2)/2 
%
% Dim(1,k) = n : degree of legendre function
% Dim(2,k) = m : degree of azimuthal angle rotation
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if ~exist('N','var'), N=10; end;
if ~exist('mode','var'), mode = 'norm'; end;

NP = length(z);

% number of legendre basis function
D  = (N+1) * (N+2)/2 ;
P  = zeros(NP,D);

Dim  = zeros(2,D);
Itbl = zeros(N+1,N+1);

% Legendre(z ,n=0 ,m=0 ) = 1 
Itbl(1,1) = 1;
P(:,1)    = 1;

i_st = 2;

for n = 1:N
    % Legendre(z ,n, m) : n른ɥؿ
    % i_st  : Start index for n-th Legendre function
    % i_end :   End index for n-th Legendre function

    i_end = i_st + n;
    
    % Make index table
    Dim( 1, i_st:i_end) = n;
    Dim( 2, i_st:i_end) = 0:n;
    Itbl( n+1, 1:n+1)   = i_st:i_end;
    
    % Legendre( z, n, m = 0:n )
    switch	mode
    case	''
	    P(:,i_st:i_end) = legendre(n,z)'; 
	case	'norm'
    	P(:,i_st:i_end) = legendre(n,z,'norm')'; 
    end
    
    i_st = i_end+1;
end
