function f = vb_multi_shell_param(R, sigma, Nmax)
% Legendre Expansion Weights for EEG multilayer spherical forward model 
% f = vb_multi_shell_param(R, sigma, Nmax)
% 
% INPUTS
%    R    : Radii of sphere from innermost to outermost    ( NL x 1 )
%   sigma : Conductivity    from innermost to outermost    ( NL x 1 )
%   Nmax  : # of Legendre Expansion     (Default: 200)      ( scalar )
%  
%     NL = # of sphere layers; 
%
% OUTPUTS:
%     f: Legendre Expansion Weights for multishell model  ( Nmax x 1 )
%
%
% Single Dipole in a multilayer sphere can be approximated 
%        by multiple dipoles in a single shell. 
% Z. Zhang,  Phys. Med. Biol., vol.40, pp. 335-349, 1995.
%
% IEEE TRANSACTIONS ON BIOMEDICAL ENGINEERING, VOL.46, 245-259, 1999
% John C. Mosher, Richard M. Leahy and Paul S. Lewis
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% # of sphere layer
NL = length(R);

if nargin < 3    
    % Default # of Legendre expansion
    Nmax = 200;  
end

% Radius of outermost layer (Sensor distance from origin)
Rmax = R(NL);     

%
% Computes the weights f(n) for a legendre expansion
% f(n) depend only on the multisphere radii and conductivity

if NL==1
   f=ones(Nmax,1);
else
	
	for k = 1:NL-1 
	   s(k) = sigma(k)/sigma(k+1);
	end
	
	a    = Rmax./R;
	ainv = R/Rmax;
	sm1  = s-1;
	npow = 2*[1:Nmax]+1;
	npow = npow(:);
	
	f = zeros(Nmax,1);
	
	for n = 1:Nmax
		np1 = n+1;
		Mc = eye(2);

		%  Eq.(17)
		for k = 2:NL-1
		   Mc = Mc*[n+np1*s(k),  np1*sm1(k)*a(k)^npow(n);...
		            n*sm1(k)*ainv(k)^npow(n) , np1+n*s(k)];
		end

		% Eq.(16)
		Mc(2,:) = [n*sm1(1)*ainv(1)^npow(n) , np1+n*s(1)]*Mc; 
		Mc = Mc/(npow(n))^(NL-1);
		f(n) = n/( n*Mc(2,2) + np1*Mc(2,1) );
	end
end

