function delta = vb_berg_error(berg, R, f)
% Error function to optimize Berg parameters for multilayer sphere model
%
% INPUTS:
%    berg : Berg parameters                                  (2*NJ-1) x 1
%    R    : Sphere Radii from innermost to outermost               NL x 1
%    f    : Legendre Expansion Weights to fit Berg Parameters  ( Nmax x 1 )
% 
%        NL = # of sphere layers; 
%      Nmax = # of Legendre Expansion
%        NJ = # of Berg Dipoles
%
% OUTPUTS:
%   delta: Error function value to be minimized
%
% Z. Zhang,  Phys. Med. Biol., vol.40, pp. 335-349, 1995.
%
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

NL	 = length(R);               % Number of Sphere Layers
Nmax = length(f);               % Number of Legendre Series Terms
NJ	 = (length(berg)+1)/2;      % Number of Berg Dipoles
mu	 = berg(1:NJ);              % Berg Eccentricity Factors
lam  = [0.0 berg(NJ+1:2*NJ-1)];	% Berg Magnitude Factors

% Error function defined by Zhang Eq. (5i")
delta = 0.0;
r     = R(1)/R(NL);

for n = 2:Nmax
   err0 = 0;
   for j=2:NJ
      err0 = err0 + lam(j)*( mu(j)^(n-1) - mu(1)^(n-1) );
   end
   
   err   = ( r^(n-1) )*( f(n) - f(1)*mu(1)^(n-1) - err0 );
   delta = delta + err*err;
end

