function [VBfilt,Jinfo,bayes_parm] = vb_job_vbfilter(varargin)
% Calculate inverse filter for current estimation. 
% (VBMEG public function)
%
% [syntax]
% [VBfilt,Jinfo,bayes_parm] = vb_job_vbfilter(filt_parm)
% [VBfilt,Jinfo,bayes_parm] = vb_job_vbfilter(proj_root,filt_parm)    [old style]
%
% [input]
% proj_root: <optional> <<string>> VBMEG project root directory. 
% filt_parm: <<struct>> Parameters for calculation of inverse filter. 
% --- fields of filt_parm
%  bayesfile : <<string>> Current variance file (.bayes.mat). 
%  filterfile: <optional> <<string>> inverse filter file (.vbfilt.mat). If
%              this field is not defined, inverse filter file is not
%              saved. 
%  basisfile : <optional> <<string>> If this field is given, leadfield
%              file of this field is used for calculation of inverse
%              filter. 
%  area_key  : <optional> <<string>> Identifier of interested area. If
%              'area_key' is empty or not given, inverse filter for the
%              (focal) active region are calculated. When using
%              vb_calc_current.m, you must NOT to set this parameter. 
%  areafile  : <optional> <<string>> Cortical area file (.area.mat). 
% ---
%
% [output]
% If input parameter 'filterfile' is given, the following variables are
% saved into the file with filename 'filterfile'. 
%
% VBfilt    : <<struct>> Inverse filter and related variables.
% --- fileds of VBfilt
%  KW       : Inverse filter (Njact,Nch,Nwindow,Ntask)
%  W        : <<matrix>> Spatial smoothing matrix of focal window.
%  ix_act   : <<vector>> Vertex index of Z-current. 
%  ix_act_ex: <<vector>> Vertex index of J-current. 
%  L        : <<int>> Number of current direction. 
%  a        : Current variance (Njact,Nwindow,Ntask)
%  sx       : Observation noise variance (Nwindow,Ntask)
%  Cov      : Normalized sensor noise covariance matrix {Ntask}(Nch,Nch)
%  Hj       : Model entropy (Nwindow,Ntask)
%  SB_cov   : posterior sensor covariance matrix (Nch,Nch,Nwindow,Ntask)
% ---
%
% Jinfo     : <<struct>> Information of cortical current. 
% --- fileds of Jinfo
%  Lact        : Number of current direction. 
%  ix_act      : Vertex index corresponding to active current Zact.
%  ix_act_ex   : Vertex index corresponding to active current Jact.
%  Wact        : Spatial smoothing matrix of focal window.
%  channel_id  : Active channel ID for VB-filter calculation.
%  channel_name: Active channel name for VB-filter calculation.
% ---
%
% bayes_parm: <<struct>> Parameters for variational Bayes method (same
%             with the input variable of vb_job_vb.m). 
%
% [history]
% 2008-08-01 M. Sato
% 2009-08-21 M. Sato
% 2010-06-21 Taku Yoshioka
%  Minor change (verbose level support)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% Constants
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
const = vb_define_verbose;
VERBOSE_LEVEL_NOTICE = const.VERBOSE_LEVEL_NOTICE;

%
% Check input arguments
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if length(varargin) == 1
  proj_root = [];
  filt_parm = varargin{1};
elseif length(varargin) == 2
  proj_root = varargin{1};
  filt_parm = varargin{2};
else
  error('Error: invalid usage of job_vbfilter.');
end

proj_root = vb_rm_trailing_slash(proj_root);

%
% Filename of inverse filter file (.vbfilt.mat)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~isfield(filt_parm,'filterfile') | isempty(filt_parm.filterfile)
  if isfield(filt_parm,'currfile') & ~isempty(filt_parm.currfile), 
    ix_ext = findstr(filt_parm.currfile, '.curr.mat');
    filterfile = [filt_parm.currfile(1:ix_ext-1) '.vbfilt.mat'];
  else
    filterfile = [];
  end
else
  filterfile = filt_parm.filterfile;
end

if ~isempty(proj_root) & ~isempty(filterfile), 
  filterfile = fullfile(proj_root, filterfile);
end

%
% Inverse filter calculation
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[VBfilt, Jinfo, bayes_parm, vb_parm] ...
    = vbmeg_current_filter_z(proj_root, filt_parm);
VBfilt.L = Jinfo.Lact;
VBfilt.W = Jinfo.Wact;
VBfilt.ix_act = Jinfo.ix_act;
VBfilt.ix_act_ex = Jinfo.ix_act_ex;

ch_info = vb_load_channel_info(fullfile(proj_root, bayes_parm.megfile{1}));

Jinfo.channel_id   = ch_info.ID  ;  % channel ID 
Jinfo.channel_name = ch_info.Name;  % channel name 

%
% Save inverse filter file
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~isempty(filterfile), 
  vb_disp(sprintf('--- Save VB-filter file:\n %s',filterfile), ...
          VERBOSE_LEVEL_NOTICE);
  vb_fsave(filterfile,'VBfilt','Jinfo','bayes_parm','vb_parm');
end

