function vb_job_leadfield(varargin)
% Calculate leadfield matrix and create leadfiled file (.basis.mat). 
% (VBMEG public function)
%
% This function can calculate leadfield matrix for using the following
% numerical methods: 
%  MEG Sphere model (Sarvas equation)
%  MEG 1-shell BEM (boundary element method)
%  MEG Spherical harmonics expansion
%  EEG 3-shell Sphere model
%  EEG 3-shell BEM
%
% [syntax]
% vb_job_leadfield(basis_parm)
% vb_job_leadfield(proj_root, basis_parm)   [old style]
%
% [input]
% proj_root : <<string>> VBMEG project root directory. 
% basis_parm: <<struct>> Parameters for leadfield calculation. 
% --- fields of basis_parm
%  brain_file : <<string>> Cortical surface model file (.brain.mat).
%  area_file  : <<string>> Cortical area file (.area.mat).
%  meg_file   : <<string>> MEG-MAT or EEG-MAT file (.meg.mat).
%  basis_file : <<string>> Leadfield file (.basis.mat). 
%  bem_mode   : <<int>> Specify numerical method for calculating
%               leadfield. 
%   =0: MEG Sphere model (Sarvas equation).
%   =1: MEG 1-shell BEM (boundary element method).
%   =2: MEG Spherical harmonics expansion.
%   =3: EEG 3-shell Sphere model.
%   =4: EEG 3-shell BEM.
%  head_file  : <optional><<string>> Head model file (.head.mat). This
%               parameter is referred if the lead field is calculated by
%               BEM or spherical harmonics expansion (see 'bem_mode'). 
%               expansion 
%  Basis_mode : <<int>> Degree of freedom of current direction. Not only
%               the DOF, but also the direction of current dipoles at each
%               vertex is also specified by this parameter. Currently
%               VBMEG supports Basis_mode=1 (1st June, 2010). 
%   =1: Single dipole perpendicular to the cortical surface. 
%   =2: Double dipoles tangental to the cortical surface. 
%   =3: Triple dipoles parallel to three axes (x, y, z). 
%  Vcenter    : <<vector>> Coordinate value of the head center (1x3 on
%               SPM_Right_m).
%  func_order : <<int?>> Order of spherical harmonics function. This
%               parameter is effective when employing spherical harmonics
%               expansion (see 'bem_mode'). 
%  sigma      : <<vector>> Conductivity from innermost to outermost. 
%  radius     : <<vector>> Relative radius of sphere from innermost to
%               outermost. 
%  area_key   : <optional><<string>>
%  normal_mode: <optional><int>
%  Recalc     : <optional>
% ---
%
% [output]
% Leadfield file (.basis.mat) is created with filename specified by
% 'basis_file' field. If BEM is used, head model file (.head.mat) is
% updated with a matrix, which depends only on the head model, not on the
% cortical surface model. So, if size of .head.mat file become large, it
% is not an error. 
%
% [note]
% This function call the following sub-functions in accordance with
% basis_parm.bem_mode: 
%  vb_job_leadfield_meg_sphere
%  vb_job_leadfield_meg_bem
%  vb_job_leadfield_meg_harmonics
%  vb_job_leadfield_eeg_sphere
%  vb_job_leadfield_eeg_bem
%
% 'vbmeg_inst' structure have constants for specifying the numerical
% method and model related to leadfield calculation. Those constants are
% recommended to specify parameters for readability of your program. 
% --- Constants for 'basis_parm.bem_mode'
%  vbmeg_inst.const.BASIS_MEG_SPHERE    = 0
%  vbmeg_inst.const.BASIS_MEG_BEM       = 1
%  vbmeg_inst.const.BASIS_MEG_HARMONICS = 2
%  vbmeg_inst.const.BASIS_MEG_SPHERE    = 3
%  vbmeg_inst.const.BASIS_MEG_BEM       = 4
% --- Constants for 'basis_parm.Basis_mode'
%  vbmeg_inst.const.CURRENT_NORMAL  = 1
%  vbmeg_inst.const.CURRENT_TANGENT = 2
%  vbmeg_inst.const.CURRENT_3D      = 3
% --- Constants for 'basis_parm.normal_mode
%  vbmeg_inst.const.NORMAL_ADJACENT_MEAN = 0
%  vbmeg_inst.const.NORMAL_NEIGHBOR_MEAN = 1
% ---
%
% [history]
%   2007-01-26 (Sako) reworked the architecture
% 2008-10-15 Taku Yoshioka
%  Support '.sensortype' parameter (argument check)
% 2010-05-?? Taku Yoshioka
%  Minor change (verbose level support)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

global vbmeg_inst;
const = vbmeg_inst.const;

if length(varargin) == 1
  proj_root = [];
  basis_parm = varargin{1};
elseif length(varargin) == 2
  proj_root = varargin{1};
  basis_parm = varargin{2};
end

% --- CHECK ARGUMENTS --- %
if ~exist('proj_root', 'var'), proj_root = []; end
if ~exist('basis_parm', 'var'), basis_parm = []; end
[proj_root, basis_parm] = inner_check_arguments(const, proj_root, basis_parm);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
switch basis_parm.bem_mode
  case const.BASIS_MEG_SPHERE, 
    [basis, basis_parm] = vb_job_leadfield_meg_sphere(proj_root, basis_parm);
    
  case const.BASIS_MEG_BEM, 
    [basis, basis_parm] = vb_job_leadfield_meg_bem(proj_root, basis_parm);
  
  case const.BASIS_MEG_HARMONICS, 
    [basis, basis_parm] = vb_job_leadfield_meg_harmonics(proj_root, basis_parm);

  case const.BASIS_EEG_SPHERE, 
    [basis, basis_parm] = vb_job_leadfield_eeg_sphere(proj_root, basis_parm);

  case const.BASIS_EEG_BEM, 
    [basis, basis_parm] = vb_job_leadfield_eeg_bem(proj_root, basis_parm);
end

inner_terminate(const, proj_root, basis_parm, basis);
%
% --- MAIN PROCEDURE --------------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_terminate()
%
function inner_terminate(const, proj_root, basis_parm, basis)

% const.BASIS_MEG_SPHERE;   %  MEG Sphere model (Sarvas)
% const.BASIS_MEG_BEM;      %  MEG BEM
% const.BASIS_MEG_HARMONICS;%  MEG Spherical harmonics expansion
% const.BASIS_EEG_SPHERE;   %  EEG 3-shell Sphere model
% const.BASIS_EEG_BEM;      %  EEG 3-shell BEM

bem_name = { ...
	'MEG spherical model (Sarvas)'; ...
	'MEG BEM 1 shell model'; ...
	'MEG spherical harmonics expansion'; ...
	'EEG spherical 3 shell model'; ...
	'EEG BEM 3 shell model'; ...
};

switch basis_parm.bem_mode
  case  const.BASIS_MEG_SPHERE
    calc_type = bem_name{1};
    
  case  const.BASIS_MEG_BEM
    calc_type = bem_name{2};
    
  case  const.BASIS_MEG_HARMONICS
    calc_type = bem_name{3};
    
  case  const.BASIS_EEG_SPHERE
    calc_type = bem_name{4};
    
  case  const.BASIS_EEG_BEM
    calc_type = bem_name{5};
    
  otherwise
    calc_type = 'unknown calculation type';
end

vb_disp(sprintf('Leadfield calculation : %s', calc_type));
vb_disp(sprintf('Current direction = %d',basis_parm.Basis_mode));

if isfield(basis_parm, 'normal_mode') ...
      & basis_parm.normal_mode == const.NORMAL_NEIGHBOR_MEAN,
  vb_disp('Normal vector is averaged over neighborhood');
end

% Output file path
basis_file = fullfile(proj_root, basis_parm.basis_file);
vb_disp(sprintf('Save basis file [%s]', basis_file));
vb_save(basis_file,'basis','basis_parm');

% project_file save
proj_file = get_project_filename;
if isempty(proj_file)
    return;
end

project_file_mgr('load', proj_file);
project_file_mgr('add', 'basis_parm', basis_parm);
%
% --- end of inner_terminate()


% --- inner_check_arguments()
%
function [proj_root, basis_parm] = ...
  inner_check_arguments(const, proj_root, basis_parm)

if isempty(basis_parm), error('basis_parm is a required parameter'); end;

proj_root = vb_rm_trailing_slash(proj_root);

% check common fields of basis_parm
if ~isfield(basis_parm, 'normal_mode')
 	basis_parm.normal_mode = const.NORMAL_ADJACENT_MEAN;
end

if ~isfield(basis_parm,'Recalc'),
	% Recalculation force flag
	basis_parm.Recalc = OFF;
end

if ~isfield(basis_parm,'DEBUG'),
	basis_parm.DEBUG = OFF;
end

if ~isfield(basis_parm,'sensortype'), 
  basis_parm.sensortype = [];
end

% if you want to check other points, you can do that here
% ...

%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
