function vb_job_fmri(varargin)
% Import fMRI results to VBMEG cortical act file(.act.mat) and area file(.area.mat)
% Support SPM earlier than spm8, tested with spm8  
%
% [Syntax]
%    vb_job_frmi(fmri_parm);
%    vb_job_fmri(proj_root, fmri_parm);   [old style]
%
% [Input]
%    proj_root : <<string>> VBMEG project root directory.
%    fmri_parm : Import parameters for fmri.
%
% [Output]
%    none
%
% [History]
%   2016/11/21 rhayashi
%     [Change] spec of SPMmapmode changed more clearly e.g. 'mni' -> 'mni2subj'
%   2016/07/26 O.Yamahista
%     [Change] spm_ver is obtained from SPM.mat 
%   2016/07/12 O.Yamahista
%     [Fix] bug fix when importing SPM maps on MNI brain to the MNI brain model 
%   2016/07/05 O.Yamashita
%     [Add] process to project normalized SPM maps on the MNI brain model
%   2016/07/03 O.Yamashita
%     initial version
%
% [Note]
%   @see vb_set_fmri_parm.m
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
spm_func = which('spm_vol');
if isempty(spm_func)
    error('Please add SPM8 path to your MATLAB.');
end
if nargin == 1
    proj_root = [];
    parm = varargin{1};
elseif nargin == 2
    proj_root = varargin{1};
    parm      = varargin{2};
else
    error('Invalid usage of vb_job_fmri.');
end

proj_root = vb_rm_trailing_slash(proj_root);

%
% --- Main Procedure
%

% PARAMETERS
% SPM results
SPMmat_file = parm.SPMmat_file;
brain_file  = fullfile(proj_root, parm.brain_file);

% Resutls output 
act_file    = fullfile(proj_root, parm.act_file);
area_file   = fullfile(proj_root, parm.area_file);
keys_postfix = parm.keys_postfix; 

% function mode
Tthres  = parm.Tthres;      % important parameters
Rthres  = parm.Rthres;      % unit:mm
PSthres = parm.PSthres;     % unit:%
SPMmapmode = parm.SPMmapmode;


switch lower(SPMmapmode)
    case {'subj2subj'}
        % SPM results on Subject brain
        % need input parameter check
        SPMimginfo = SPMmat2SPMimginfo(SPMmat_file, ['_T' keys_postfix], 1);
        vb_fmri_project_Tmap_cortex(brain_file,act_file,area_file,Tthres,Rthres,SPMimginfo);

        SPMimginfo = SPMmat2SPMimginfo(SPMmat_file, ['_PS' keys_postfix],0);
        vb_fmri_project_PSmap_cortex(brain_file,act_file,area_file,PSthres,Rthres,SPMimginfo);

    case {'mni2subj'}
        % SPM results on MNI brain --> Subject brain
        %  need input parameter check
        sn_file     = parm.spm_normalization_file;
        SPMimginfo = SPMmat2SPMimginfo(SPMmat_file, ['_T' keys_postfix], 1);
        vb_fmri_unsn_project_Tmap_cortex(brain_file,act_file,area_file,sn_file,Tthres,Rthres,SPMimginfo);

        SPMimginfo = SPMmat2SPMimginfo(SPMmat_file, ['_PS' keys_postfix],0);
        vb_fmri_unsn_project_PSmap_cortex(brain_file,act_file,area_file,sn_file,PSthres,Rthres,SPMimginfo);

    case {'mni2mni'}
        %  SPM results on MNI brain --> MNI brain model
        fprintf(' SPM maps on MNI brain will projected to the MNI standard brain model ! \n');

        SPMimginfo = SPMmat2SPMimginfo(SPMmat_file, ['_T' keys_postfix], 1);
        vb_fmri_project_Tmap_cortex(brain_file,act_file,area_file,Tthres,Rthres,SPMimginfo,1);

        SPMimginfo = SPMmat2SPMimginfo(SPMmat_file, ['_PS' keys_postfix],0);
        vb_fmri_project_PSmap_cortex(brain_file,act_file,area_file,PSthres,Rthres,SPMimginfo,1);

    otherwise
        error('SPMmapmode must be either ''subj2subj'', ''mni2subj'' or ''mni2mni'' !');
end

%--------------------------
% Save project file
%--------------------------
proj_file = get_project_filename;
if ~isempty(proj_file)
    project_file_mgr('load', proj_file);
    project_file_mgr('add', 'fmri_parm', parm);
end



%
% inner function
%

function SPMimginfo = SPMmat2SPMimginfo(SPMmatfile,postfix,flagT)

load(SPMmatfile);

spm_ver = SPM.SPMid(1:strfind(SPM.SPMid,':')-1);

for nn = 1 : length(SPM.xCon)

    % image labels
    labels{nn} = [SPM.xCon(nn).name postfix];

    % image filenames
    if flagT  % T-map
        filenames{nn}   = [SPM.xCon(nn).Vspm.fname];
    else     % Con-map
        filenames{nn}   = [SPM.xCon(nn).Vcon.fname];
    end

    % ncon
    ncon(nn)   = sum(SPM.xCon(nn).c > 0);
    % peak
    ixreg = find(SPM.xCon(nn).c);
    X     = SPM.xX.X(:,ixreg);
    peak(nn) = max(X(:));
    % baseline img
    a = strfind(SPM.xX.name, 'constant');
    for ii = 1 : length(a)
        if ~isempty(a{ii})
            ixreg = ii;
        end
    end
    baseimgfiles{nn} = ['beta_' num2str(ixreg, '%04d') '.img'];

end

spmdir = [fileparts(SPMmatfile) filesep];

for nn = 1 : length(SPM.xCon)
    SPMimginfo(nn) = struct('dirname', spmdir, 'imgfilename', filenames{nn}, 'imglabel', labels{nn},...
        'ncon',ncon(nn),'peak', peak(nn), 'baseimgfilename', baseimgfiles{nn},'spm_ver',spm_ver);
end