function  [fileinfo,flg] = vb_job_reject_channels_trials_gui(datafile, amp_thres,...
	thred_val, per_ch,per_tr, rej_criterion)
% Find bad channel/trials based on maximum amplitude and 1st differece
% signals. This program is based on vb_job_bad_trial_interact.m but run structure is completely ignored. 
%
% [Usage]
%  > vb_job_reject_channels_trials_gui(datafile, 2e-12)  for gradio-MEG
%  > vb_job_reject_channels_trials_gui(datafile, 1e-4)   for EEG
%
% [Input]
%   datafile: cell string for multiple MEG/EEG epoch data files
%   amp_thres : amplitude threshold in abolute range. 
%               For axial-gradio MEG data, magnetic field more than 1~2pT/cm is not considered from brain. 
%               Thus amp_thres = 2e-12 (T/cm) may be a reasonable choice.
%               For EEG data, electric potential more than 100 microV is
%               not considered from brain. Thus amp_thres = 1e-4 may be a
%               reasonable choice.
%
% [Optional Input]
%   thres_val : threshold value for amplitude and 1st differential signals
%               see below for more details  {Def:[9 8]}
%   per_ch   : percent threshold for channel rejection. {Def : 0.05}
%   per_tr   : percent threshold for trial rejection. {Def: 0.05}
%   rej_criterion : rejection criterion {Def:0}. 
%                  = 0 uses criterion 1 and 2 below.
%                  = 1 uses criterion 1, 2 and 3 below.
%                
% * Rejection criteria
%   Timeseries at a specified channel and trial is marked as bad if either
%   of following three conditions is satisfied. 
%   1. maximum ampulitude exeeds amp_thres.  Note that this is absolute value threshold. 
%     This value must be adjusted accroding to measurement dynamic range (EEG or MEG).
%   2. maximum amplitude relative to standard deviation (within sessions or
%   all session average) exceed thres_val(1)
%   3. maximum 1st differential siganal relative to standard deviation
%   (within sessions or all session average) exceed thres_val(2)
%  
% 2014/09/14 Okito Yamashita
% 2009/02/01 Masa-aki Sato
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if isempty(datafile) | nargin < 2
 eval(['help vb_job_reject_channels_trials_gui']);
 error('see help comments above !');
end

if ~isempty(datafile) && iscell(datafile),
	error('datafile is a string specifying one data file ! \n');
end;

if nargin < 6 | isempty(rej_criterion) 
    rej_criterion = 0;
end
if nargin < 5 | isempty(per_tr)
    per_tr = 0.05;
end
if nargin < 4 | isempty(per_ch)
    per_ch = 0.05;
end

if nargin < 3 | isempty(thred_val)
    thred_val = [9 8];
end


fprintf('----------------------------------------------------------\n');
fprintf(' Trial and channel rejections (ver.1) \n');
fprintf('----------------------------------------------------------\n');
fprintf('  Absolute amplitude threshold : %e \n', amp_thres);
fprintf('  Relative amplitude threshold : %f \n', thred_val(1));
if rej_criterion == 1, fprintf('  Relative 1st differential signal threshold : %f \n', thred_val(2)); end
fprintf('  Percent threshold for channels : %f \n', per_ch);
fprintf('  Percent threshold for trials   : %f \n', per_tr);


% ystd: amplitude mean calculated by histrgam for all trials in one channel
% ymax : Nch x Ntrial
% ystd : Nch x 1
	
load(datafile, 'fileinfo'); 
if ~exist('fileinfo', 'var')
    error('datafile is not ''fileinfo'' format.');
end

% for compatibility
if isfield(fileinfo, 'channel_id')
    data_format = 'old';
elseif isfield(fileinfo, 'ActiveChannel')
    data_format = 'new';
end
if strcmp(data_format, 'new')
    fileinfo = vb_fileinfo_active_field_convert_to('old', fileinfo, datafile);
end

ixchact = fileinfo.channel_id;
ixtract = fileinfo.act_trial;
    
if isfield(fileinfo,'ymax1')
    fprintf('  Load data  \n')
    ymax1 = fileinfo.ymax1;
    ymax2 = fileinfo.ymax2;
    ystd1 = fileinfo.ystd1;
    ystd2 = fileinfo.ystd2;
    
    data = vb_load_meg_data(datafile);
        
else
    fprintf('    Load data and compute statistics \n')
    data = vb_load_meg_data(datafile);
    [ymax1,ymax2,ystd1,ystd2] = vb_channel_statics(data);

	fprintf('\n')
end

%----------------------------
% standardization 
%----------------------------

ratio1 = ymax1;
ratio2 = ymax2;

ix1  = find(ystd1 > 0);
ix2  = find(ystd2 > 0);

if size(ystd1,2) == 1
	ratio1(ix1,:) = vb_repmultiply(ymax1(ix1,:), 1./ystd1(ix1));
	ratio2(ix2,:) = vb_repmultiply(ymax2(ix2,:), 1./ystd2(ix2));
else
	ratio1(ix1) = ymax1(ix1)./ystd1(ix1);
	ratio2(ix2) = ymax2(ix2)./ystd2(ix2);
end

%
[Nch,T,Ntry] = size(data);
fprintf('#Channel = %d, #Time Points = %d, #Trials = %d \n',Nch,T,Ntry)

[Nch,Ntry] = size(ymax1);

%---------------------------------------------------------
% detect rejected channels/trials and visual inspection
%---------------------------------------------------------


fprintf('Channel rejection, Then Trial rejection \n')

% Find Bad Channel with large max_ratio
flg1 = ymax1 > amp_thres;
flg2 = ratio1 > thred_val(1);
flg3 = ratio2 > thred_val(2);
if rej_criterion == 0
    flg = flg1+flg2;
else
    flg = flg1+flg2+flg3;
end
flg = flg > 0;
figure, imagesc(flg), colormap('hot'); xlabel('trial indicies'); ylabel('channel indices');
fprintf('press any keys \n');
pause


%----------------------------
% visual confirmation
%----------------------------
ch_bad = find( sum(flg, 2) > per_ch*size(flg,2));

% Check Bad Channels
if ~isempty(ch_bad)
ch_bad = vb_plot_bad_channels2(data,ch_bad,flg,'ylimits', [-amp_thres amp_thres],'label',ixchact);
end
ch_act  = vb_setdiff2([1:Nch],ch_bad);
fprintf('   rejected channels %d \n', ixchact(ch_bad));
fprintf('\n')

% Find bad trials with large max_ratio after removing bad channel
flg_tmp = flg(ch_act,:);
try_bad = find( sum(flg_tmp, 1) > floor(per_tr*Nch));

% Check Bad trials
if ~isempty(try_bad)
try_bad = vb_plot_bad_trials2(data(ch_act,:,:),try_bad,flg(ch_act,:), 'ylimits', [-amp_thres amp_thres],'label',ixtract);
end
try_act = vb_setdiff2([1:Ntry],try_bad);
fprintf('   rejected trials %d \n', ixtract(try_bad));
fprintf('\n')

%-------------------------
%  Save Results
%-------------------------

Ntry = length(try_act);
Nact = length(ch_act);


% channel & trial statistics
fileinfo.ymax1 = ymax1;
fileinfo.ymax2 = ymax2;
fileinfo.ystd1 = ystd1;
fileinfo.ystd2 = ystd2;

fileinfo.channel_id = fileinfo.channel_id(ch_act);
fileinfo.act_trial  = fileinfo.act_trial(try_act);
fileinfo = vb_fileinfo_active_field_convert_to(data_format, fileinfo, datafile);

%% ---- END ---- %%
