function vb_job_ersp(proj_root,tf_parm)
% Calculate ERSP for cortical current.
% (VBMEG public function)
%
% This function calculates event-related spectral perturbation (ERSP;
% Delorme and Makeig, 2004) for cortical current. The value is
% baseline-normalized and its unit is dB (10log10). Analyses can be done
% on the original subject's cortex or on the other subject's cortex
% (commonly template cortical model like fsaverage). 
%
% [syntax]
% vb_job_ersp(proj_root,tf_parm)
%
% [input]
% proj_root: <<string>> VBMEG project root directory. 
% tf_parm  : <<struct>> Parameters for time-frequency analysis
% --- fields of fs_parm
%  curr_file     : <<string>> Cortical current file. 
%  curr_file_base: <optional> <<string>> Cortical current files from
%                  which baseline current amplitude is calculated. If not
%                  given, curr_file is used to calculate baseline. 
%  brain_file    : <<string>> Cortical surface model file on which
%                  cortical current was calculated (subject's brain). 
%  key           : <<string>> ID of coregistration matrix for each subjects
%                  corresponding to brain files. In common, coregistration
%                  matrix from the subject to a template cortical model is
%                  specified. 
%  tbrain_file   : <<string>> Cortical surface model filename of template
%                  brain. This file is used to reduce the number of
%                  vertices. 
%  reduce        : <<int or double>> Vertex reduction ratio 
%                  (@see vb_reducepatch). 
%  fmin          : <<double>> Minimum frequency.
%  fmax          : <<double>> Maximum frequency.
%  baseline      : <<double vector>> Start and end of baseline time window
%                  in millisecond. 
%  elim_time     : <optional> <<double>> In order to eliminate edge
%                  effect, time points with length (elim_time)*(original
%                  time length) are eliminated from start and end of data
%                  after time-frequency decomposition (wavelet). Default
%                  value is 0.05. 
%  tf_file_out   : <<string>> Output filename. 
%  timefreq_parm : <optional> <<struct>> EEGLAB timefreq parameters. If
%                  this parameter is given, elim_time is ignored. 
%  --- fields of timefreq_parm
%   cycles        : <<scalar or vector>> Parameter of timefreq function. 
%   winsize       : <<scalar>> Parameter of timefreq function. 
%   padratio      : <<scalar>> Parameter of timefreq function. 
%   ntimesout     : <<int>> Parameter of timefreq function.
%   nfreqs        : <<int>> Parameter of timefreq function. 
%   ntimesout_base: <<int>> Parameter of timefreq function for baseline. 
%  --- 
% ---
%
% [output]
% TimeFrequency file (.tf.mat) is created. This file has the following
% variables: 
% data  : <<matrix>> Time-frequency data. F-by-T-by-I matrix, where F, T, 
%         I are number of frequency bins (=length(TFinfo.freq), time
%         window length (=length(TFinfo.Tmsec) and the number of vertices
%         (=length(TFinfo.ix_act)), respectively.
% TFinfo: <<struct>> Information of time-frequency data. 
% --- fields of TFinfo
%  tf_type   : <<string>> Set to 'ERSP'.
%  Tsample   : <<vector>> Timepoint indices of the original data. 
%  Pretrigger: <<int>> Pretrigger period (timepoints). 
%  SampleFreq: <<float>> Sampling frequency of the original data. 
%  Tmsec     : <<vector>> Actual time of each timepoints in msec
%              (length(Tsample) = length(Tmsec)). 
%  Ntrial    : <<vector>> Number of trials of each current data. 
%  freq      : <<vector>> Frequency vector [Hz]. Each row of
%              'data(:,f,:)' has frequency TFinfo.freq(f). 
%  ix_act    : <<int vector>> Vertex indices on which time-frequency data
%              is calculated.
%  Wact      : <<double matrix>> Smoothing filter matrix. The size of
%              smoothing filter is automatically determined from mean
%              distance between vertices after reduction
%              (tf_parm.reduce). This matrix will be used for displaying
%              purpose. 
%  ix_act_ex : <<int vector>> Vertex indices corresponding to the row of
%              the smoothing matrix TFinfo.Wact. 
%  base      : <<double matrix>> Baseline at each frequency and
%              vertex. It is temporal average of the baseline ERSP. 
% ---
%
% [example]
% The following code is an example to calculate ERSP on template cortical
% surface model. 
% >> tf_parm.curr_file = './subjects/0001/result/0001_LR.curr.mat';
% >> tf_parm.curr_file_base = tf_parm.curr_file;
% >> tf_parm.brain_file = './subjects/0001/brain/0001.brain.mat';
% >> tf_parm.key = 'fsaverage.sphere.reg';
% >> tf_parm.tbrain_file ...
%     = './subjects/fsaverage/brain/fsaverage.brain.mat';
% >> tf_parm.reduce = 0.2;
% >> tf_parm.fmin = 2.0;
% >> tf_parm.baseline = [-300 0];
% >> tf_parm.tf_file_out ...
%     = './subjects/fsaverage/result/fsaverage_LR_ERSP.tf.mat';
% >> vb_job_ersp(proj_root,tf_parm);
%
% [history]
% 2011-01-02 Taku Yoshioka
% 2011-01-25 taku-y
%  [enhancement] EEGLAB timefreq function supported. 
% 2011-02-03 taku-y
%  [trivial] Baseline ERSP 'base' added to TFinfo. 
% 2011-02-28 taku-y
%  [debug] [0 fmax] -> [fmin fmax] in timefreq().
% 2011-07-01 taku-y
%  [minor] vb_disp() was replaced by vb_disp_nonl() for displaying
%  progress messages. 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

vb_disp('--- ERSP calculation');

%
% Verbose level
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
const = vb_define_verbose;
vb_struct2vars(const,{'VERBOSE_LEVEL_NONE','VERBOSE_LEVEL_EMERGENCY', ...
                    'VERBOSE_LEVEL_WARNING','VERBOSE_LEVEL_NOTICE', ...
                    'VERBOSE_LEVEL_INFO','VERBOSE_LEVEL_DEBUG'});
VERBOSE_LEVEL_DEBUG = const.VERBOSE_LEVEL_DEBUG;

%
% Input arguments
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_struct2vars(tf_parm,{'curr_file','brain_file','key', ...
                    'tf_file_out','tbrain_file','reduce','fmin','fmax', ...
                    'baseline','elim_time','curr_file_base', ...
                    'timefreq_parm'});

if isempty(curr_file_base), curr_file_base = curr_file; end
if isempty(elim_time), elim_time = 0.05; end

%
% Check arguments
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Sampling frequency
Jinfo1 = vb_load_current_info(curr_file);
Jinfo2 = vb_load_current_info(curr_file_base);
if Jinfo1.SampleFreq~=Jinfo2.SampleFreq, 
  error(['Sampling frequencies of current file and baseline ' ...
         'current file must be the same.']);
end

%
% Preparation of cortical dipole indices
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Number of dipoles on which TF analysis conducted. 
if ~isempty(key), % use template brain
  % note: consistency check between tbrain_file and key{1} is required,
  % but postponed. (7 Dec. 2010, taku-y)
  brain_file_tmp = tbrain_file;
else % single subject analysis
  brain_file_tmp = brain_file;
end

% Reduce vertex of the cortical model
V = vb_load_cortex(brain_file_tmp);
ix_act = vb_reduce_cortex(brain_file_tmp,1:size(V,1),reduce);
I = length(ix_act);
clear V;

% For single subject analysis, find overlap between ix_act and
% Jinfo.ix_act_ex. ix_act is replaced by it and I is overwritten. 
if isempty(key), % single subject analysis
  Jinfo = vb_load_current_info(curr_file);
  [tmp,ixx] = intersect(Jinfo.ix_act_ex,ix_act);
  ix_act = Jinfo.ix_act_ex(ixx);
  I = length(ix_act);
  Wact = Jinfo.Wact(ixx,:);
end

% Calculate mean distance between reduced vertices to determine smoothing
% filter size
R = 0.0;
N = 0;
[nextDD,nextIX] = vb_load_cortex_neighbour(brain_file_tmp);
for i=1:length(ix_act)
  [tmp,ixx] = union(nextIX{ix_act(i)},ix_act);
  R = R+sum(nextDD{ix_act(i)}(ixx));
  N = N+length(ixx);
end
R = R/N;

% Calculate smoothing filter. It will be used for displaying purpose, not
% affect further analysis.
V = vb_load_cortex(brain_file_tmp);
[W,ix_act_ex] ...
    = vb_spatial_gauss_filter(brain_file_tmp,R,2*R,ix_act,1,-1,1);
W = W./repmat(sum(W,2),[1 size(W,2)]);
TFinfo.Wact = W;
TFinfo.ix_act_ex = ix_act_ex;
clear V;

%
% Preparation of variables time-frequency data stored
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if isempty(timefreq_parm), % wavelet() parameters
  % Number of timepoints
  Jinfo = vb_load_current_info(curr_file);
  T  = length(Jinfo.Tmsec);    % time points
  DJ = 0.25;                   % scale of time points spacing
  DT = 1/(Jinfo.SampleFreq);   % original time points spacing
  S0 = 2*DT;                   % smallest freq. scale

  % Number of frequency bins
  [tmp,period,coi] = wavelet(randn(1,T),DT,1,DJ);
  TFinfo.freq      = 1./period;
  ix_freq          = find(TFinfo.freq>=fmin);
  TFinfo.freq      = TFinfo.freq(ix_freq);
  F                = length(TFinfo.freq);

  % Eliminate start and end time points
  t_elim  = ceil(T*elim_time);
  ix_time = (t_elim+1):(T-t_elim);
  Tmsec   = Jinfo.Tmsec(ix_time);
  T       = length(Tmsec);

  % Whole data
  data       = zeros(F,T,I);

  % Baseline
  if ~isempty(baseline), 
    Jinfo_base      = vb_load_current_info(curr_file_base);
    Tbase           = length(Jinfo_base.Tmsec);
    t_elim_base     = ceil(Tbase*elim_time);
    ix_time_base    = (t_elim_base+1):(Tbase-t_elim_base);
    Tmsec_base      = Jinfo_base.Tmsec(ix_time_base);
    Tbase           = length(Tmsec_base);
    base            = zeros(F,Tbase,I);
    Ntrial_base     = Jinfo_base.Ntrial;
  else
    Tbase           = [];
  end
else % timefreq() parameters
  vb_struct2vars(timefreq_parm,{'cycles','winsize','padratio', ...
                      'ntimesout','nfreqs','ntimesout_base'});
  
  % Number of timepoints and frequency bins
  Jinfo  = vb_load_current_info(curr_file);
  frames = length(Jinfo.Tmsec);
  tlimits = [min(Jinfo.Tmsec) max(Jinfo.Tmsec)];
  srate   = Jinfo.SampleFreq;
  if isempty(fmax), 
    fmax = srate/2;
  end
  
  [tmp,freqs,Tmsec] ...
      = timefreq(randn(frames,1),srate,'cycles',cycles, ...
                 'freqs',[fmin fmax],'padratio',padratio, ...
                 'winsize',winsize,'tlimits',tlimits, ...
                 'ntimesout',ntimesout,'nfreqs',nfreqs, ...
                 'verbose','off');
  
  T                = length(Tmsec);
  ix_time          = 1:length(Tmsec);
  TFinfo.freq      = freqs;
  ix_freq          = find(TFinfo.freq>=fmin);
  TFinfo.freq      = TFinfo.freq(ix_freq);
  F                = length(TFinfo.freq);
  
  % Whole data
  data = zeros(F,T,I);
  
  % Baseline
  if ~isempty(baseline),
    Jinfo_base   = vb_load_current_info(curr_file_base);
    frames_base  = length(Jinfo_base.Tmsec);
    tlimits_base = [min(Jinfo.Tmsec) max(Jinfo.Tmsec)];
    
    [tmp,freqs_base,Tmsec_base] ...
      = timefreq(randn(frames_base,1),srate,'cycles',cycles, ...
                 'freqs',[fmin fmax],'padratio',padratio, ...
                 'winsize',winsize,'tlimits',tlimits_base, ...
                 'ntimesout',ntimesout_base,'nfreqs',nfreqs, ...
                 'verbose','off');
    
    Tbase           = length(Tmsec_base);
    ix_time_base    = 1:Tbase;
    base            = zeros(F,Tbase,I);
    Ntrial_base     = Jinfo_base.Ntrial;
  else
    Tbase           = [];
  end
end

vb_disp(['Number of frequency bins       : ' num2str(F)]);
vb_disp(['Number of timepoints           : ' num2str(T)]);
vb_disp(['Number of vertices             : ' num2str(I)]);

if ~isempty(Tbase),
  vb_disp(['Number of timepoints (baseline): ' num2str(Tbase)]);
end

%
% Main routine
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Jinfo      = vb_load_current_info(curr_file);
Ntrial     = Jinfo.Ntrial;

if ~isempty(key), % use template brain
  C = vb_load_cortex_pmat(brain_file,key);

  if strcmp(Jinfo.curr_type,'Z-current'), 
    CW = sparse(C(ix_act,Jinfo.ix_act_ex)*Jinfo.Wact);
  else
    CW = sparse(C(ix_act,Jinfo.ix_act_ex));
  end
else % single subject analysis
  CW = Wact;
end

prg = 0;
prg_all = Ntrial;
prg_str = sprintf('Current : %d%% processed',0);
vb_disp_nonl(prg_str);

for j=1:Ntrial % trial loop
  [tmp,Z] = vb_load_current(curr_file,Jinfo.curr_type,false,j);
    
  for k=1:I % vertex loop
    ix = find(abs(CW(k,:))>0);
    J = CW(k,ix)*Z(ix,:); % J-current at ix_act-th dipole
    
    if isempty(timefreq_parm), 
      wave = wavelet(J,DT,1,DJ);
    else
      wave = timefreq(J',srate,'cycles',cycles, ...
                      'freqs',[fmin fmax],'padratio',padratio, ...
                      'winsize',winsize,'tlimits',tlimits, ...
                      'ntimesout',ntimesout,'nfreqs',nfreqs, ...
                      'verbose','off');
    end

    wave = wave(ix_freq,ix_time);
    data(:,:,k) = data(:,:,k)+abs(wave).^2;
  end
  
  % progress message
  prg = prg+1;
  for ii=1:length(prg_str); vb_disp_nonl(sprintf('\b')); end
  prg_str = sprintf('Current : %d%% processed',ceil(100*(prg/prg_all)));
  vb_disp_nonl(prg_str);
end

for ii=1:length(prg_str); vb_disp_nonl(sprintf('\b')); end
vb_disp('Current : finished');

% Baseline
Jinfo_base = vb_load_current_info(curr_file_base);
Ntrial     = Jinfo_base.Ntrial;

if ~isempty(key), % use template brain
  C = vb_load_cortex_pmat(brain_file,key);

  if strcmp(Jinfo_base.curr_type,'Z-current'), 
    CW = sparse(C(ix_act,Jinfo_base.ix_act_ex)*Jinfo_base.Wact);
  else
    CW = sparse(C(ix_act,Jinfo_base.ix_act_ex));
  end
else % single subject analysis
  CW = Wact;
end

prg = 0;
prg_all = Ntrial_base;
prg_str = sprintf('Baseline: %d%% processed',0);
vb_disp_nonl(prg_str);

for j=1:Ntrial_base % trial loop
  [tmp,Z] = vb_load_current(curr_file_base, ...
                            Jinfo_base.curr_type,false,j);

  for k=1:I % vertex loop
    ix = find(abs(CW(k,:))>0);
    J = CW(k,ix)*Z(ix,:); % J-current at ix_act-th dipole
    if isempty(timefreq_parm), 
      wave = wavelet(J,DT,1,DJ);
    else
      wave = timefreq(J',srate,'cycles',cycles, ...
                      'freqs',[fmin fmax],'padratio',padratio, ...
                      'winsize',winsize,'tlimits',tlimits_base, ...
                      'ntimesout',ntimesout_base,'nfreqs',nfreqs, ...
                      'verbose','off');
    end

    wave = wave(ix_freq,ix_time_base);        
    base(:,:,k) = base(:,:,k)+abs(wave).^2;
  end
  
  % progress message
  prg = prg+1;
  for ii=1:length(prg_str); vb_disp_nonl(sprintf('\b')); end
  prg_str = sprintf('Baseline: %d%% processed',ceil(100*(prg/prg_all)));
  vb_disp_nonl(prg_str);
end

for ii=1:length(prg_str); vb_disp_nonl(sprintf('\b')); end
vb_disp('Baseline: finished');

% Normalization
[tmp,ix_base_start] = min(abs(Tmsec_base-tf_parm.baseline(1)));
[tmp,ix_base_end]   = min(abs(Tmsec_base-tf_parm.baseline(2)));
ix_base = ix_base_start:ix_base_end;
data = data./Ntrial;
base = base./Ntrial_base;
data = 10*log10(data./repmat(mean(base(:,ix_base,:),2),[1 T 1]));

%
% Make struct 'TFinfo'
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Jinfo = vb_load_current_info(curr_file);

TFinfo.tf_type    = 'ERSP';
TFinfo.Tmsec      = Tmsec;
TFinfo.Ntrial     = Ntrial;
TFinfo.ix_act     = ix_act;
TFinfo.base       = mean(base(:,ix_base,:),2);

if isempty(timefreq_parm),
  TFinfo.Tsample    = Jinfo.Tsample(ix_time);
  TFinfo.Pretrigger = Jinfo.Pretrigger-t_elim;
  TFinfo.SampleFreq = Jinfo.SampleFreq;
  TFinfo.freq_scale = 'log'; % for wavelet()
else
  Tmsec = TFinfo.Tmsec;
  [tmp,TFinfo.Pretrigger] = min(abs(Tmsec(:)));
  TFinfo.SampleFreq = ((Tmsec(end)-Tmsec(1))./1000)./length(Tmsec);
  TFinfo.freq_scale = 'linear';
end

%
% Save result
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_fsave(tf_file_out,'tf_parm','data','TFinfo');

%
% Save project file
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
proj_file = get_project_filename;
if isempty(proj_file), 
  return;
end

project_file_mgr('load', proj_file);
project_file_mgr('add', 'tf_parm', tf_parm);

return;
