function vb_job_brain_add_mni_tal_coord(proj_root, brain_parm, disp_flag)
% Add MNI and Talairach coordinate value to Cortical model file.
% [USAGE]
%    vb_job_brain_add_mni_tal_coord(proj_root, brain_parm[,disp_flag]);
% [IN]
%     proj_root : VBMEG project root directory.
%    brain_parm : <<struct>> parameters for add.
%            .brain_file   : Cortical surface model file (.brain.mat)
%                            relative path from proj_root.
%                            MNI and Talairach information will be saved.
%
%            .spm_normalization_file : SPM normalization file (_sn.mat)
%
%            .FS_right_sphere_file : FreeSurfer sphere file (.sphere.reg.asc)
%                                    corresponding to the right hemisphere of
%                                    the source model.
%            .FS_left_sphere_file  : FreeSurfer sphere file (.sphere.reg.asc)
%                                    corresponding to the left hemisphere of
%                                    the source model.
%            .FS_sphere_key        : <<optional>> default: 'sphere.reg'
%                                    A registration key to Cortical model file.
%                                    Sphere coordinate value can be loaded as below.
%                                    vb_load_cortex(brain_file, FS_sphere_key);
%    disp_flag : =true  : plot result. [default]
%                =false : Do not plot result.
%
% case1: If you want to use SPM normalization file to transform coordinate system,
%        please specify .spm_normalization_file.
% case2: If you want to use FreeSurfer sphere file to transform coordinate system,
%        please specify .FS_right_sphere_file. and .FS_right_sphere_file.
%
% [OUT]
%    none
%
% [example]
%  <<spm case>>
%    proj_root               = '/home/foo/brain';
%    brain_parm.brain_file   = 'foo.brain.mat'; % target: /home/foo/brain/foo.brain.mat
%    brain_parm.spm_normalization_file = '/home/foo/foo_sn.mat';
%    vb_job_brain_add_mni_tal_coord(proj_root, brain_parm);
%
%  <<FS sphere file case>>
%    proj_root               = '/home/foo/brain';
%    brain_parm.brain_file   = 'foo.brain.mat'; % target: /home/foo/brain/foo.brain.mat
%    brain_parm.FS_right_sphere_file = '/home/foo/FS/bem/rh.sphere.reg.asc';
%    brain_parm.FS_left_sphere_file  = '/home/foo/FS/bem/lh.sphere.reg.asc';
%    brain_parm.FS_sphere_key = 'sphere.reg';
%    vb_job_brain_add_mni_tal_coord(proj_root, brain_parm);
%
% [HISTORY]
%    2013-01-04 rhayashi Initial version.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('proj_root', 'var')
    error('proj_root is a required parameter.');
end
if ~exist('brain_parm', 'var')
    error('brain_parm is a required parameter.');
end
if ~exist('disp_flag', 'var')
    disp_flag = true;
end

% field check
case_num = 0;
if isfield(brain_parm, 'spm_normalization_file') && ~isempty(brain_parm.spm_normalization_file)
    vb_disp('Extract MNI coordinate value using SPM normalise result.');
    case_num = 1;
elseif isfield(brain_parm, 'FS_right_sphere_file') && isfield(brain_parm, 'FS_left_sphere_file')
    vb_disp('Extract MNI coordinate value using FreeSurfer sphere registration file.');
    case_num = 2;
else
    error('Please specify appropriate field(s).');
end

%
% --- Main Procedure
%
proj_root = vb_rm_trailing_slash(proj_root);

switch(case_num)
case 1
    inner_spm_case(proj_root, brain_parm);
case 2
    inner_FS_case(proj_root, brain_parm);
otherwise
    error('Unknown method was specified.');
end
if disp_flag
    vb_check_mni_coregistration(fullfile(proj_root, brain_parm.brain_file));
end

function inner_spm_case(proj_root, brain_parm)
% Calculate MNI and Talairach coordinate from SPM normalization file
% and save them into Cortical surface model file(.brain.mat).
% [USAGE]
%    inner_spm_case(proj_root, brain_parm);
% [IN]
%     proj_root : VBMEG project root directory.
%    brain_parm : <<struct>> parameters for add.
%            .brain_file   : Cortical surface model file (.brain.mat)
%                            relative path from proj_root.
%                            MNI and Talairach information will be saved.
%
%            .spm_normalization_file : SPM normalization file (_sn.mat)
%
const = vb_define_verbose;
VERBOSE_LEVEL_NOTICE = const.VERBOSE_LEVEL_NOTICE;
global vbmeg_inst;

%% MNI coordinate
tic 
  vb_disp(sprintf('-- Computing MNI and Talairach coordinate '), ...
          VERBOSE_LEVEL_NOTICE);

  brain_file = fullfile(proj_root, brain_parm.brain_file);
  brain_file = strrep(brain_file, '\', '/');

  mniparm.mask_file  = vbmeg_inst.const.FILE_MNI_BETMASK_IMG; % fixed
  mniparm.brainfile  = brain_file;
  mniparm.snfile     = brain_parm.spm_normalization_file;

  [Vmni, Vtal, derr, IMGinfo, pXYZmni0] = vb_calc_mni_coord(mniparm);

  vb_save(brain_file, 'Vmni','Vtal','derr');
  vb_disp_nonl(sprintf('Done...%f[sec]\n\n',toc), VERBOSE_LEVEL_NOTICE);

function inner_FS_case(proj_root, brain_parm)
% Calculate MNI and Talairach coordinate from FS spherefile
% and save them into Cortical surface model file(.brain.mat).
% [USAGE]
%    inner_FS_case(proj_root, brain_parm);
% [IN]
%     proj_root : VBMEG project root directory.
%    brain_parm : <<struct>> parameters for add.
%            .brain_file   : Cortical surface model file (.brain.mat)
%                            relative path from proj_root.
%                            MNI and Talairach information will be saved.
%            .FS_right_sphere_file : FreeSurfer sphere file (.sphere.reg.asc)
%                                    corresponding to the right hemisphere of
%                                    the source model.
%            .FS_left_sphere_file  : FreeSurfer sphere file (.sphere.reg.asc)
%                                    corresponding to the left hemisphere of
%                                    the source model.
%            .FS_sphere_key        : A registration key to Cortical model file.
%                                    Sphere coordinate value can be loaded as below.
%                                    vb_load_cortex(brain_file, FS_sphere_key);
%
const = vb_define_verbose;
VERBOSE_LEVEL_NOTICE = const.VERBOSE_LEVEL_NOTICE;
tic
  vb_disp(sprintf('-- Computing MNI and Talairach coordinate '), ...
          VERBOSE_LEVEL_NOTICE);

  brain_file = fullfile(proj_root, brain_parm.brain_file);
  brain_file = strrep(brain_file, '\', '/');

  mniparm.brain_file = brain_file;
  mniparm.FS_right_sphere_file = brain_parm.FS_right_sphere_file;
  mniparm.FS_left_sphere_file  = brain_parm.FS_left_sphere_file;
  if ~isfield(mniparm, 'FS_sphere_key') || isempty(mniparm.FS_sphere_key)
      mniparm.FS_sphere_key = 'sphere.reg';
  end
  [Vmni, Vtal] = vb_calc_mni_coord_fs(mniparm);

  vb_save(brain_file, 'Vmni','Vtal');
  vb_disp_nonl(sprintf('Done...%f[sec]\n\n',toc), VERBOSE_LEVEL_NOTICE);
