function [property_list] = job_plot_cmap_setting_dlg(properties, mode)
% [USAGE]
%  property_list = job_plot_cmap_setting_dlg(properties, mode);
% 
% [IN]
%   properties : dialog initial properties <<struct array>> [Nx1]
%   properties(k).key : property name [string]
%                    = 'brain_file'        : Set cortical model file.
%                    = 'area_file'         : Set cortical area file.
%                    = 'act_file'          : Set cortical act file.
%                    = 'current_file'      : Set cortical current file.
%                    = 'current_load_mode' : Set load mode.
%                    = 'current_trial'     : Set trial number.
%                    = 'tf_file'           : Set time frequency file.
%                .value :  property value
%                    when key = '*_file'            : fullpath filename. [string]
%                    when key = 'current_load_mode' : 'ave' or 'raw'     [string]
%                    when key = 'current_trial'     : trial number       [integer]
%                .update_flag  : update checkbox status  [boolean]
%                                  =  true : check.
%                                  = false : uncheck.
%
%   mode : determines required file(s) on this dialog
%          <<integer>> : optional [default=1]
%          set *SUMMATION* value of bits by *DECIMAL* number.
%          = 1 (bit:001) : current/timefrequency file is required. 
%          = 2 (bit:010) : area file is required.
%          = 4 (bit:100) : act file is required.
%          e.g.
%          = 7 (bit:111) : (current/timefrequency + area + act)file is required.
%          - You can set "OR value of bits" by decimal notation.
%          * note: brain_file is always required.
%
% [OUT]
%  property_list : dialog input values.     <<struct array>> [Mx1] 
%                  The meaning of key, value, update_flags are same as input spec.
%  property_list(k).key         : property name  [string]
%                  .value       : property value [string]
%                  .update_flag : update flag    [boolean]
%                                   =  true : want to update.
%                                   = false : don't want to update.
%
%  GUI is closed by CANCEL or x button, property_list = [].
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('properties', 'var')
    error('properties is a required parameter.');
end
if ~exist('mode', 'var')
    mode = 1; % current/timefrequency display mode.
end

%
% --- Main Procedure
%
fig = openfig(mfilename);
H = guihandles(fig);
property_list = [];


% set default to current-file
job_plot_cmap_setting_dlg_callback(H,...
                                   H.data_file_current_radiobutton);

% set description text
desc = vb_define_description_table;
setappdata(H.brain_file_text, 'description', ...
           vb_table_query_value(desc, 'brain_file'));
setappdata(H.area_file_text, 'description', ...
           vb_table_query_value(desc, 'area_file'));
setappdata(H.act_file_text, 'description', ...
           vb_table_query_value(desc, 'act_file'));
setappdata(H.current_file_text, 'description', ...
           vb_table_query_value(desc, 'current_file'));
setappdata(H.load_mode_text, 'description', ...
           vb_table_query_value(desc, 'current_load_mode'));
setappdata(H.tf_file_text, 'description', ...
           vb_table_query_value(desc, 'tf_file'));

% Check file read status of callee application
file_on_app = 0;     % file read status on application which calls this dialog.
                     % set value by bitset.(bit number)
                     %  bit1:brain_file 
                     %  bit2:area_file
                     %  bit3:act_file
                     %  bit4:current_file
                     %  bit5:time frequency file

%
% --- set given properties to GUI
%
keys = vb_table_get_keys(properties);
for k=1:length(keys)
    prop = keys{k};
    val  = vb_table_query_value(properties, keys{k});
    flag = vb_table_query_update_flag(properties, keys{k});

    switch(prop)
        case 'brain_file'
            str = get(H.brain_file_text, 'String');
            set(H.brain_file_text, 'String', ['*', str]);
            job_plot_cmap_setting_dlg_set_file(H, val, 'brain_file');
            set(H.brain_file_update_checkbox, 'Value', flag);
            if exist(val, 'file') == 2 && flag == 0
                file_on_app = bitset(file_on_app, 1);
            end
        case 'area_file'
            if bitget(mode, 2)
                str = get(H.area_file_text, 'String');
                set(H.area_file_text, 'String', ['*', str]);
            end
            job_plot_cmap_setting_dlg_set_file(H, val, 'area_file');
            set(H.area_file_update_checkbox, 'Value', flag);
            if exist(val, 'file') == 2 && flag == 0
                file_on_app = bitset(file_on_app, 2);
            end
        case 'act_file'
            if bitget(mode, 3)
                str = get(H.act_file_text, 'String');
                set(H.act_file_text, 'String', ['*', str]);
            end
            job_plot_cmap_setting_dlg_set_file(H, val, 'act_file');
            set(H.act_file_update_checkbox, 'Value', flag);
            if exist(val, 'file') == 2 && flag == 0
                file_on_app = bitset(file_on_app, 3);
            end
        case 'current_file'
            if bitget(mode, 1)
                str = get(H.current_file_text, 'String');
                set(H.current_file_text, 'String', ['*', str]);
            end
            job_plot_cmap_setting_dlg_callback(H,...
                                    H.data_file_current_radiobutton);
            set(H.current_file_edit, 'String', val);

            % Current mode( 'ave' or 'raw' )
            cur_mode  = vb_table_query_value(properties, 'current_load_mode');
            if isempty(cur_mode)
                cur_mode = 'ave';
            end
            % Current trial number
            cur_trial = vb_table_query_value(properties, 'current_trial');
            if isempty(cur_trial)
                cur_trial = 1;
            end
            job_plot_cmap_setting_dlg_set_current_file(H, val, cur_mode, cur_trial);

            % Upate state
            set(H.current_file_update_checkbox, 'Value', flag);
            
            if exist(val, 'file') == 2 && flag == 0
                file_on_app = bitset(file_on_app, 4);
            end
        case 'current_load_mode'
            % This fields are used at current_file.
        case 'current_trial'
            % This fields are used at current_file.
        case 'tf_file'
            if bitget(mode, 1)
                str = get(H.tf_file_text, 'String');
                set(H.tf_file_text, 'String', ['*', str]);
            end
            if ~isempty(val)
                job_plot_cmap_setting_dlg_callback(H,...
                                    H.data_file_tf_radiobutton);
            end
            job_plot_cmap_setting_dlg_set_file(H, val, 'tf_file');
            set(H.tf_file_update_checkbox, 'Value', flag);

            if exist(val, 'file') == 2 && flag == 0
                file_on_app = bitset(file_on_app, 5);
            end
        otherwise
            delete(fig);
            error('unknown property : %s', prop);
    end
end

% Private data for this dialog.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
private_data = struct;
private_data.mode        = mode;
private_data.file_on_app = file_on_app;
private_data.input_properties = properties;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
guidata(fig, private_data);
ret = job_plot_cmap_setting_dlg_is_requirement_filled(fig);
if ret
    set(H.ok_push, 'Enable', 'on');
else
    set(H.ok_push, 'Enable', 'off');
end

%
% --- Wait for user input
%
set(fig, 'WindowStyle', 'modal');
waitfor(fig, 'Visible', 'off');

if ~ishandle(fig)
    % GUI is closed by cancel/close push
    % disp('cancel/close push');
    return;
end

%
% --- Create return value
%
if ishandle(fig)
    props = {'brain_file', 'area_file', 'act_file', 'current_file', 'tf_file'};
    % get value from figure
    for k=1:length(props)
        % get(H.brain_file_edit, 'String')
        cmd_get_file        = ['get(H.' props{k}, '_edit, ''String'');'];
        % get(H.brain_file_update_checkbox, 'Value')
        cmd_get_update      = ['get(H.', props{k}, '_update_checkbox, ''Value'');'];

        key         = props{k};
        value       = eval(cmd_get_file);
        update_flag = eval(cmd_get_update);
        
        record = vb_key_value_pair(key, value);
        record.update_flag = update_flag;
        
        property_list = vb_table_insert(property_list, record);
    end
    if get(H.data_file_current_radiobutton, 'Value')
        % 'current file', tf_file is removed.
        property_list = vb_table_delete(property_list, 'tf_file');
        % 'current_load_mode'
        if(get(H.current_file_mode_average_radiobutton, 'Value'))
            load_mode = 'ave';
        else
            load_mode = 'raw';
        end
        key   = 'current_load_mode';
        value = load_mode;
        update_flag = false;

        record = vb_key_value_pair(key, value);
        record.update_flag = update_flag;
        property_list = vb_table_insert(property_list, record);
        % 'current_trial'
        key = 'current_trial';
        value = 1;
        update_flag = false;
        if strcmp(load_mode, 'raw')
            value = str2double(...
                get(H.current_file_mode_raw_edit, 'String'));
        end
        record = vb_key_value_pair(key, value);
        record.update_flag = update_flag;
        property_list = vb_table_insert(property_list, record);
    else
        % 'tf_file', current_file is removed.
        property_list = vb_table_delete(property_list, 'current_file');
    end
    delete(fig);
end
