function job_edit_area(brainfile, areafile)
%---
% function job_edit_area(brainfile, areafile)
%
% Edit area data. 
%
% Input parameters
% - brainfile  : Cortex model file (.brain.mat)
% - areafile   : Area file (.area.mat)
%
% Example
% job_edit_area('./data/10000fb.brain.mat',...
%               './data/10000fb.area.mat'); 
%
% 2006-02-24 Taku Yoshioka
%---
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

inst_id = create_instance;
if isempty(inst_id), return; end; 
create_GUI(inst_id,areafile,brainfile);
job_edit_area_update_spatialpattern(inst_id);

%%%
%%% Inner functions 
%%%

%
% Create instance of GUI
% i: Instance ID of job_edit_area
%
function inst_id = create_instance

global vbmeg_inst

if isfield(vbmeg_inst,'editarea'),
  for inst_id=1:vbmeg_inst.const.EDITAREA_MAX
    if isempty(vbmeg_inst.editarea{inst_id}), return; end;
  end
  fprintf(['The number of instances of ''job_edit_area''' ...
	   ' GUIs reaches the maximum number.\n']);
  inst_id = [];
elseif isfield(vbmeg_inst,'const'), 
  vbmeg_inst.editarea = cell(vbmeg_inst.const.EDITAREA_MAX,1);
  inst_id = 1;
else
  fprintf(['''vbmeg_inst'' was not correct. You have to invoke' ...
	   '''vbmeg'' to create an instance of VBMEG.\n']); 
  inst_id = []; 
end

%
% Create GUI
%
function create_GUI(inst_id,areafile,brainfile)

global vbmeg_inst;

% GUI open
h_fig = open('job_edit_area.fig');
set(h_fig,'HandleVisibility','on');
drawnow; 
set(h_fig,'Pointer','watch');

% Load folded and inflated cortical models 
[V,F] = vb_load_cortex(brainfile); 
editarea.V0 = V; 
editarea.F0 = F;
[V,F,xx,inf_C] = vb_load_cortex(brainfile,'Inflate');
editarea.V = V;
editarea.F = F;
editarea.inf_C = inf_C;

% Load neighbour data for morphological operations
[nextDD,nextIX] = vb_load_cortex_neighbour(brainfile); 
editarea.nextDD = nextDD;
editarea.nextIX = nextIX;

% Area file
editarea.areafile = areafile;

% Area data
editarea.Iextract = cell(30,1); 

% Handles of GUI components
editarea.h_fig = h_fig;
editarea.h_space = get_child_handle(h_fig,'plot_brain');
editarea.ed1 = get_child_handle(h_fig,'dilation_radius');
editarea.ed2 = get_child_handle(h_fig,'erosion_radius');
editarea.ed3 = get_child_handle(h_fig,'vertex_number');
editarea.ed4 = get_child_handle(h_fig,'area_file');
editarea.lb1 = get_child_handle(h_fig,'area_list');
editarea.pb1 = get_child_handle(h_fig,'dilation');
editarea.pb2 = get_child_handle(h_fig,'erosion');
editarea.pb4 = get_child_handle(h_fig,'clear');
editarea.pb5 = get_child_handle(h_fig,'delete');
editarea.pb6 = get_child_handle(h_fig,'overwrite');
editarea.pb7 = get_child_handle(h_fig,'registration');
editarea.pb8 = get_child_handle(h_fig,'undo');
editarea.pb9 = get_child_handle(h_fig,'reverse');
editarea.pb10 = get_child_handle(h_fig,'remove_corpus');
editarea.zt  = get_child_handle(h_fig, 'zoom_toggle');
editarea.operation_text = get_child_handle(h_fig, 'operation_text');

% Set GUI value
set(editarea.ed1,'String','6');
set(editarea.ed2,'String','6');
set(editarea.ed3,'String','0');
set(editarea.ed4,'String',areafile);

% Area list
keyset = vb_get_keyset_area(areafile);
set(editarea.lb1,'String',keyset); 
set(editarea.lb1,'Value',1);

% Cortex plot parameters
editarea.plot_parm = vb_set_plot_parm;
editarea.plot_parm.index = [];
editarea.plot_parm.LRflag = 'LR';

%
% Set callback functions
%
inst_str = num2str(inst_id); 
editarea_str = ['vbmeg_inst.editarea{' inst_str '}'];

% Change filter radius
command = ['job_edit_area_change_radius(' inst_str ');']; 
set(editarea.ed1,'Callback',command);
set(editarea.ed2,'Callback',command);

% Dilation
command = ['job_edit_area_dilation(' inst_str ');'];
set(editarea.pb1,'Callback',command);

% Erosion
command = ['job_edit_area_erosion(' inst_str ');'];
set(editarea.pb2,'Callback',command);

% Clear current area
command = ['job_edit_area_clear(' inst_str ');'];
set(editarea.pb4,'Callback',command);

% Delete area
command = ['job_edit_area_delete(' inst_str ');'];
set(editarea.pb5,'Callback',command);

% Overwrite area
command = ['job_edit_area_overwrite(' inst_str ');'];
set(editarea.pb6,'Callback',command);

% Area registration
command = ['job_edit_area_registration(' inst_str ');'];
set(editarea.pb7,'Callback',command);

% Undo
command = ['job_edit_area_undo(' inst_str ');'];
set(editarea.pb8,'Callback',command);

% Reverse
command = ['job_edit_area_reverse(' inst_str ');'];
set(editarea.pb9,'Callback',command);

% Remove corpus
command = ['job_edit_area_remove_corpus(' inst_str ');'];
set(editarea.pb10,'Callback',command);

% Close window
command = ['global vbmeg_inst; '...
	   'vbmeg_inst.editarea{' inst_str '}=[];'];
set(h_fig,'DeleteFcn',command);

% Zoom on/off button
command = ['job_edit_area_change_zoom(' inst_str ');'];
set(editarea.zt, 'Callback', command);
if vb_matlab_version('<=', '7.5.0')
    set(editarea.zt, 'Visible', 'on');
else
    str       = get(editarea.operation_text, 'String');
    zoom_help = '   Zoom: mouse wheel';
    set(editarea.operation_text, 'String', [str, zoom_help]);
    set(editarea.zt, 'Visible', 'off');
end

% 
% Properties of axis for cortex plot
set(h_fig,'Renderer','zbuffer');
set(h_fig,'CurrentAxes',editarea.h_space);
axis off;
%axis tight; % 2006/11/24 hayashi delete
%axis equal; % These commands causes axes LimMode change(auto-->manual)
             % after MATLAB 7.0.1. and brain model is not displayed. 

% Set instance
vbmeg_inst.editarea{inst_id} = editarea; 

% Set pointer to arrow
set(h_fig,'Pointer','arrow');

% not to show colorbar warning
warning('off', 'MATLAB:colorbar:DeprecatedV6Argument');
