function [varargout] = project_file_mgr(varargin)
% This function manages project file.
% Usage :
%  key : 'make', 'load', 'add', 'get_all_parameters', 'get_parameter',
%        'get_project_root', 'get_project_name', 'get_project_file'
%
%  project_file_mgr('make', <filename>, <project_name>, <project_root>);
%    projec file create. 
%
%  project_file_mgr('load', <filename>); 
%    load project file on memory.
%
%  project_file_mgr('add', <parm_name>, <parm>);
%    save parameter to file. you should 'load' project file before this operation.
%    e.g project_file_mgr('add', 'brain_parm', brain_parm);
%
%  project_file_mgr('get_all_parameters');
%    return [parm, parm_name]
%                parm : parameter list(Nx1)
%           parm_name : parameter name list(Nx1)
%    parameter order is sorted by timestamp(registered order.)
%
%  project_file_mgr('get_parameter', <itemNo>)
%    return [parm, parm_name]
%                parm : parameter
%           parm_name : parameter name
%    itemNo : registered order.
%
%  project_file_mgr('get_project_root');
%    return proj_root
%
%  project_file_mgr('get_project_name');
%    return project_name
%
%  project_file_mgr('get_project_file');
%    return project file name loaded on memory.
%
% [NOTE]
%  associate with project_mgr :
%   project_file_mgr notify to the project_mgr when the status(persistent value)
%   changed. And then, project_mgr receives this notify, project_mgr calls 
%   project_file_mgr to get updated parameters.
%
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)
mlock; 
persistent project_name;
persistent proj_root;
persistent project_file;

persistent all_parms;
persistent parm_types;

key = varargin{1};

switch(key)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% File operation function
%
case 'make'
    try
        project_file = varargin{2};
        project_name = varargin{3};
        proj_root = varargin{4};
        vb_fsave(project_file, 'project_name', 'proj_root');
        notify_status_change;
        varargout{1} = false;
    catch
        varargout{1} = true;
    end

case 'load'
    err = false;
    project_file = varargin{2};
    if exist(project_file, 'file')
        all_parms = [];
        parm_types = [];
        load(project_file, 'project_name', 'proj_root');
        contains = whos('-file', project_file, '*_parm');
        for k=1:length(contains)
            parm_name = contains(k).name;
            % *_parm : cell is loaded.
            tmp = load(project_file, parm_name);
            if ~iscell(tmp.(parm_name))
                parm = cell(1);
                parm{1} = tmp.parm_name;
            else
                parm = tmp.(parm_name);
            end
            all_parms  = [all_parms; parm(:)];
            parm_types = [parm_types; cellstr(repmat(parm_name, length(parm), 1))];
        end

        % remove empty line
        empty_ix = [];
        for k=1:length(parm_types)
            if isempty(parm_types{k})
                empty_ix = [empty_ix;k];
            end
        end
        if ~isempty(empty_ix)
            parm_types(empty_ix) = [];
        end

        % parameter sort by time_stamp
        time_stamps = [];
        for k=1:length(all_parms)
            if ~isfield(all_parms{k}, 'time_stamp')
                break;
            end
            time_stamps = [time_stamps; all_parms{k}.time_stamp];
        end
        if ~isempty(time_stamps)
            [time_stamps, ix] = sortrows(time_stamps);
            all_parms = all_parms(ix);
            parm_types = parm_types(ix);
        end

        notify_status_change;
        err = false;
    else
        err = true;
    end
    varargout{1} = err;

case 'add'
    if isempty(project_file) || ~exist(project_file, 'file')
       varargout{1} = false;
       return;
    end
    % caller_function is supposed to the name of job function.
    stack = dbstack;
    [a, caller_function] = vb_get_file_parts(stack(2).name);
    if nargin == 4
        caller_function = varargin{4};
    end
    parm_name = varargin{2};
    parm = varargin{3};
    
    % Add execution information
    parm.time_stamp      = datestr(fix(clock), 31);
    parm.caller_function = caller_function;
    
    Nparm = 0;
    tmp = whos('-file', project_file, parm_name);
    if length(tmp)
        load(project_file, parm_name);
        Nparm = length(eval(parm_name)); % Nparm = length(brain_parm)
    else
        command = [parm_name '=cell(0);'];
        eval(command);
    end
    command = [parm_name '{' num2str(Nparm+1) '} = parm;'];
    eval(command);
    
    vb_save(project_file, parm_name);
    project_file_mgr('load', project_file);
    notify_status_change;
    varargout{1} = true;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Parameters get function
%
case 'get_all_parameters'
    % return sorted data
    varargout{1} = all_parms;
    varargout{2} = parm_types;

case 'get_parameter'
    itemNum = varargin{2};
    [all_parms, parm_types] = project_file_mgr('get_all_parameters');
    if length(all_parms) >= itemNum
        varargout{1} = all_parms{itemNum};
        varargout{2} = parm_types{itemNum};
    else
        varargout{1} = [];
        varargout{2} = [];
    end

case 'get_project_root'
    varargout{1} = proj_root;
case 'get_project_name'
    varargout{1} = project_name;
case 'get_project_file'
    varargout{1} = project_file;
end

function notify_status_change
    current_hidden_state = get(0, 'ShowHiddenHandles');
    set(0, 'ShowHiddenHandles', 'on');
    fig = findobj('Tag', 'project_mgr');
    set(0, 'ShowHiddenHandles', current_hidden_state);
    if ~isempty(fig)
        project_mgr_view_update(fig);
    end
