function [fig] = vb_timeseries_viewer_new(meg_file, window_len, data_ch_list, ...
                                          trial_list, trig_ch_list, trigger_file)
% GUI for checking time series and trial data.
% [USAGE]
%    [fig] = vb_timeseries_viewer_new(org_file, window_len, ...
%                              data_ch_list, trial_list, trig_ch_list);
% [IN]
%      meg_file : MEG/EEG file(.meg.mat/.eeg.mat)
%    window_len : (optional) display window length [sec]
%  data_ch_list : (optional) data channel name list {Nx1}
%    trial_list : (optional) trial list [Nx1]
%                             N x <<struct>>
%                              .from : trial starts here [sec]
%                              .to   : trial ends here   [sec]
%                           If you speficy trigger_file, this is
%                           overwritten.
%  trig_ch_list : (optional)trigger channel name list {Nx1}
%                           If you speficy trigger_file, this is
%                           overwritten.
%  trigger_file : (optional)trigger filename(.trig.mat).
%
% [OUT]
%    fig : figure handle.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('meg_file', 'var')
    error('meg_file is a required parameter.');
end
if ~exist('window_len', 'var')
    window_len = 30;
end
if ~exist('data_ch_list', 'var')
    ch_info = vb_load_channel_info(meg_file);
    data_ch_list = ch_info.Name;
end
if ~exist('trial_list', 'var')
    trial_list = [];
end
if ~exist('trig_ch_list', 'var')
    trig_ch_list = [];
end
if ~exist('trigger_file' ,'var')
    trigger_file = [];
end
if exist(meg_file, 'file') ~= 2
    error('Specified MEG/EEG file not found.');
end
if ischar(trig_ch_list)
    trig_ch_list = {trig_ch_list};
end

%
% --- Main Procedure
%
obj = struct;

% Prepare for read MEG/EEG Data channel
obj.data_ch_list = data_ch_list;
obj.data_ch_list_plot_ix = [1:length(data_ch_list)]';
if exist(meg_file, 'file') ~= 2
    return;
end
obj.meg_obj = vb_continuous_file_new(meg_file, data_ch_list);

% Get measurement type('MEG' or 'EEG')
measurement = vb_continuous_file_get_measurement(obj.meg_obj);


% if trigger file specified, trial and trigger used for trial extraction.
if exist(trigger_file, 'file') == 2
    % Load trigger file
    [trial_list, trig_ch_list] = ...
         vb_timeseries_viewer_load_trigger_file(obj.meg_obj, trigger_file);
elseif ~isempty(trigger_file)
    error('Specified trigger file(.trig.mat) not found.');
end

% Prepare for read External channel
obj.trigger_obj = vb_continuous_file_new(meg_file, trig_ch_list);

% Trigger properties.
obj.trig_ch_list         = trig_ch_list;
obj.trig_ch_list_plot_ix = [1:length(trig_ch_list)]';
obj.trigger_file         = trigger_file;

% Open figure
obj.figure = openfig('vb_timeseries_viewer.fig', 'new', 'invisible');
obj.H      = guihandles(obj.figure);
%set(obj.figure, 'Visible', 'on');% debug

% Set trial information
obj = vb_timeseries_viewer_set_trial(obj, trial_list);

%
% --- Read and cache MEG/EEG data
%
H = obj.H;

if ~isempty(obj.meg_obj)
    Nsample = vb_continuous_file_get_Nsample(obj.meg_obj);
    freq    = vb_continuous_file_get_sample_freq(obj.meg_obj);
    % read and cache data
    [obj.meg_obj, data] = vb_continuous_file_get_sample(obj.meg_obj, ...
                                                          obj.data_ch_list, ...
                                                          1, Nsample);
    % Fix Ylimit
    if ~isempty(data)
        max_value = max(data(:));
        min_value = min(data(:));
        ylim(H.data_axes, [min_value, max_value]);
        set(H.ylimit_high_edit, 'String', num2str(max_value));
        set(H.ylimit_low_edit, 'String', num2str(min_value));
    end
    clear data;
end

%
% --- Read and cache Trigger data
%
if ~isempty(obj.trigger_obj)
    Nsample = vb_continuous_file_get_Nsample(obj.trigger_obj);
    freq    = vb_continuous_file_get_sample_freq(obj.trigger_obj);
    % read and cache data
    [obj.trigger_obj] = vb_continuous_file_get_sample(obj.trigger_obj, ...
                                                   obj.trig_ch_list, ...
                                                   1, Nsample);
end

obj.freq = freq;
obj.Nsample = Nsample;
obj.total_time_sec = (Nsample / freq);

%
% --- Scaling Slider
%
if window_len >= obj.total_time_sec;
    window_len = obj.total_time_sec;
end
obj.window_len = window_len;

max_value = get(H.Yscale_slider, 'Max');
set(H.Yscale_slider, 'Value', max_value-(obj.window_len / 100));

% Time point setup
axis(H.timepoint_axes, 'manual');
xlim(H.timepoint_axes, [0, Nsample/freq]);
set(H.timepoint_axes, 'YTickLabel', '');

h = xlabel(H.timepoint_axes, 'Display position[sec]', ...
                         'FontName', 'Times', ...
                         'FontSize', 14, 'FontUnit', 'points');
set(h, 'FontSize', 14); % ??? If it remove this line, font size doesn't set collectly.
set(h, 'FontUnit', 'normalized');

% update display
obj.event_number = 0;
obj = vb_timeseries_viewer_update_display_time_range(obj);

% save data
guidata(obj.figure, obj);

%
% --- Time slider
%

% slider continuous callback(using MATLAB hack)
try
tp_listener = handle.listener(H.timeseries_slider, ...
                              'ActionEvent', ...
                              @vb_timeseries_viewer_slider_event);
setappdata(H.timeseries_slider, 'mySliderListener', tp_listener);
catch
addlistener(H.timeseries_slider, 'Value', 'PostSet', @vb_timeseries_viewer_slider_event);
end

setappdata(H.timeseries_slider, 'Figure', obj.figure);

try
yscale_listener = handle.listener(H.Yscale_slider, ...
                                  'ActionEvent', ...
                                  @vb_timeseries_viewer_slider_event);
setappdata(H.Yscale_slider, 'mySliderListener', yscale_listener);
catch
addlistener(H.Yscale_slider, 'Value', 'PostSet', @vb_timeseries_viewer_slider_event);
end
setappdata(H.Yscale_slider, 'Figure', obj.figure);

%
% --- Axes settings
%

%%%%%%%%%%%%%%%%%%%%%
% Data axes settings
%%%%%%%%%%%%%%%%%%%%%
% h = xlabel(H.data_axes, 'Time[sec]', ...
%                         'FontName', 'Times', 'FontUnit', 'points', ...
%                         'FontSize', 14);
% set(h, 'FontUnit', 'normalized');

if strcmpi(measurement, 'MEG')
    h1 = title(H.data_axes, 'MEG');
    h2 = ylabel(H.data_axes, 'Magnetic field [T]', 'FontUnit', 'points', ...
                        'FontName', 'Times', 'FontSize', 14);
    h = [h1; h2];
elseif strcmpi(measurement, 'EEG')
    h1 = title(H.data_axes, 'EEG');
    h2 = ylabel(H.data_axes, 'Electric potential [V]', 'FontUnit', 'points', ...
                        'FontName', 'Times', 'FontSize', 14);
    h = [h1; h2];
end
set(h, 'FontUnit', 'normalized');

% Set Axes font
set(H.data_axes, 'FontUnit', 'points', 'FontName', 'Times', ...
                 'FontSize', 14);
set(H.data_axes, 'FontUnit', 'normalized');

if isempty(data_ch_list)
    set(H.data_channel_push, 'Enable', 'off');
end


%%%%%%%%%%%%%%%%%%%%%%%%%%
% External axes settings
%%%%%%%%%%%%%%%%%%%%%%%%%%
h = xlabel(H.trigger_axes, 'Time[sec]', ...
                        'FontName', 'Times', ...
                        'FontUnit', 'points', 'FontSize', 14);
set(h, 'FontUnit', 'normalized');
if isempty(trial_list)
    h = title(H.trigger_axes, 'External channel', ...
                       'FontName', 'Times', ...
                       'FontUnit', 'points', 'FontSize', 14);
else
    h = title(H.trigger_axes, 'Trigger channel', ...
                       'FontName', 'Times', ...
                       'FontUnit', 'points', 'FontSize', 14);
end
set(h, 'FontUnit', 'normalized');

% Set Axis font
set(H.trigger_axes, 'FontName', 'Times', ...
                    'FontUnit', 'points');
set(H.trigger_axes, 'FontSize', 14);
set(H.trigger_axes, 'FontUnit', 'normalized');

if isempty(trig_ch_list)
    set(H.trigger_channel_push, 'Enable', 'off');
end

% Zoom in/out button Icons
cdata_zoom_in  = load('zoomplus.mat');
cdata_zoom_out = load('zoomminus.mat');

set(H.scale_plus_push,  'Cdata', cdata_zoom_in.cdata);
set(H.scale_minus_push, 'Cdata', cdata_zoom_out.cdata);

%%%%%%%%%%%%%%%%%%%%%%%%%%
% Menu settings
%%%%%%%%%%%%%%%%%%%%%%%%%%

% File->Save trigger file
if ~isempty(obj.trigger_file)
    set(H.save_trigger_file, 'Enable', 'on');
else
    set(H.save_trigger_file, 'Enable', 'off');
end

%
% --- Final step
%
set(obj.figure, 'Name', meg_file);
set(obj.figure, 'Menubar', 'none');
set(obj.figure, 'Toolbar', 'none');
set(obj.figure, 'HandleVisibility', 'callback');
set(obj.figure, 'Visible', 'on');

fig = obj.figure;
