function varargout = bayes_parm_basic_gui(fig, command, varargin)
% Process event abouts bayes parameter basic gui.
% [IN]
%         fig : figure handle of bayes basic parameter gui.
%     command : event type
%    varargin :
% [OUT]
%    varargout :
%
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('fig', 'var') || isempty(fig) || ~ishandle(fig)
    error('invalid figure handle is specified.');
end

%
% --- Main Procedure
%

% load data(@see bayes_parm_basic_init)
data = guidata(fig);
H = data.H;

switch(command)
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % GUI initialize/exit/update functions
    %
    case 'is_exist'
        if isfield(H, 'fig')
            varargout{1} = true;
        else
            varargout{1} = false;
        end
    case 'figure_pull_up_to_top'
        figure(H.fig);
    case 'gui_update'
        bayes_parm = varargin{1};
        data = set_gui_parms(data, bayes_parm);
    case 'get_project_root'
        project_root = get(H.project_root_edit, 'String');
        varargout{1} = project_root;
    case 'set_project_root'
        set(H.project_root_edit, 'String', varargin{1});

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % GUI component handles getter functions
    %
    case 'get_gui_handles'
        varargout{1} = H;

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % GUI component's call back functions
    case 'call_back'
        hObj = varargin{1};
        data = callback(data, hObj);
        if ~ishandle(data.H.figure), return; end
        % gui parameters collection
        bayes_parm = get_gui_parms(data);
        % send parameters to main function
        bayes_parm_editor(data.main_obj, 'merge_bayes_parm', bayes_parm);
end

function data = set_gui_parms(data, bayes_parm)
% This function set parameters to the GUI.
% see : get_gui_parms
% [IN]
%          data : application data (@see bayes_parm_basic_init)
%    bayes_parm : initialize parameters.
% [OUT]
%    H : structure of all object handles

    H = data.H;

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % files
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    if isfield(bayes_parm, 'proj_dir') && isempty(get(H.project_root_edit, 'String'))
        set(H.project_root_edit, 'String', bayes_parm.proj_dir);
    end
    data = set_gui_meg_file(data, bayes_parm.megfile);
    set_gui_basis_file(H, bayes_parm.basisfile);
    set(H.brain_file_edit, 'String', bayes_parm.brainfile);
%    [output_dir, filename, ext] = fileparts(bayes_parm.bayesfile);
    [output_dir, filename, ext] = vb_get_file_parts(bayes_parm.bayesfile);
    set(H.result_file_outputdir_edit, 'String', output_dir);
    set(H.result_filename_edit, 'String', [filename, ext]);
    set_gui_act_file(H, bayes_parm.actfile);
    set_gui_area_file(H, bayes_parm.areafile);
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % keys
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    area_keys = get(H.area_key_listbox, 'String');
    if isempty(area_keys)
        set(H.area_key_listbox, 'String', bayes_parm.area_key);
        area_keys = bayes_parm.area_key;
    end
    idx = strmatch(bayes_parm.area_key, area_keys, 'exact');
    if ~isempty(idx)
        set(H.area_key_listbox, 'Value', idx);
    else
        set(H.area_key_listbox, 'Value', 1);
    end

    act_keys = get(H.act_key_listbox, 'String');
    if isempty(act_keys)
        set(H.act_key_listbox, 'String', bayes_parm.act_key);
        act_keys = bayes_parm.act_key;
    end
    idx = [];
    % convert char to cell
    if ischar(bayes_parm.act_key)
        bayes_parm.act_key = cellstr(bayes_parm.act_key);
    end
    for k=1:length(bayes_parm.act_key)
        idx = [idx; strmatch(bayes_parm.act_key{k}, act_keys, 'exact')];
    end
    if ~isempty(idx)
        set(H.act_key_listbox, 'Value', idx);
    else
        set(H.area_key_listbox, 'Value', 1);
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Time parameters
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %Whole time window
    if length(bayes_parm.twin_meg) == 2
        wtime_from_index = bayes_parm.twin_meg(1);
        wtime_to_index = bayes_parm.twin_meg(2);
    
        wtime_from_msec = bayes_parm_editor(data.main_obj, ...
                                            'index_to_time',...
                                            wtime_from_index,...
                                            'meg');
        wtime_to_msec = bayes_parm_editor(data.main_obj, ...
                                            'index_to_time',...
                                            wtime_to_index,...
                                            'meg');
        set(H.whole_time_window_from_edit,...
            'String', sprintf('%.2f', wtime_from_msec));
        set(H.whole_time_window_to_edit,...
            'String', sprintf('%.2f', wtime_to_msec));
    end
    %
    % --- Window parameters
    %
    set(H.time_shift_start_pos_edit, 'String', '');
    set(H.time_shift_end_pos_edit, 'String', '');
    set(H.time_shift_window_size_edit, 'String', '');
    set(H.time_shift_shift_length_edit, 'String', '');

    if (length(bayes_parm.Tperiod) == 1) &&...
       (length(bayes_parm.Tnext) == 1)
        change_time_shift_mode(H, 'equally');
        %Time shift equally
        window_size = bayes_parm_editor(data.main_obj, ...
                                        'data_len_to_time_len',...
                                        bayes_parm.Tperiod,...
                                        'meg');%msec
        shift_len = bayes_parm_editor(data.main_obj, ...
                                        'data_len_to_time_len',...
                                        bayes_parm.Tnext,...
                                        'meg');%msec
        set(H.time_shift_window_size_edit,...
            'String', sprintf('%.2f', window_size));
        set(H.time_shift_shift_length_edit,...
            'String', sprintf('%.2f', shift_len));
    elseif (length(bayes_parm.Tperiod) ~= 0) &&...
            (length(bayes_parm.Tnext) ~= 0)
        %Manual settings        
        change_time_shift_mode(H, 'manual');
        start_idx = bayes_parm.Tnext + bayes_parm.twin_meg(1) -1;
        start_pos = bayes_parm_editor(data.main_obj, ...
                                      'index_to_time',...
                                      start_idx,...
                                      'meg');
        end_idx = bayes_parm.Tnext + bayes_parm.Tperiod + ...
                  bayes_parm.twin_meg(1)-2;
        end_pos = bayes_parm_editor(data.main_obj, ...
                                    'index_to_time',...
                                    end_idx,...
                                    'meg');
        spos_comma_text = bayes_parm_gui_util('comma_text', start_pos);
        epos_comma_text = bayes_parm_gui_util('comma_text', end_pos);
        set(H.time_shift_start_pos_edit, 'String', spos_comma_text);
        set(H.time_shift_end_pos_edit, 'String', epos_comma_text);
    else
        change_time_shift_mode(H, 'equally');
        set(H.time_shift_window_size_edit, 'String', '');
        set(H.time_shift_shift_length_edit,'String', '');
        set(H.time_shift_start_pos_edit, 'String', '');
        set(H.time_shift_end_pos_edit, 'String', '');
    end
    
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Estimation Model Parameters
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %Estimation model

    if IsTreatableEstimationModel(bayes_parm) == true
        %
        % -- Forward Model
        %
        set_forward_model(H, bayes_parm.forward_model, true);

        %
        % -- Noise estimation model
        %
        model = bayes_parm.noise_estimation_model;
%	    if(model == 1)||(model == 2)
%	        value = model;     % listbox line number
%	    elseif(model == 3)||(model == 4)||(model == 5)
%	        value = model - 2; % listbox line number
%	    else
%	        disp('bayes_parm_editor:set_gui_parms estimation_model error');
%	    end

        set(H.noise_estimation_model_popup, 'Value', model);
        set_estimation_GUI_properties(H);

        %
        % -- update_sx
        %
        %Covariance
        set(H.obsn_cov_popup, 'Value', bayes_parm.noise_model);

        %Update
        if bayes_parm.update_sx == ON
            update_state = 1;    
        else
            update_state = 0;    
        end
        set(H.update_sx_checkbox, 'Value', update_state);

        %
        % -- update_v
        %

        %Update
        if bayes_parm.update_v == ON
            update_state = 1;
        else
            update_state = 0;
        end
        set(H.update_v_checkbox, 'Value', update_state);

    else
        error_str = sprintf(['Cannot read Estimation Model Parameters.\n', ...
                            '(The combination not to be able to treat was specified.)\n', ...
                            'Default Estimation Model parameters are set.']);
        errordlg(error_str, 'notice');

        % Default estimation parameters are set by this.
	    set_forward_model(H, STR_FORWARD_MODEL_FOCAL, true);
    end

    %Sensor noise file and Noise Time window
    %set(H.sensor_noise_file_listbox, 'String', bayes_parm.megnoisefile);
    set(H.sensor_noise_file_listbox,'String',bayes_parm.megfile_baseline);
    data = set_noise_file_time_window(data, bayes_parm);

    % Smoothness filter radius
    set(H.Rfilt_edit, 'String', sprintf('%.2f', bayes_parm.Rfilt(1)*1e3));

    % Reduce ratio
    set(H.reduce_edit, 'String', sprintf('%.2f', bayes_parm.reduce(1)));
    
    % Magnification parameter (a0_act)
    set(H.magnification_edit, 'String', num2str(bayes_parm.a0_act));
    
    % Prior weight (prior_weight)
    set(H.prior_weight_edit, 'String', num2str(bayes_parm.prior_weight)); 
    
function bayes_parm = get_gui_parms(data)
% This function get basic bayes parameters from GUI.
% see : set_gui_parms
% [IN]
%    data : application data (@see bayes_parm_basic_init)
% [OUT]
%    bayes_parm : bayes basic parameters.

    H = data.H;
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % files
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%    bayes_parm.proj_dir = get(H.project_root_edit, 'String');

    bayes_parm.megfile = get(H.meg_file_listbox, 'String');
    bayes_parm.basisfile = get(H.basis_file_listbox, 'String');
    bayes_parm.brainfile = get(H.brain_file_edit, 'String');
    bayes_parm.bayesfile = [get(H.result_file_outputdir_edit, 'String'),...
                            filesep,...
                            get(H.result_filename_edit, 'String')];
    bayes_parm.actfile = get(H.act_file_edit, 'String');
    bayes_parm.areafile = get(H.area_file_edit, 'String');

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % keys
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    area_keys = get(H.area_key_listbox, 'String');
    if ischar(area_keys)
        bayes_parm.area_key = area_keys;
    else
        bayes_parm.area_key = area_keys{get(H.area_key_listbox, 'Value')};
    end
    act_keys = get(H.act_key_listbox, 'String');
    % bayes_parm.act_key is cell strings
    if ischar(act_keys)
        bayes_parm.act_key = {act_keys};
    else
        bayes_parm.act_key = act_keys(get(H.act_key_listbox, 'Value'));
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Time parameters
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %Duration settings
    Tperiod = [];
    Tnext = [];
    twin_meg = [];
    try
    if get(H.time_shift_equally_radiobutton, 'Value')
        %
        % --- Time shift Equally
        %
        
        % get parameters from gui
        wtime_from_msec = str2double(...
                            get(H.whole_time_window_from_edit, 'String'));
        wtime_to_msec = str2double(get(H.whole_time_window_to_edit, 'String'));
	
        % convert vbmeg data format
        wtime_from_index = bayes_parm_editor(data.main_obj, ...
                                             'time_to_index',...
                                             wtime_from_msec, 'meg');
        wtime_to_index   = bayes_parm_editor(data.main_obj, ...
                                             'time_to_index',...
                                             wtime_to_msec, 'meg');

        twin_meg = [wtime_from_index, wtime_to_index];
        
        window_size = str2double(...
                            get(H.time_shift_window_size_edit, 'String'));
        time_shift_len = str2double(...
                            get(H.time_shift_shift_length_edit, 'String'));

        Tperiod = bayes_parm_editor(data.main_obj, ...
                                    'time_len_to_data_len',...
                                    window_size, 'meg');
        Tnext = bayes_parm_editor(data.main_obj, ...
                                  'time_len_to_data_len',...
                                  time_shift_len, 'meg');
    else
        %
        % --- Time shift manual settings
        %
        
        % get time lists from gui
        command = ['[' get(H.time_shift_start_pos_edit, 'String') ']']; 
        start_pos = eval(command);
        command = ['[' get(H.time_shift_end_pos_edit, 'String') ']']; 
        end_pos = eval(command);
        [start_idx, err] = bayes_parm_editor(data.main_obj, ...
                                             'time_to_index',...
                                             start_pos, 'meg');
        if err, return, end;
        [end_idx, err] = bayes_parm_editor(data.main_obj, ...
                                           'time_to_index',...
                                           end_pos, 'meg');
        if err, return, end;
        
        % make parameters.
        twin_meg = [min(start_idx), max(end_idx)];
        Tperiod = end_idx - start_idx + 1;
        Tnext   = [start_idx - twin_meg(1) + 1];
        if length(Tnext) == 1
            Tnext = Tperiod;
        end
    end
    catch
    end

    bayes_parm.twin_meg = twin_meg;
    bayes_parm.Tnext = Tnext;
    bayes_parm.Tperiod = Tperiod;

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Estimation Model Parameters
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %Estimation model
    bayes_parm.forward_model = get_forward_model(H);
    bayes_parm.noise_estimation_model = get_noise_estimation_model(H);

    %Sensor noise file
    if get(H.specified_checkbox, 'Value')
        num_files = length(bayes_parm.megfile);
        meg_file = get(H.sensor_noise_file_edit, 'String');
        % copy sensor noise file(num_files)
        %bayes_parm.megnoisefile = repmat(cellstr(meg_file), num_files, 1);
        bayes_parm.megfile_baseline = repmat(cellstr(meg_file), num_files, 1);
    else
        %bayes_parm.megnoisefile = get(H.sensor_noise_file_listbox, 'String');
        bayes_parm.megfile_baseline = get(H.sensor_noise_file_listbox,'String');
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%
    % Observation noise
    %%%%%%%%%%%%%%%%%%%%%%%%%
    %Covariance
    bayes_parm.noise_model = get(H.obsn_cov_popup, 'Value');

    %Time window
    twin_from_msec = str2double(get(H.obsn_time_window_from_edit, 'String'));
    twin_to_msec = str2double(get(H.obsn_time_window_to_edit, 'String'));
    twin_from_index = bayes_parm_editor(data.main_obj, ...
                                        'time_to_index',...
                                        twin_from_msec, 'meg');
    twin_to_index = bayes_parm_editor(data.main_obj, ...
                                      'time_to_index',...
                                      twin_to_msec, 'meg');
    bayes_parm.twin_noise = [twin_from_index twin_to_index];

    %Update
    update_state = get(H.update_sx_checkbox, 'Value');
    if update_state
        bayes_parm.update_sx = ON;
    else
        bayes_parm.update_sx = OFF;
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%
    % Baseline activity
    %%%%%%%%%%%%%%%%%%%%%%%%%

    %Time window
    %twin_from_msec = str2double(get(H.pcv_time_window_from_edit, 'String'));
    %twin_to_msec = str2double(get(H.pcv_time_window_to_edit, 'String'));
    % Time window is the same that of the observation noise
    twin_from_msec = str2double(get(H.obsn_time_window_from_edit, 'String'));
    twin_to_msec = str2double(get(H.obsn_time_window_to_edit, 'String'));
    twin_from_index = bayes_parm_editor(data.main_obj, ...
                                        'time_to_index',...
                                        twin_from_msec, 'meg');
    twin_to_index = bayes_parm_editor(data.main_obj, ...
                                      'time_to_index',...
                                      twin_to_msec, 'meg');
    %bayes_parm.twin_global = [twin_from_index twin_to_index];
    bayes_parm.twin_baseline = [twin_from_index twin_to_index];

    %Update
    update_state = get(H.update_v_checkbox, 'Value');
    if update_state
        bayes_parm.update_v = ON;
    else
        bayes_parm.update_v = OFF;
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % flag_baseline, flag_noise_set
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    bayes_parm.flag_baseline  = str2double(get(H.flag_baseline_edit, 'String'));
    bayes_parm.flag_noise_set = str2double(get(H.flag_noise_set_edit, 'String'));
    
    % Smoothness filter radius
    bayes_parm.Rfilt = str2num(get(H.Rfilt_edit, 'String'))*1e-3; 

    % Reduce ratio
    bayes_parm.reduce = str2num(get(H.reduce_edit, 'String'));
    
    % Magnification parameter (a0_act)
    bayes_parm.a0_act = str2num(get(H.magnification_edit, 'String'));
    
    % Prior weight(prior_weight)
    bayes_parm.prior_weight = str2num(get(H.prior_weight_edit, 'String'));

    %bayes_parm.Ta0 = str2num(get(H.confidence_edit, 'String'));
    %bayes_parm.Ta0_act = str2num(get(H.confidence_edit, 'String'));

function data = callback(data, hObj)
% This function processes an GUI action.
% we can detect the action which object is changed by hObj.
%
% [IN]
%    data : application data (@see bayes_parm_basic_init)
%    hObj : event component handle
% [OUT]
%    data : some updated application data
%
global vbmeg_inst;
define = vbmeg_inst.const;

H = data.H;

switch(hObj)
    case H.project_root_push
        dirname = dir_dialog(get(H.project_root_edit, 'String'));
        if dirname ~= 0
            set(H.project_root_edit, 'String', dirname);
        end
    case H.brain_file_push
        project_root = get(H.project_root_edit, 'String');
        [dirname fname selected] = bayes_parm_gui_util(...
                                            'filedlg_relative',...
                                            {define.BRAIN_EXTENSION},...
                                            0, project_root, ...
                                            'Select cortical model file');
        if selected
            brain_file = fullfile(dirname, fname{1});
            set(H.brain_file_edit, 'String', brain_file);
            
            area_file = strrep(brain_file, define.BRAIN_EXTENSION, ...
                                           define.AREA_EXTENSION);
            area_file_abs = fullfile(project_root, area_file);
            
            act_file = strrep(brain_file, define.BRAIN_EXTENSION, ...
                                           define.FMRI_EXTENSION);
            act_file_abs  = fullfile(project_root, act_file);
            if exist(area_file_abs, 'file') == 2
                data = reload_gui_area_file(data, area_file);
            end
            if exist(act_file_abs, 'file') == 2
                set_gui_act_file(H, act_file);
            end
        end
    case H.basis_file_add_push
        project_root = get(H.project_root_edit, 'String');
        [dirname fname selected] = bayes_parm_gui_util(...
                                            'filedlg_relative',...
                                            {define.BASIS_EXTENSION},...
                                            0, project_root, ...
                                            'Select leadfield file');
        if selected
            num = length(fname);
            add_files = {};
            for n=1:num
                add_files = ...
                    [add_files; cellstr([dirname filesep fname{n}])];
            end
            filenames = get(H.basis_file_listbox, 'String');
            set_gui_basis_file(H, [filenames; add_files]);
        end
    case H.basis_file_remove_push
        idx = get(H.basis_file_listbox, 'Value');
        filenames = get(H.basis_file_listbox, 'String');
        if ~isempty(filenames)
            filenames(idx) = [];
            list_focus = idx(1)-1;
            if idx(1)-1 == 0
                list_focus = 1;
            end
            set(H.basis_file_listbox, 'Value', list_focus);
            set_gui_basis_file(H, filenames);
        end
    case H.meg_file_add_push
        project_root = get(H.project_root_edit, 'String');
        [dirname fnames selected] = bayes_parm_gui_util(...
                                        'filedlg_relative',...
                                        {define.MEG2_EXTENSION, ...
                                         define.EEG_EXTENSION},...
                                        0, project_root, ...
                                        'Select MEG/EEG data');
        if selected
            num = length(fnames);
            add_files = {};
            for n=1:num
                add_files = ...
                    [add_files; cellstr([dirname filesep fnames{n}])];
            end
            filenames = get(H.meg_file_listbox, 'String');
            data = set_gui_meg_file(data, [filenames; add_files]);
        end
    case H.meg_file_remove_push
        idx = get(H.meg_file_listbox, 'Value');
        filenames = get(H.meg_file_listbox, 'String');
        
        if ~isempty(filenames)
            filenames(idx) = [];   % remove selected filenames
            list_focus = idx(1)-1;
            if idx(1)-1 == 0
                list_focus = 1;
            end
            % update list focus.
            set(H.meg_file_listbox, 'Value', list_focus);
            data = set_gui_meg_file(data, filenames);
        end
    case H.result_file_outputdir_push
        project_root = get(H.project_root_edit, 'String');
        dirname = dir_dialog(get(H.project_root_edit, 'String'), false);
        if (dirname ~= 0) & (~isempty(project_root))
            path_relative = vb_relativepath(dirname, project_root);
            path_relative = path_relative(1:length(path_relative)-1);
            set(H.result_file_outputdir_edit, 'String', path_relative);
        end
    case H.result_filename_edit
        result_file = get(hObj, 'String');
%        [path_name, fname, ext] = fileparts(result_file);
        [path_name, fname, ext] = vb_get_file_parts(result_file);
        if isempty(ext)
            ext = define.BAYES_EXTENSION;
        end
        set(hObj, 'String', [fname ext]);
    case H.area_file_push
        project_root = get(H.project_root_edit, 'String');
        [dirname fname selected] = bayes_parm_gui_util(...
                                        'filedlg_relative',...
                                        {define.AREA_EXTENSION},...
                                        0, project_root, ...
                                        'Select cortical area file');
        if selected
            area_file = [dirname filesep fname{1}];
            data = reload_gui_area_file(data, area_file);
        end
    case H.act_file_push
        project_root = get(H.project_root_edit, 'String');
        [dirname fname selected] = bayes_parm_gui_util(...
                                        'filedlg_relative',...
                                        {define.FMRI_EXTENSION},...
                                        0, project_root, ...
                                        'Select cortical activity file');
        if selected
            act_file = [dirname filesep fname{1}];
            set_gui_act_file(H, act_file);
        end
    case { H.whole_time_window_from_edit, H.whole_time_window_to_edit }
        time = str2double(get(hObj, 'String'));
        [match_time, err] = time_match_sample_point(data, time, 'meg');
        if err == 1
            errordlg('select meg file.', 'error');
            set(hObj, 'String', '');
        elseif err == 2
            errordlg('Illigal input.', 'error');
            set(hObj, 'String', '');
        else
            set(hObj, 'String', sprintf('%.2f', match_time));
        end
    case H.time_shift_manual_radiobutton
        change_time_shift_mode(H, 'manual');
    case H.time_shift_equally_radiobutton
        change_time_shift_mode(H, 'equally');
    case {H.time_shift_window_size_edit, H.time_shift_shift_length_edit}
        time_len = str2double(get(hObj, 'String'));
        if time_len < 0
            errordlg('The value should be positive value.', 'Error');
            set(hObj, 'String', '');
        else
            [ceil_time_len, err] = ceil_time_length(data, time_len, 'meg');
            if err == 1
                errordlg('select meg file.', 'error');
            elseif err == 2
                errordlg('Illigal input.', 'error');
            else
                set(hObj, 'String', sprintf('%.2f', ceil_time_len));
            end
        end
    case { H.time_shift_start_pos_edit, H.time_shift_end_pos_edit }
        %start positions, end positions check
        command = ['[' get(hObj, 'String') ']']; 
        try
            pos = eval(command);
            [match_time, err] = time_match_sample_point(data, pos, 'meg');
        catch
            errordlg('positions are invalid.', 'error');
            return;
        end
        match_time = sort(match_time);
        pos_text = bayes_parm_gui_util('comma_text', match_time);
        set(hObj, 'String', pos_text);
    case H.duration_check_push
        % Display durations
        [start_pos, end_pos, err] = get_interval(H);
        if err
            return;
        end
        time_window_string = [num2str(start_pos, '%-8.2f'),...
                            repmat(' - ', length(start_pos), 1),...
                            num2str(end_pos, '%-8.2f')];
        % set to listbox
        set(H.duration_listbox, 'String', time_window_string);
        % set listbox line
        set(H.duration_listbox, 'Value', 1);
    case H.noise_estimation_model_popup
        set_estimation_GUI_properties(H);
        %case H.forward_model_popup
        %forward_model = get_forward_model(H);
        %H = set_forward_model(H, forward_model);
    case H.specified_checkbox
        if get(hObj, 'Value')
            data = change_sensor_noise_file_selection_mode(data, ON);
        else
            data = change_sensor_noise_file_selection_mode(data, OFF);
        end
    case H.sensor_noise_file_edit
        data = set_noise_file_time_window(data);
    case H.sensor_noise_file_push
        project_root = get(H.project_root_edit, 'String');
        [dirname fname selected] = bayes_parm_gui_util('filedlg_relative',...
                                                        {define.MEG2_EXTENSION},...
                                                        0,...
                                                        project_root);
        if selected
            set(H.sensor_noise_file_edit, 'String', [dirname filesep fname{1}]);
            data = set_noise_file_time_window(data);
        end
    case {H.obsn_time_window_from_edit, ...
          H.obsn_time_window_to_edit} %, ...
          %H.pcv_time_window_from_edit, ...
          %H.pcv_time_window_to_edit}
        value = str2double(get(hObj, 'String'));
        [match_time, err] = time_match_sample_point(data, value, 'noise');
        if err == 1
            errordlg('select meg file.', 'error');% current(v0.50) GUI,
            set(hObj, 'String', '');              % Sensor noise file = Meg noise file
        elseif err == 2
            errordlg('Illigal input.', 'error');
            set(hObj, 'String', '');
        else
            set(hObj, 'String', sprintf('%.2f', match_time));
        end
    case H.advanced_parm_push
        bayes_parm_editor_init_advanced_gui(data.main_obj);
    case H.save_push
        % gui parameters collection
        bayes_parm = get_gui_parms(data);
        % send parameters to main function
        bayes_parm_editor(data.main_obj, 'merge_bayes_parm', bayes_parm);
        % send save message to main function
        bayes_parm_editor(data.main_obj, 'save_bayes_parm');
    case H.load_push
        % send load message to main function
        bayes_parm_editor(data.main_obj, 'load_bayes_parm');
    case H.exec_push
        push_button_caption = get(H.exec_push, 'String');
        set(H.exec_push, 'String', 'Processing...');
        pause(0.01); % for update button string
        
        % gui parameters collection
        bayes_parm = get_gui_parms(data);
        % send parameters to main function
        bayes_parm_editor(data.main_obj, 'merge_bayes_parm', bayes_parm);
        % send execution message to main function
        bayes_parm_editor(data.main_obj, 'exec_estimation');
        
        set(H.exec_push, 'String', 'Exec');
    %%%%%%%%%%%%%%%%
    % Menu
    case {H.figure, H.menu_close}
        bayes_parm_basic_close(H.figure);
    case H.menu_current_variance_estimation
        url = vb_users_manual_url('Current variance estimation');
        vb_browser_show(url);
end

function [start_pos, end_pos, err] = get_interval(H,data)
% This function gets small window start points and end points.
% [IN]
%      H : GUI component handle.
% [OUT]
%      start_pos : time window start positions
%      end_pos   : time window end positions
%      err       : return true when time window is invalid.

    start_pos = [];
    end_pos = [];
    err = false;
    
    meg_info = data.meg_info;

    if get(H.time_shift_manual_radiobutton, 'Value')
        % manual Shift
        command = ['[' get(H.time_shift_start_pos_edit, 'String') ']']; 
        start_pos = eval(command);
        start_pos = start_pos(:);
        command = ['[' get(H.time_shift_end_pos_edit, 'String') ']']; 
        end_pos = eval(command);
        end_pos = end_pos(:);

        if length(start_pos) ~= length(end_pos)
            errordlg('Check Start positions and End positions.', 'Error');
            err = true;
        end
    else
        % equally Shift
        window_size = str2double(...
                            get(H.time_shift_window_size_edit, 'String'));
        shift_length = str2double(...
                            get(H.time_shift_shift_length_edit, 'String'));
        begin = str2double( ...
                            get(H.whole_time_window_from_edit, 'String'));
        last  = str2double(...
                            get(H.whole_time_window_to_edit, 'String'));
        
        index = vb_time_to_index([begin last],meg_info);
        Tsample = index(2)-index(1)+1;
        Tperiod = vb_time_to_numsamp(window_size,sample_freq);
        Tnext = vb_time_to_numsamp(shift_length,sample_freq);
        Twindow = vb_calc_timewindow(Tsample,Tperiod,Tnext);
        
        start_pos = Twindow(:,1);
        end_pos = Twindow(:,2);
        start_pos = vb_index_to_time(start_pos,meg_info);
        end_pos = vb_index_to_time(end_pos,meg_info);
        
        % small window start, end positions
        %w_start_pos = begin;
        %w_end_pos = 0;

        %n = 1;
        %st = clock;
        %while( (w_end_pos <= last) && (w_start_pos < last) )
        %    loop_out = false;
        %    w_end_pos = w_start_pos + window_size;
        %    if w_end_pos >= last
        %        w_end_pos = last;
        %        loop_out = true;
        %%    end
        %
        %    % add small window start_pos, end_pos
        %    start_pos = [start_pos; w_start_pos];
        %    end_pos = [end_pos; w_end_pos];
        %
        %    w_start_pos = w_start_pos + shift_length;
        %    n = n + 1;
        %    if loop_out || etime(clock, st) >= 3
        %        break;
        %    end
        %end
    end

function data = set_noise_file_time_window(data, bayes_parm)
% This function loads sensor noise file and sets time window.
% [IN]
%    data : application data(@see bayes_parm_basic_gui_init)
% [OUT]
%    data : some updated application data

    H = data.H;

    project_root = get(H.project_root_edit, 'String');
    noise_file = [];
    
    if get(H.specified_checkbox, 'Value') % specified sensor noise file
        noise_file = get(H.sensor_noise_file_edit, 'String');
    else
        filenames = get(H.sensor_noise_file_listbox, 'String'); % same with meg_file
        if length(filenames)
            noise_file = filenames{1}; % meg_file{1}
        end
    end
    
    % store sensor_noise_file data to the data cabinet.
    filename = [];
    if ~isempty(noise_file)
        filename = [project_root filesep noise_file];
    end
    set_sensor_noise_file_info(data, filename);

    
    % get sensor noise info
    meg_info = bayes_parm_editor(data.main_obj, 'get_sensor_noise_file_info');
    if isempty(meg_info)
        return;
    end

    pretrigger = vb_meginfo_get_pre_trigger(meg_info);
    % convert index to the msec
    if exist('bayes_parm', 'var') && ...
       isfield(bayes_parm, 'twin_noise') && ...
         size(bayes_parm.twin_noise, 2) == 2

        time_window_noise_from    = bayes_parm.twin_noise(1);
        time_window_noise_to      = bayes_parm.twin_noise(2);
    else
        time_window_noise_from    = 1;
        time_window_noise_to      = pretrigger;
    end
    if exist('bayes_parm', 'var') && ...
       isfield(bayes_parm, 'twin_baseline') && ...
         size(bayes_parm.twin_baseline, 2) == 2
        time_window_global_from   = bayes_parm.twin_baseline(1);
        time_window_global_to     = bayes_parm.twin_baseline(2);
    else
        time_window_global_from   = 1;
        time_window_global_to     = pretrigger;
    end

    [noise_from, err] = bayes_parm_editor(data.main_obj, ...
                                          'index_to_time', ...
                                          time_window_noise_from, ...
                                          'noise');
    [noise_to, err]   = bayes_parm_editor(data.main_obj, ...
                                          'index_to_time', ...
                                          time_window_noise_to, ...
                                          'noise');
    noise_from_str = sprintf('%.2f', noise_from);
    noise_to_str = sprintf('%.2f', noise_to);

    [global_from, err] = bayes_parm_editor(data.main_obj, ...
                                           'index_to_time', ...
                                           time_window_global_from, ...
                                           'noise');
    [global_to, err]   = bayes_parm_editor(data.main_obj, ...
                                           'index_to_time', ...
                                           time_window_global_to, ...
                                           'noise');
    global_from_str = sprintf('%.2f', global_from);
    global_to_str = sprintf('%.2f', global_to);
    
    % Observation noise time window
    set(H.obsn_time_window_from_edit, 'String', noise_from_str);
    set(H.obsn_time_window_to_edit, 'String', noise_to_str);

    % Prior current variance time window
    %set(H.pcv_time_window_from_edit, 'String', global_from_str);
    %set(H.pcv_time_window_to_edit, 'String', global_to_str);

function data = change_sensor_noise_file_selection_mode(data, mode)
% This function changes GUI properties of Sensor noise file selection.
% [IN]
%    data : application data(@see bayes_parm_basic_init)
%    mode : ON(specified one file) or OFF
% [OUT]
%    data : some updated application data.

    H = data.H;
    switch(mode)
        case ON
            set(H.sensor_noise_file_edit, 'Visible', 'on');
            set(H.sensor_noise_file_push, 'Visible', 'on');
            set(H.sensor_noise_file_listbox, 'Visible', 'off');
            data = set_noise_file_time_window(data);
        case OFF
            set(H.sensor_noise_file_edit, 'Visible', 'off');
            set(H.sensor_noise_file_push, 'Visible', 'off');
            set(H.sensor_noise_file_listbox, 'Visible', 'on');
            data = set_noise_file_time_window(data);
    end

function change_time_shift_mode(H, mode)
% This function changes GUI properties of Time shift.
% [IN]
%       H : GUI component handle.
%    mode : 0-manual 1- equally shift 
% [OUT]
%    none
    switch(mode)
        case 'manual'
            set(H.time_shift_equally_radiobutton, 'Value', 0);
            set(H.time_shift_manual_radiobutton, 'Value', 1);
            % start_pos, end_pos gui visible
            set(H.time_shift_start_pos_label, 'Visible', 'on');
            set(H.time_shift_end_pos_label, 'Visible', 'on');
            set(H.time_shift_start_pos_edit, 'Visible', 'on');
            set(H.time_shift_end_pos_edit, 'Visible', 'on');
            % equally setting gui invisible
            set(H.time_shift_window_size_label, 'Visible', 'off');
            set(H.time_shift_shift_length_label, 'Visible', 'off');
            set(H.time_shift_window_size_edit, 'Visible', 'off');
            set(H.time_shift_shift_length_edit, 'Visible', 'off');
            % time range makes enable off
            set(H.analysis_time_range_text, 'Visible', 'off');
            set(H.whole_time_window_from_edit, 'Visible', 'off');
            set(H.whole_time_window_to_edit,   'Visible', 'off');
            set(H.whole_time_window_minus_text, 'Visible', 'off');
        otherwise
            set(H.time_shift_manual_radiobutton, 'Value', 0);
            set(H.time_shift_equally_radiobutton, 'Value', 1);
            % start_pos, end_pos gui invisible
            set(H.time_shift_start_pos_label, 'Visible', 'off');
            set(H.time_shift_end_pos_label, 'Visible', 'off');
            set(H.time_shift_start_pos_edit, 'Visible', 'off');
            set(H.time_shift_end_pos_edit, 'Visible', 'off');
            % equally setting gui visible
            set(H.time_shift_window_size_label, 'Visible', 'on');
            set(H.time_shift_shift_length_label, 'Visible', 'on');
            set(H.time_shift_window_size_edit, 'Visible', 'on');
            set(H.time_shift_shift_length_edit, 'Visible', 'on');
            % time range makes enable on
            set(H.analysis_time_range_text, 'Visible', 'on');
            set(H.whole_time_window_from_edit, 'Visible', 'on');
            set(H.whole_time_window_to_edit,   'Visible', 'on');
            set(H.whole_time_window_minus_text, 'Visible', 'on');
    end

function H = change_forward_model(H, forward_model)
% Don't call this function.
% If you change forward model, you should call through set_forward_model().
% This function changes forward model on GUI.
% 
% [IN]
%    H : GUI component handle.
%    forward_model : 'focal' or 'focal+global'
% [OUT]
%    H : GUI component handle.

    %forward_models = get(H.forward_model_popup, 'String');
    %idx = strmatch(forward_model, forward_models, 'exact');
    %set(H.forward_model_popup, 'Value', idx);
    forward_model = STR_FORWARD_MODEL_FOCAL;

    switch(forward_model)
        case STR_FORWARD_MODEL_FOCAL
            models = { STR_COVARIANCE_NOISE,...
                       STR_COVARIANCE_NOISE_NO_UPDATE };
            value = 1;
        case STR_FORWARD_MODEL_FOCAL_GLOBAL
            models = { STR_COVARIANCE_NOISE,...
                       STR_COVARIANCE_NOISE_NO_UPDATE,...
                       STR_ISOTROPIC_NOISE,...
                       STR_ISOTROPIC_NOISE_NO_GLOBAL_UPDATE,...
                       STR_ISOTROPIC_NOISE_NO_UPDATE };
            value = 1;
    end

    set(H.noise_estimation_model_popup, 'String', models);
    set(H.noise_estimation_model_popup, 'Value', value);

    set_estimation_GUI_properties(H);

function set_estimation_GUI_properties(H)
% This function changes GUI properties of 'Estimation Model Parameters'
%  by Estimation model.
% [IN]
%   H : GUI component handle.
% [OUT]
%   none

    %models = get(H.forward_model_popup, 'String');
    %ix     = get(H.forward_model_popup, 'Value');
    forward_model= STR_FORWARD_MODEL_FOCAL; %models{ix};

    estimation_models = get(H.noise_estimation_model_popup, 'String');
    ix                = get(H.noise_estimation_model_popup, 'Value');
    estimation_model  = estimation_models{ix};

    switch(forward_model)
    case STR_FORWARD_MODEL_FOCAL
        switch(estimation_model)
        case STR_COVARIANCE_NOISE
            set(H.update_sx_checkbox, 'Value', 1);
            set(H.update_v_checkbox, 'Value', 0);
            set(H.obsn_cov_popup, 'Value', 3); % noise model = 3
            set(H.flag_baseline_edit, 'String', '0');
            set(H.flag_noise_set_edit, 'String', '0');
        case STR_COVARIANCE_NOISE_NO_UPDATE
            set(H.update_sx_checkbox, 'Value', 0);
            set(H.update_v_checkbox, 'Value', 0);
            set(H.obsn_cov_popup, 'Value', 3); % noise model = 3
            set(H.flag_baseline_edit, 'String', '0');
            set(H.flag_noise_set_edit, 'String', '0');
        end
    case STR_FORWARD_MODEL_FOCAL_GLOBAL
        switch(estimation_model)
        case STR_COVARIANCE_NOISE
            set(H.update_sx_checkbox, 'Value', 1);
            set(H.update_v_checkbox, 'Value', 0);
            set(H.obsn_cov_popup, 'Value', 3); % noise model = 3
            set(H.flag_baseline_edit, 'String', '0');
            set(H.flag_noise_set_edit, 'String', '0');
        case STR_COVARIANCE_NOISE_NO_UPDATE
            set(H.update_sx_checkbox, 'Value', 0);
            set(H.update_v_checkbox, 'Value', 0);
            set(H.obsn_cov_popup, 'Value', 3); % noise model = 3
            set(H.flag_baseline_edit, 'String', '0');
            set(H.flag_noise_set_edit, 'String', '0');
        case STR_ISOTROPIC_NOISE
            set(H.update_sx_checkbox, 'Value', 1);
            set(H.update_v_checkbox, 'Value', 1);
            set(H.obsn_cov_popup, 'Value', 1); % noise model = 1
            set(H.flag_baseline_edit, 'String', '0');
            set(H.flag_noise_set_edit, 'String', '1');
        case STR_ISOTROPIC_NOISE_NO_GLOBAL_UPDATE
            set(H.update_sx_checkbox, 'Value', 1);
            set(H.update_v_checkbox, 'Value', 0);
            set(H.obsn_cov_popup, 'Value', 1); % noise model = 1
            set(H.flag_baseline_edit, 'String', '0');
            set(H.flag_noise_set_edit, 'String', '1');
        case STR_ISOTROPIC_NOISE_NO_UPDATE
            set(H.update_sx_checkbox, 'Value', 0);
            set(H.update_v_checkbox, 'Value', 0);
            set(H.obsn_cov_popup, 'Value', 1); % noise model = 1
            set(H.flag_baseline_edit, 'String', '0');
            set(H.flag_noise_set_edit, 'String', '1');
        end
    end

function data = set_gui_meg_file(data, filenames)
% This function adds filenames to the meg file listbox.
% note : meg file is used as sensor noise file by default.
%        This function sets the sensor noise file when the file is not 
%        specifield.
% [IN]
%    data : application data(@see bayes_parm_basic_init)
%    filenames : meg files(relative path from project_root)
% [OUT]
%    data : some updated application data

    H = data.H;

    set(H.meg_file_listbox, 'String', filenames);

    % store megfile info to the data cabinet.
    if length(filenames)
        %
        % --- File is inputted
        %
        project_root = get(H.project_root_edit, 'String');
        if ischar(filenames)
            filename = [project_root filesep, filenames];
        else
            filename = [project_root filesep filenames{1}];
        end
        err = set_meg_file_info(data, filename);
        if err, return; end
        meg_info = get_meg_file_info(data);
        [time_window_from, err] = bayes_parm_editor(data.main_obj, ...
                                                    'index_to_time', ...
                                                    1, 'meg');
        [time_window_to, err]   = bayes_parm_editor(data.main_obj, ...
                                                    'index_to_time', ...
                                                    meg_info.Nsample, 'meg');
        
        time_window_from_str = sprintf('%.2f', time_window_from);
        time_window_to_str = sprintf('%.2f', time_window_to);

        % analysis time range (default analysis window covers all the data secsions.)
        wsize_msec = bayes_parm_editor(data.main_obj, ...
                                       'data_len_to_time_len', ...
                                       meg_info.Nsample, 'meg');
        wsize_msec = sprintf('%.2f', wsize_msec);
    else
        %
        % --- File not specified
        %
        % for clean up GUI
        err = set_meg_file_info(data, '');
        time_window_from_str = [];
        time_window_to_str = [];
        wsize_msec = [];
    end

    % time window range for display
    set(H.whole_time_range_from_edit, 'String', time_window_from_str);
    set(H.whole_time_range_to_edit, 'String', time_window_to_str);

    % Analysis time range
    set(H.whole_time_window_from_edit, 'String', time_window_from_str);
    set(H.whole_time_window_to_edit, 'String', time_window_to_str);

    % Window size, Shift length
    set(H.time_shift_window_size_edit, 'String', wsize_msec);
    set(H.time_shift_shift_length_edit, 'String', wsize_msec);
    
    set(H.sensor_noise_file_listbox, 'String', filenames);
    set_noise_file_time_window(data);

function set_gui_basis_file(H, basis_files)
% This function adds filenames to the basis file listbox.
% [IN]
%              H : component handles.
%    basis_files : basis filename list(relative path from project_root)
% [OUT]
%    none

    set(H.basis_file_listbox, 'String', basis_files);

function set_gui_act_file(H, act_file)
% This function loads actkeys and set it to the actkey listbox.
% [IN]
%            H : component handles.
%     act_file : actfile(relative path from project_root)
% [OUT]
%    H : component handles.

    % set filename
    set(H.act_file_edit, 'String', act_file);
    
    keys = [];
    if ~isempty(act_file)
        % get act keys
        project_root = get(H.project_root_edit, 'String');
        act_file_full = fullfile(project_root, act_file);
        if exist(act_file_full, 'file') == 2
            load(act_file_full, 'Act');
            for n = 1 : length(Act)
                keys{n} = Act{n}.key;
            end
        end
    end
    
    % set act keys to the listbox
    set(H.act_key_listbox, 'String', keys);

function set_gui_area_file(H, area_file)
% This function load areakeys and set it to the areakey popup.
% see :  reload_gui_area_file
% [IN]
%            H : component handles.
%    area_file : areafile(relative path from project_root)
% [OUT]
%    none

    % set filename
    set(H.area_file_edit, 'String', area_file);

    keys = [];
    if ~isempty(area_file)
        % get area keys
        keys = {};
        project_root = get(H.project_root_edit, 'String');
        area_file_full = fullfile(project_root, area_file);
        if exist(area_file_full, 'file') == 2
            load([project_root filesep area_file], 'Area');
            for n = 1 : length(Area)
                keys{n} = Area{n}.key;
            end
        end
    end
    
    % set area keys to the popup
    set(H.area_key_listbox, 'String', keys);
    set(H.area_key_listbox, 'Value', 1);

function data = reload_gui_area_file(data, area_file)
% This function load areakeys and set it to the areakey popup.
% note:  this function changes the advanced parameter(area_key_global).
% see :  set_gui_area_file
% [IN]
%         data : application data (@see bayes_parm_basic_init)
%    area_file : areafile(relative path from project_root)
% [OUT]
%    data : some updated application data

    H = data.H;

    % set filename
    set(H.area_file_edit, 'String', area_file);

    % get area keys
    keys = {};
    project_root = get(H.project_root_edit, 'String');
    load([project_root filesep area_file], 'Area');
    for n = 1 : length(Area)
        keys{n} = Area{n}.key;
    end

    % set area keys to the popup
    set(H.area_key_listbox, 'String', keys);
    set(H.area_key_listbox, 'Value', 1);
    
    % advanced parameter(area_key_global) will be keys{1}.
    bayes_parm.area_key_global = keys{1};
    bayes_parm_editor(data.main_obj, 'merge_bayes_parm', bayes_parm);
    
function [match_time, err] = time_match_sample_point(data, time, file_type)
% This function returns time on the nearest sample point at the 
% specified time. The value is rounded up. 
% (ex. There are sample points : 1.0, 2.0 ,...
%      specifield time: 1.0<x<=2.0
%      match_time is 2.0)
%
% [IN]
%         data : application data (@see bayes_parm_basic_init)
%         time : user input time data
%    file_type : calculate time by 'meg'(megfile) or 'noise'(sensor_noise file)
% [OUT]
%    match_time : user input time data matched with sample data.
%           err : 1 or 2 (1:file does not specified, 2:user input error)

    % nearest index with specified time.
    match_time = [];
    err = false;
    
    [index, err] = bayes_parm_editor(data.main_obj, ...
                                     'time_to_index', ...
                                     time, file_type);
    if err, return; end
    [match_time, err] = bayes_parm_editor(data.main_obj, ...
                                          'index_to_time', ...
                                          index, file_type);
    if err, return; end
    
function [ceil_time_len, err] = ceil_time_length(data, time_len, file_type)
% This function rounds up the specified time length to the multiple
% of the sample length . 
%
% [IN]
%         data : application data (@see bayes_parm_basic_init)
%    time_len  : time length(msec)
%    file_type : calculate time_len by 'meg'(megfile)
%                                   or 'noise'(sensor_noise file)
% [OUT]
%    ceil_time_len : rounded up time length
%              err : 1 or 2 (1:file does not specified, 2:user input error)

    ceil_time_len = [];
    err = false;

    [data_len, err] = bayes_parm_editor(data.main_obj, ...
                                        'time_len_to_data_len',...
                                        time_len, ...
                                        file_type);

    % Calculate analysis time window size(sample)
    analysis_time_from_msec = ...
        str2double(get(data.H.whole_time_window_from_edit, 'String'));
    analysis_time_to_msec = ...
        str2double(get(data.H.whole_time_window_to_edit, 'String'));

    [analysis_time_from_sample, err] = bayes_parm_editor(data.main_obj, ...
                                            'time_len_to_data_len', ...
                                            analysis_time_from_msec, ...
                                            file_type);

    if err, return; end
    [analysis_time_to_sample, err] = bayes_parm_editor(data.main_obj, ...
                                            'time_len_to_data_len', ...
                                            analysis_time_to_msec, ...
                                            file_type);
    analysis_time_sample = ...
        analysis_time_to_sample - analysis_time_from_sample + 1;

    % Convert inputted time(msec) to sample number.
    [time_sample, err] = bayes_parm_editor(data.main_obj, ...
                                            'time_len_to_data_len', ...
                                            time_len, ...
                                            file_type);

    if time_sample > analysis_time_sample
        time_sample = analysis_time_sample;
    end
    [ceil_time_len, err] = bayes_parm_editor(data.main_obj, ...
                                        'data_len_to_time_len', ...
                                        time_sample, ...
                                        file_type);
    if err, return;end;

function [meg_info, err] = get_meg_file_info(data)
% get megfile info from bayes_parm_editor object
% [IN]
%    data : application data (@see bayes_parm_basic_init)
% [OUT]
%    meg_info : megfile information.
%    err      : true(megfile data is not set.) or false

    err = false;
    meg_info = bayes_parm_editor(data.main_obj, 'get_megfile_info');
    if isempty(meg_info)
        errordlg('select meg file.', 'Error');
        err = true;
    end

function [err] = set_meg_file_info(data, meg_file)
% set megfile info to bayes_parm_editor main object.
% [IN]
%        data : application data (@see bayes_parm_basic_init);
%    meg_file : meg file name.
% [OUT]
%    err : true(meg_file is invalid.) or false

    err = false;
    try
        bayes_parm_editor(data.main_obj, 'set_megfile_info', meg_file);
    catch
        err = true;
        errordlg('file read error occured.', 'Error');
    end

function [s_noise_file_info, err] = get_sensor_noise_file_info(data)
% get sensor noise file info from bayes_parm_editor main object.
% [IN]
%    data : application data (@see bayes_parm_basic_init);
% [OUT]
%    s_noise_info : sensor noise file information.
%             err : true(sensor noise file data is not set.) or false

   err = false;
   s_noise_file_info = bayes_parm_editor(data.main_obj, ...
                                         'get_sensor_noise_file_info');
   if isempty(s_noise_file_info)
       errordlg('select meg file.', 'Error'); % current(v0.50) GUI,
                                              % Sensor noise file = Meg noise file
       err = true;
   end

function [err] = set_sensor_noise_file_info(data, s_noise_file)
% This function sets the sensor noise file info to the data cabinet.
% see : get_sensor_noise_file_info
%     : bayes_parm_basic_gui('set_sensor_noise_file_info')
% [IN]
%            data : application data (@see bayes_parm_basic_init)
%    s_noise_file : sensor noise file name.
% [OUT]
%     err : true(sensor noise file is invalid.) or false

    err = false;
    try
        bayes_parm_editor(data.main_obj, ...
                          'set_sensor_noise_file_info', s_noise_file);
    catch
        err = true;
        errordlg('file read error occured.', 'Error');
    end

function [forward_model] = get_forward_model(H)
% This function returns forward model parameter
% [IN]
%    H : handles of components
% [OUT]
%    forward_model : forward model parameter
%

    %models = get(H.forward_model_popup, 'String');
    %forward_model = models{get(H.forward_model_popup, 'Value')};
    forward_model = STR_FORWARD_MODEL_FOCAL;

function H = set_forward_model(H, forward_model, force_update)
% This function sets GUI forward model.
% [IN]
%    H : handles of components
%    forward_model : to be set foward_model
%    force_update : all parameters should updates forcelly,
%                   specify this value true.
% [OUT]
%    H : handles of components

    if ~exist('force_update', 'var')
        force_update = false;
    end

    % forward model change.
    % update timing:
    %   1.force_update flag is true.
    %   2.If current_model is different from forward_model.
    %
    current_model = get_forward_model(H);
    if ((force_update == true) || ~(strcmp(current_model, forward_model)))
        % all parameters change
        change_forward_model(H, forward_model);
    end

function [estimation_model_number] = get_noise_estimation_model(H)
% This function returns estimation_model number.
% [IN]
%    H : handles of components
% [OUT]
%    estimation_model_number : defined estimation model number
%

    models = get(H.noise_estimation_model_popup, 'String');
    noise_estimation_model = models{get(H.noise_estimation_model_popup, 'Value')};

    switch(noise_estimation_model)
        case STR_COVARIANCE_NOISE
            estimation_model_number = 1;
        case STR_COVARIANCE_NOISE_NO_UPDATE
            estimation_model_number = 2;
        case STR_ISOTROPIC_NOISE
            estimation_model_number = 3;
        case STR_ISOTROPIC_NOISE_NO_GLOBAL_UPDATE
            estimation_model_number = 4;
        case STR_ISOTROPIC_NOISE_NO_UPDATE
            estimation_model_number = 5;
    end

function [treatable] = IsTreatableEstimationModel(bayes_parm)
% This function checks estimation model parameters section of bayes_parm.
% GUI has a restriction about combination between foward model and
% noise estimation model. 
%
% [IN]
%    bayes_parm : check target
% [OUT]
%    treatable  : true or false. (true means treatable.)

    treatable = false;

    switch(bayes_parm.forward_model)
    case STR_FORWARD_MODEL_FOCAL
        switch(bayes_parm.noise_estimation_model)
        case 1   % COVARIANCE_NOISE
            if (bayes_parm.update_sx == ON) && ...
               (bayes_parm.update_v  == OFF) && ...
               (bayes_parm.noise_model == 3) && ...
               (bayes_parm.flag_baseline == 0) && ...
               (bayes_parm.flag_noise_set == 0)
               
               treatable = true;
            end
        case 2   % COVARIANCE_NOISE_NO_UPDATE
            if (bayes_parm.update_sx == OFF) && ...
               (bayes_parm.update_v  == OFF) && ...
               (bayes_parm.noise_model == 3) && ...
               (bayes_parm.flag_baseline == 0) && ...
               (bayes_parm.flag_noise_set == 0)
               
               treatable = true;
            end
        end
    case STR_FORWARD_MODEL_FOCAL_GLOBAL
        switch(bayes_parm.noise_estimation_model)
        case 1   % COVARIANCE_NOISE
            if (bayes_parm.update_sx == ON) && ...
               (bayes_parm.update_v  == OFF) && ...
               (bayes_parm.noise_model == 3) && ...
               (bayes_parm.flag_baseline == 0) && ...
               (bayes_parm.flag_noise_set == 0)
               
               treatable = true;
            end
        case 2   % COVARIANCE_NOISE_NO_UPDATE
            if (bayes_parm.update_sx == OFF) && ...
               (bayes_parm.update_v  == OFF) && ...
               (bayes_parm.noise_model == 3) && ...
               (bayes_parm.flag_baseline == 0) && ...
               (bayes_parm.flag_noise_set == 0)
               
               treatable = true;
            end
        case 3   % ISOTROPIC_NOISE
            if (bayes_parm.update_sx == ON) && ...
               (bayes_parm.update_v  == ON) && ...
               (bayes_parm.noise_model == 1) && ...
               (bayes_parm.flag_baseline == 0) && ...
               (bayes_parm.flag_noise_set == 1)
               
               treatable = true;
            end
        case 4   % ISOTROPIC_NOISE_NO_GLOBAL_UPDATE
            if (bayes_parm.update_sx == ON) && ...
               (bayes_parm.update_v  == OFF) && ...
               (bayes_parm.noise_model == 1) && ...
               (bayes_parm.flag_baseline == 0) && ...
               (bayes_parm.flag_noise_set == 1)
               
               treatable = true;
            end
        case 5   % ISOTROPIC_NOISE_NO_UPDATE
            if (bayes_parm.update_sx == OFF) && ...
               (bayes_parm.update_v  == OFF) && ...
               (bayes_parm.noise_model == 1) && ...
               (bayes_parm.flag_baseline == 0) && ...
               (bayes_parm.flag_noise_set == 1)
               
               treatable = true;
            end
        end
    end


%%%%%%%%%%%%%%%%%%%%%%%
% Define constants
%

%
% -- forward model strings (This string should be set to global.)
%
function [v] = STR_FORWARD_MODEL_FOCAL
    v = 'focal';
function [v] = STR_FORWARD_MODEL_FOCAL_GLOBAL
    v = 'focal+global';

%
% -- Noise model estimation strings
%
function [v] = STR_COVARIANCE_NOISE
    v = 'COVARIANCE NOISE';
function [v] = STR_COVARIANCE_NOISE_NO_UPDATE
    v = 'COVARIANCE NOISE(NO UPDATE)';
function [v] = STR_ISOTROPIC_NOISE
    v = 'ISOTROPIC NOISE';
function [v] = STR_ISOTROPIC_NOISE_NO_GLOBAL_UPDATE
    v = 'ISOTROPIC NOISE(NO GLOBAL UPDATE)';
function [v] = STR_ISOTROPIC_NOISE_NO_UPDATE
    v = 'ISOTROPIC NOISE(NO UPDATE)';

