function [fMRINew, AreaNew, SPM] = vb_make_fmri_data(proj_root, fmri_parm)
% Make fMRI activity data from SPM voxel file. 
% 
% [syntax]
% [fMRINew, AreaNew, SPM] = vb_make_fmri_data(proj_root, fmri_parm)
%
% [input]
% proj_root: <<string>> Project root directory.
% fmri_parm: <<struct>> Parameters for SPM voxel mapping.
% --- fields of fmri_parm
%  brain_file  : <<string>> Cortical surface model file (.brain.mat).
%  spm_file    : <<string>> SPM voxel file (.spm.mat). 
%  fmri_id     : <<string>> fMRI ID.
%  SPM_Radius  : <<float>> Search radius for SPM active points.
%  SPM_zstep   : <<float>> Slice step size in z-direction.
%  Gauss_radius: <<float>> Half-width of Gaussian filter.
%  Gauss_max   : <<float>> Maximum distance for averaging SPM.
% ---
% 
% [output]
% fMRINew: <<struct>> fMRI activity information. 
% --- fields of fMRINew
%  xxP: <<vector>> fMRI activity. 
%  key: <<string>> ID of this data for retrieval. 
% ---
%
% AreaNew: <<struct>> Activated area information based on fMRI. 
% --- fields of AreaNew
%  Iextract: <<vector>> Index of area. 
%  key     : <<string>> ID of the area created. 
% ---
%
% [history]
% Originaly written by S.Kajihara
% Ver 1.1  modified by M. Sato  2003-4-15
%
% 2005-03-20 Modified by TY
% Ver 0.20  modified by M. Sato  2005-4-8
% Coordinate system is changed SPM right-handed coordinate.
% 2005-09-06 O.Y Ver 0.30b
% An area information is also created.
% 2008-05-28 rhayashi  removed Gauss_step parameter
% 2008-11-10 Taku Yoshioka
%  Struct 'SPM' is returned. 
% 2010-06-25 Taku Yoshioka
%  RAS coordinate system is assumed for SPM voxel file, but this function
%  maintains backward compatibility with old version of SPM voxel
%  file. See line 85-93. 
%  Verbose level support. 
% 2016-5-31 M Sato
%  cortex mapping is done according to MNI or SPM coordinate
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% VBMEG constants and cortical model file
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
coord_const = vb_define_coordinate;

if isempty(proj_root)
	brain_file = [fmri_parm.brain_file];
else
	brain_file = [proj_root filesep fmri_parm.brain_file];
end
%
% Parameters about fMRI 
%
% A vertex point of cortical surface V(n) is chosen as an activity when 
% it is located inside the sphere of radius 'SPM_Radius' with the center
% V(n).
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

vb_disp(sprintf('SPM_Radius  : %g',fmri_parm.SPM_Radius));
vb_disp(sprintf('SPM_zstep   : %g',fmri_parm.SPM_zstep));

% Gauss filter parameter to average SPM t-value
vb_disp(sprintf('Gauss_radius: %g',fmri_parm.Gauss_radius));
vb_disp(sprintf('Gauss_max   : %g',fmri_parm.Gauss_max));

%
% Load data
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%load([brain_file],'V'); % Vertex in SPM coordinate [m]
%if isfield(fmri_parm,'MNI') && fmri_parm.MNI == ON
%	V = vb_load_cortex(brain_file,'MNI');
%else
%	V = vb_load_cortex(brain_file);
%end

load([fmri_parm.spm_file]); % SPM data
vb_disp(['--- Load SPM voxel data: ' fmri_parm.spm_file]);

spmx = SPM.XYZmm'; % SPM point coordinate [x, y, z] (mm)
spmz = SPM.Z;      % t (z) -value


%  check MNI or SPM coordinate
if isfield(SPM,'coord_type') 
	if strcmp(SPM.coord_type, coord_const.COORDINATE_MNI_RIGHT_MM)
		%  load MNI coordinate
		V = vb_load_cortex(brain_file,'MNI');
	elseif strcmp(SPM.coord_type, coord_const.COORDINATE_SPM_RIGHT_MM)
		%  load SPM coordinate
		V = vb_load_cortex(brain_file);
	else
		error('SPM.coord_type is not correct')
	end
else
	error('SPM.coord_type is not set')
end
%
% Flip left-right (X-axis) in SPM coordinate for old SPM voxel file
% Left-hand to Right-hand  coordinate 
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%if (~isfield(SPM,'vbmeg_RAS') | ~SPM.vbmeg_RAS) ...
%      & (~isfield(SPM,'coord_type') ...
%      | ~strcmp(SPM.coord_type,coord_const.COORDINATE_SPM_RIGHT_MM)), 
%  spmx(:,1) = -spmx(:,1);
%end;

clear pack;
%
% Select the dipoles around the active area in fMRI
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
tic;
vb_disp('--- Select dipoles ');

% Search active vertex in 3D-sphere with Rmax [mm]
[xxix ,xxP3] = vb_spm_to_cortex(spmx, spmz, V*1000, fmri_parm);

spm_parm.SPM_Radius   = fmri_parm.SPM_Radius;
spm_parm.SPM_zstep    = fmri_parm.SPM_zstep;
spm_parm.Gauss_radius = fmri_parm.Gauss_radius;
spm_parm.Gauss_max    = fmri_parm.Gauss_max;

% xxix : active dipole index
% xxP3 : fMRI activity
% Act information
tmp             = zeros(size(V,1),1);
tmp(xxix)       = xxP3;
fMRINew.xxP     = tmp;
fMRINew.key     = fmri_parm.fmri_id;
fMRINew.comment = spm_parm;

% Area information
AreaNew.Iextract = xxix;
AreaNew.key = fmri_parm.area_id;

vb_disp(sprintf('%f[sec]',toc));
  
clear spmx3 spmx spmz;

return; 
