function [Zact ,Jinfo, bayes_parm, vb_parm, MEGinfo] ...
          = vbmeg_feature_calc_z(proj_root, curr_parm);
% Current reconstruction using Bayesian inverse filter
%
% --- Syntax
%  [Zact ,Jinfo, bayes_parm, vb_parm, MEGinfo] ...
%          = vbmeg_feature_calc_z(proj_root, curr_parm);
% --- Input
% curr_parm.filterfile : VB filter file (.vbfilt.mat)
% curr_parm.currfile   : output file for estimated current feature (.curr.mat)
% curr_parm.megfile    : MEG/EEG data file
% 
% curr_parm.freq_range : band frequency for feature extraction [Nband x 2]
%          .freq_range(n ,:)  = [freq_low freq_high] Hz (for n-th band)
%   For each time window specified by Tperiod & Tnext,
%   sum of bandpass frequency component power (square of abs) are calculated.
% The following parameters are independent of bayes_parm used in VB filter
% If there are multiple time windows in VB filter,
%    average filter is calculated before current calculation
% curr_parm.twin_meg:  Time range of meg/eeg data used for analysis
%   e.g. twin_meg = [1 600]  
% curr_parm.Tperiod :  Length of the time windows. 
% curr_parm.Tnext   :  Moving steps of the time window. 
%   e.g. Tperiod = 100, Tnext = 50  
%               --> [1 100; 51 150; 101 200;...]
% --- Optional Input
% curr_parm.trial_average = ON : average current over all sessions 
%                         = [OFF]  : current for each session
%
% curr_parm.ix_area : Position index to calculate estimated current
%           If 'ix_area' is empty or not given, 
%              currents in the active region are calculated
% ---Output
% Zact    : active current feature
%
% Zact(:, n + Nband*(j-1), :) : n-th bandpass component at j-th time window
% Zact(Nact,Nfeature)         for trial_average = ON 
% Zact(Nact,Nfeature,Ntrial)  for trial_average = OFF
%   Nact     : # of active vertex, 
%   Nfeature : # of feature = Nwindow * Nband
%   Ntrial   : # of trials in all session]
%
% Jinfo.ix_act  : Vertex index corresponding to active current
%
% 2008/12/2 M. Sato
% 2010/04/22 (Sako) substituted BMI_load_eeginfo with vb_load_measurement_info
% ---
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if ~isempty(proj_root)
	filterfile = fullfile(proj_root, curr_parm.filterfile);
else
	filterfile = curr_parm.filterfile;
end  

%%%%%% load VBMEG estimated result
load(filterfile, 'VBfilt','Jinfo','bayes_parm','vb_parm');

%%%%%% check parameter of 'curr_parm'
% Values of 'curr_parm' fields dominates over 
%   those of 'bayes_parm' in bayesfile
[bayes_parm, ix_area, trial_average, tsubsamp, overlap_mode, verbose] = ...
			check_arg(bayes_parm, curr_parm);
			
if ~isempty(proj_root);
  bayes_parm_abs = vb_parm_absolute_path(proj_root, bayes_parm);
else
  bayes_parm_abs = bayes_parm;
end

%%%%%% MEG data preparation
% B      : MEG data
[B, Ntrial, Nch, Tsample, Twindow] = ...
	vb_megdata_preparation(bayes_parm_abs);

MEGinfo = vb_load_measurement_info(bayes_parm_abs.megfile{1});

% sampling frequency
Fs = MEGinfo.SampleFrequency;

if ~isfield(curr_parm,'freq_range'), 
	freq_range = [5 9; 11 16; 21 26]
else
	freq_range = curr_parm.freq_range; 
end;
if ~isfield(curr_parm,'fft_window'), 
	fft_window = 0;
else
	fft_window = curr_parm.fft_window;
end

% Number of session
Nsession = length(B);	 
fprintf('Number of session: %d\n', Nsession);

% Temporal window
Nwindow = size(Twindow,1);
Tindex  = cell(Nwindow,1);

for j=1:Nwindow
	Tindex{j} = Twindow(j,1):Twindow(j,2);
end

% FFT point number
NFFT = 0;
for j=1:Nwindow
	Nt   = length(Tindex{j});
	nfft = 2^nextpow2(Nt);
	if NFFT < nfft, NFFT=nfft; end;
end

% frequency for FFT
df=Fs/NFFT;
freq=0:df:df*(NFFT/2);

if max(freq_range(:)) > Fs/2, 
	error('max freq is larger than Nyquist freq')
end

% band frequency
[Nband ,D] = size(freq_range);
if D ~=2, error('freq_range should be [Nband x 2]'); end;

% index list for band frequency
flist  = zeros(Nband ,2);
for n=1:Nband*2
	[tmp, ix] = min(abs(freq - freq_range(n)));
	flist(n) = ix;
end

fxlist = [];
ixlist = [];
nxlist = [];
Nlast  = 0;
%Ilist  = zeros(Nband,2);

for n=1:Nband
	Nlist  = flist(n,2) - flist(n,1) + 1;
	Nstart = Nlast + 1;
	Nlast  = Nlast + Nlist;
	
	fxlist = [fxlist, flist(n,1):flist(n,2)];
	ixlist = [ixlist, Nstart:Nlast];
	nxlist = [nxlist, repmat(n, [1, Nlist])];
	
%	Ilist(n,:) = [Nstart Nlast];
end

% Sparse matrix to get sum of frequency component in each band
Nall = length(ixlist);
Wfreq = sparse( ixlist , nxlist , 1 , Nall, Nband) ;
% Bf = fft( vb_repmultiply(B, hanning(Nt)') ,NFFT, 2);
% Zf = Bf(:,fxlist) ).^2 * Wfreq 
% Zf(:,n) : n-th band power summed over [freq_range(n,1) freq_range(n,2)]

% VBfilt.KW = VB inverse filter  (Njact,Nch,Nwindow,Ntask)
[Njact,Nch,Nw,Ntask] = size(VBfilt.KW);

if Nw==1 && Ntask==1, 
	KW = VBfilt.KW(:,:,1,1) ;
else
	fprintf('(Nwindow=%d ,Ntask=%d)\n',Nw,Ntask)
	fprintf('Take average of VB filters\n')
	KW = squeeze(mean( mean( VBfilt.KW,3) ,4));
end

fprintf('Active vertex number = %d\n',Njact)
fprintf('Active sensor number = %d\n',Nch)
fprintf('Number of time window= %d\n',Nwindow)
fprintf('Number of bandpass components= %d\n',Nband)
fprintf('Number of FFT points= %d\n',NFFT)

%
% --- Set Current Info
%

%Jinfo.Pretrigger = MEGinfo.Pretrigger;
%Jinfo.Wact = Wact;

%Lact      = Jinfo.Lact;
%ix_act    = Jinfo.ix_act;
%ix_act_ex = Jinfo.ix_act_ex;
%
%Jinfo = [];
%Jinfo.Lact       = Lact;
%Jinfo.ix_act     = ix_act;
%Jinfo.ix_act_ex  = ix_act_ex;

VBfreq.KW         = KW;
VBfreq.fxlist     = fxlist;
VBfreq.Wfreq      = Wfreq;
VBfreq.NFFT       = NFFT;
VBfreq.Twindow    = Twindow;
VBfreq.freq_range = freq_range;

Jinfo.VBfreq     = VBfreq;

Jinfo.Nwindow    = Nwindow;
Jinfo.SampleFreq = Fs;

% MEG time window index
Tstart  = bayes_parm.twin_meg(1);
Tend    = bayes_parm.twin_meg(2);
if isempty(tsubsamp)
	Jinfo.Tsample = Tstart:Tend;
else
	Jinfo.Tsample = tsubsamp + Tstart - 1;
end

%%%%%% Time window loop
% MEG for each trial	
err  = 0;
nall = 0;
Zact = zeros( Njact, Nwindow*Nband, sum(Ntrial));

for n=1:Nsession
	Ntry = Ntrial(n);
	
	if Ntry == 0, continue; end;

	Bs = B{n};
	
	for m=1:Ntry
		nall = nall + 1;
		for j=1:Nwindow
			% Subsampling time index
			Tid = Tindex{j};	% subsampled time index
			Nt  = length(Tid);
			
			if Nt == 0, continue; end;
			
			% FFT of MEG/EEG data with Hanning window
			Bt = Bs(:,Tid,m);
			
			if fft_window==1,
				Bt = vb_repmultiply(Bt, hanning(Nt)');
			end
			
			Bf = fft( Bt ,NFFT, 2);
			
			% Current reconstruction in frequency space
			Zf = abs( KW * Bf(:,fxlist) ).^2 ;
			
			% Frequency band component
			Zact(:, (1:Nband) + (j-1)*Nband, nall) = Zf * Wfreq;
			
%			% Debug
%			Zb  = Zf * Wfreq;
%			Zbb = zeros(Njact,Nband);
%			
%			for k=1:Nband
%				Zbb(:,k) = sum(Zf(:,Ilist(k,1):Ilist(k,2)) ,2);
%			end
%			
%			err = err + sum(sum(abs(Zb - Zbb)))/sum(sum(abs(Zbb)));
		end % Timewindow loop
	end; % Trial loop
end

if trial_average == ON
	Zact = mean(Zact,3);
end

Jinfo.Ntrial   = Ntrial;
Jinfo.Nsession = Nsession;

if verbose==1,
	fprintf('\n')
end

%err

% ix_act : Vertex index corresponding to active current Zact
% ix_act_ex : Vertex index corresponding to active current Jact
% Wact   : Spatial smoothing matrix of focal window
% Jact   = Wact * Zact

return

%%%% ---------------
function [bayes_parm,ix_area,trial_average,tsubsamp,overlap_mode,verbose]= ...
			check_arg(bayes_parm,curr_parm)

if isfield(curr_parm,'basisfile'), 
	bayes_parm.basisfile = curr_parm.basisfile;
end;
if isfield(curr_parm,'megfile'), 
	bayes_parm.megfile   = curr_parm.megfile  ;
end;
if isfield(curr_parm,'twin_meg'), 
	bayes_parm.twin_meg  = curr_parm.twin_meg ;
end;
if isfield(curr_parm,'Tperiod'), 
	bayes_parm.Tperiod   = curr_parm.Tperiod  ;
end;
if isfield(curr_parm,'Tnext'), 
	bayes_parm.Tnext     = curr_parm.Tnext    ;
end;

if ~isfield(curr_parm,'trial_average'), 
	trial_average = ON; 
else
	trial_average = curr_parm.trial_average; 
end;

bayes_parm.trial_average = trial_average;

if ~isfield(curr_parm,'ix_area'),  
	ix_area = []; 
else
	ix_area = curr_parm.ix_area; 
end;
if ~isfield(curr_parm,'tsubsamp'), 
	tsubsamp = []; 
else
	tsubsamp = curr_parm.tsubsamp; 
end;
if ~isfield(curr_parm,'overlap_mode'),     
	overlap_mode = 0; 
else
	overlap_mode = curr_parm.overlap_mode; 
end;
if ~isfield(curr_parm,'verbose'),     
	verbose = 1; 
else
	verbose = curr_parm.verbose; 
end;
