function [trial, id_list] = vb_info_get_trial_data(info, trial_id, field_name)
% return trial data that you want from info.Trial struct
% [usage]
%   [trial, id_list] = vb_info_get_trial_data(info, trial_id, field_name)
% [input]
%         info : <required> <<struct>> MEGinfo or EEGinfo
%              :  which must have Trial field
%     trial_id : <optional> [Ntrial_req x 1]
%              :  if this is empty, return all trial data
%   field_name : <optional> <<string>> you can get only a field data you want
% [output]
%     trial : required data [Ntrial_req x 1]
%   id_list : trial ID list of returned data
% [note]
%   MEGinfo.Trial [Ntrial x 1 struct]
%     .number [x1] trial number when this data was imported
%     .sample [Nsample x 1] list of sample numbers
%     .Active [x1] <<boolean>>
% [history]
%   2009-07-13 (Sako) initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('info', 'var'), info = []; end
if ~exist('trial_id', 'var'), trial_id = []; end
if ~exist('field_name', 'var'), field_name = ''; end
[info, trial_id, field_name] = ...
  inner_check_arguments(info, trial_id, field_name);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
if isempty(info)
  trial = [];
  id_list = [];
  return;
end

n_trial = length(info.Trial);

if isempty(trial_id)
  % --- data of all the channels
  id_list = zeros(n_trial,1);
  for i_tr = 1:n_trial
    id_list(i_tr) = info.Trial(i_tr).number;
  end

  if isempty(field_name)
    trial = info.Trial;
  elseif strcmp(field_name, 'number') || strcmp(field_name, 'Active')
    exec_str = sprintf('trial = [info.Trial.%s];', field_name);
    eval(exec_str);

  elseif strcmp(field_name, 'sample')
    for i_tr = 1:n_trial
      trial(i_tr,:) = info.Trial(i_tr).sample;
    end

  else
    warning('(%s) unknown field of Trial : %s - return all\n', ...
      mfilename, field_name);
    trial = info.Trial;
  end

  % --- arrange to [N x 1]
  trial = vb_util_arrange_list(trial, 0);
  id_list = vb_util_arrange_list(id_list, 0);
  return;
end

% --- the case that trial id is specified
n_trial_req = length(trial_id);

n_return = 0;

for i_req = 1:n_trial_req
  
  cur_id = trial_id(i_req);
  
  for i_trial = 1:n_trial
    
    cur_trial = info.Trial(i_trial);

    if cur_id == cur_trial.number
      n_return = n_return + 1;
      id_list(n_return) = cur_id;

      if isempty(field_name)
        trial(n_return) = cur_trial;
      else
        exec_str = sprintf('trial(%d) = cur_trial.%s', n_return, field_name);
        eval(exec_str);
      end
      break;
    end
  end
end

% --- arrange to [N x 1]
trial = vb_util_arrange_list(trial, 0);
id_list = vb_util_arrange_list(id_list, 0);
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [info, trial_id, field_name] = ...
  inner_check_arguments(info, trial_id, field_name)
func_ = mfilename;

if isempty(info)
  error('(%s) info is a required parameter', func_);
end

if ~isfield(info, 'Trial')
  info = [];
  warning('(%s) This info does not have ''Trial'' field\n', func_);
  return;
end

if isempty(trial_id)
  % require no action
end

if isempty(field_name)
  % require no action
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
