function [data, loadinfo] = vb_meginfo_read_data_file( ...
  meginfo, ch_list, sample_list, datatype, tr_list, root_dir)
% read data from data file of channel or trial
% [usage]
%   [data, loadinfo] = vb_meginfo_read_data_file( ...
%     meginfo, ch_list, sample_list, datatype, tr_list, root_dir)
% [input]
%    meginfo : <required> <<struct>> MEG information
%            :  (required fields are as follows)
%            :   .saveman
%    ch_list : <optional> list of channel label to read []
%            :  e.g. [1 2 3 4]
%            :  if this is empty, all channels are read
%sample_list : <optional>
%            :  sample index array of each trial
%            :  [from, to]
%            :  if this is empty, all samples are read as single trial
%            :  if data type is "continue", [Ntrial x 2]
%            :  if data type is "evoked raw", [1 x 2]
%   datatype : <optional> data type to read [0]
%            :  this is used when ch_list is not specified
%            :  if ch_list is valid, this is ignored
%            :  0) ALL (MEG + EXTRA)
%            :  1) MEG
%            :  2) EXTRA
%    tr_list : <optional> trial index list [N_trial_new x 1]
%            :  if tr_list is valid and this is empty, all trials are selected
%   root_dir : <optional> root directory of data file ['.']
%            :  normally it is path of megfile
% [output]
%       data : read data
%            :  [Nchannel x Nsample x Ntrial]
%   loadinfo : loading information (undefined yet)
% [note]
%   Continuous Data --> be able to be extracted by channel and sample
%        Epoch Data --> be able to be extracted by channel and trial
% [history]
%   2007-06-12 (Sako) initial version
%   2007-08-22 (Sako) added an argument 'root_dir'
%   2008-02-19 (Sako) renewed
%   2008-04-24 (Sako) numbered Trial(i).number sequentially - continuous data
%   2009-07-14 (Sako) supported new format of trial data (each channel)
%   2009-07-28 (Sako) supported 'BASIC' format
%   2010-07-01 (Sako) modified help regarding 'tr_list'
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('meginfo', 'var'), meginfo = []; end
if ~exist('ch_list', 'var'), ch_list = []; end
if ~exist('sample_list', 'var'), sample_list = []; end
if ~exist('datatype', 'var'), datatype = []; end
if ~exist('tr_list', 'var'), tr_list = []; end
if ~exist('root_dir', 'var'), root_dir = []; end
[meginfo, ch_list, sample_list, tr_list, root_dir] = ...
  inner_check_arguments(meginfo, ...
    ch_list, sample_list, datatype, tr_list, root_dir);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
loadinfo = [];

n_sample = vb_meginfo_get_sample_number(meginfo);
n_trial  = vb_info_get_Nrepeat(meginfo);

if n_trial == 1
  trial_cond = 'single';
  % --- not be extracted by trial
  tr_list = [];
else
  trial_cond = 'plural';
  if size(sample_list, 1) > 1
    % --- something wrong
    fprintf('(%s) loadspec.Trigger may be wrong\n', mfilename);
    sample_list = sample_list(1,:);
  end
end

% --- <<temporary struct>> SAVEINFO --------- %
%   .dir  : directory
%   .prec : precision string like 'float32'
% ------------------------------------------- %

% changed saveman spec
saveinfo.dir = [root_dir filesep vb_saveman_get_dir(meginfo.saveman)];
saveinfo.trial_cond = trial_cond;
saveinfo.prec = inner_determine_precision(meginfo);
saveinfo.n_trial = n_trial;

data = inner_read_data_file(saveinfo, n_sample, ch_list, sample_list, tr_list);
% data = inner_read_data_file(saveinfo, n_sample, ch_list, sample_list,
% tr_info);
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [meginfo, ch_list, sample_list, tr_list, root_dir] = ...
  inner_check_arguments(meginfo, ...
    ch_list, sample_list, datatype, tr_list, root_dir)

func_ = mfilename;

if isempty(meginfo)
  error('(%s)meginfo is a required parameter', func_);
end

if isempty(ch_list)
  if isempty(datatype)
    datatype = 0;
  end

  if datatype == 1
    ch_list = vb_meginfo_get_channel_label_meg(meginfo);
  elseif datatype == 2
    ch_list = vb_meginfo_get_channel_label_extra(meginfo);
  else
    meg_ch_list = vb_meginfo_get_channel_label_meg(meginfo);
    ex_ch_list = vb_meginfo_get_channel_label_extra(meginfo);
    ch_list = [meg_ch_list; ex_ch_list];
  end
end

if isempty(sample_list)
  sample_list = [1,vb_meginfo_get_sample_number(meginfo)];
end

if isempty(tr_list)
  n_trial = vb_info_get_trial_number(meginfo);
  tr_list = (1:n_trial);
end

if isempty(root_dir)
  % default directory is current
  root_dir = '.';
end
return;
%
% --- end of inner_check_arguments()


% --- inner_read_data_file()
%
% function [loaded_data] = inner_read_data_file( ...
%   saveinfo, n_sample, ch_list, sample_list, tr_info)
function [loaded_data] = inner_read_data_file( ...
  saveinfo, n_sample, ch_list, sample_list, tr_list)

const = vb_define_yokogawa;
func_ = mfilename;

% decompose SAVEINFO
savedir = saveinfo.dir;
trial_cond = saveinfo.trial_cond;
PRECISION = saveinfo.prec;
n_trial = saveinfo.n_trial;

n_channel_new = length(ch_list);

n_sample_new = sample_list(1,2)-sample_list(1,1) + 1;

switch trial_cond
  case 'single'
    % --- sample_list [N_new_trial x 2]
    n_trial_new = size(sample_list,1);

  case 'plural'
    % --- tr_list is a trial number list to extract [N_new_trial x 1]
    n_trial_new = length(tr_list);
end

% --- file extension
ext = const.EXT_CHANNEL_BIN;

% --- allocate buffer
loaded_data = zeros(n_channel_new, n_sample_new, n_trial_new);

for i_ch = 1:n_channel_new
  ch_label = ch_list{i_ch};
    
  fpath = sprintf('%s%s%s.%s',savedir, filesep, ch_label, ext);
  fid = fopen(fpath, 'rb');
  if fid == -1
    warning('(%s)cannot open %s', func_, fpath);
    continue;
  end
  
  % --- read data
  ch_data = fread(fid, inf, PRECISION);
  fclose(fid);    

  switch trial_cond
    case 'single'
      % --- The data which trial size is one like 'Continuous_Raw' can be
      %     loaded by channel or sample, but not by trial
      %
      % --- sample_list is expected [N_new_trial x 2]
      % ------- ch_data is expected [n_sample * 1]

      % --- chop data by trial
      for i_trial = 1:n_trial_new
        loaded_data(i_ch,:,i_trial) = ...
          ch_data(sample_list(i_trial,1):sample_list(i_trial,2),:);
      end

    case 'plural'
      % --- The data which trial size if over 1 like 'Evoked_Raw' can be loaded
      %     by channel or trial, but not by sample
      % --- Sample_from must be 1 and sample_to must be the length of sample
      
      % --- ch_data is expected [n_sample * n_trial]
      org_ch_data = reshape(ch_data, n_sample, n_trial);
      
      % --- sample_list expected to be [1 x 2]
      sample_from = sample_list(1,1);
      sample_to   = sample_list(1,2);
      sample_idx = sample_from:sample_to;
      loaded_data(i_ch,:,:) = org_ch_data(sample_idx, tr_list);
  end
end
return;
%
% --- end of inner_read_data_file()


% --- inner_determine_precision(meginfo)
%
function [prec] = inner_determine_precision(meginfo)
func_ = mfilename;

if isempty(meginfo)
  error('(%s)cannot determine precision wihout meginfo', func_);
end
prec = vb_meginfo_get_precision(meginfo);

if isempty(prec)
  % if value is not be set, give default value (defined in vb_define_yokogawa.m)
  const = vb_define_yokogawa;
  prec = const.STANDARD_BIN_DATA_TYPE;
end
%
% --- end of inner_determine_precision()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
