function [idx, result] = vb_meginfo_get_channel_index_extra(meg_info, ...
  ex_type, ch_name, active_swt, ref_swt)
% <getter> return index list of extra channel (except for reference channel)
% [usage]
%   [idx, result] = vb_meginfo_get_channel_index_extra(meg_info, ...
%     ex_type, ch_name, active_swt, ref_swt)
% [input]
%     meg_info : <required> <<struct>> MEGinfo
%              :  which is expected to have ExtraChannelInfo
%      ex_type : <optional> extra channel type [1] | 0 | -1 | -2 | -3 | -4
%              :  - YOKOGAWA definition
%              :  [1]) all channel
%              :   0 ) Null Channel
%              :  -1 ) Trigger Channel
%              :  -2 ) Eeg Channel
%              :  -3 ) Ecg Channel
%              :  -4 ) Etc Channel
%              :  this has priority over ch_name
%      ch_name : <optional> {Nch x 1 cell} channel name list ['']
%              :  if this is not empty and ex_type is empty, 
%              :  return specified index of original
%   active_swt : <optional> <<boolean>> active filter switch [false]
%              :   true) filter active channels
%              :  false) all the channels
%      ref_swt : <optinal> <<boolean>> switch to include reference channels
%              :     true ) include
%              :   [false]) not include
% [output]
%          idx : index list of channels [N x 1]
%       result : <<integer>> error code
%              :  0) success
%              :  1) error - bad meginfo
% [note]
%   for only YOKOGAWA now
%   if device is not YOKOGAWA, return index of all channel
% [history]
%   2007-06-29 (Sako) initial version
%   2008-02-21 (Sako) modified thoroughly according to new specification
%   2008-04-18 (Sako) added active_swt and result
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('meg_info', 'var'), meg_info = []; end
if ~exist('ex_type', 'var'), ex_type = []; end
if ~exist('ch_name', 'var'), ch_name = ''; end
if ~exist('active_swt', 'var'), active_swt = []; end
if ~exist('ref_swt', 'var'), ref_swt = []; end
[meg_info, ex_type, ch_name, active_swt, ref_swt, result] = ...
  inner_check_arguments(meg_info, ex_type, ch_name, active_swt, ref_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
idx = [];

if result ~= 0
  return;
end
  
% include reference channels
all_index = vb_meginfo_get_channel_id_extra(meg_info);
all_types = vb_meginfo_get_channel_type_extra(meg_info);
active_list = vb_meginfo_get_channel_active_extra(meg_info, active_swt);

if isempty(ex_type) && isempty(ch_name)
  criterion = bitand((active_list == 1),(all_types <= 0));
  idx = all_index(criterion == 1);
  return;
end

if ~isempty(ex_type)
  if ex_type == 1
    criterion = bitand((active_list == 1),(all_types <= 0));
    idx = all_index(criterion == 1);
  else
    criterion = bitand((active_list == 1),(all_types == ex_type));
    idx = all_index(criterion == 1);
  end
else
  % --- select by channel names
  % ----- get labels of all the channels
  ch_label = vb_meginfo_get_channel_label_extra(meg_info, false, ref_swt);
  ch_idx = vb_util_get_index(ch_label, ch_name);
  a_list = find(active_list == 1);
  alist_idx = vb_util_get_index(a_list, ch_idx);
  idx = a_list(alist_idx);
end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [meg_info, ex_type, ch_name, active_swt, ref_swt, result] = ...
  inner_check_arguments(meg_info, ex_type, ch_name, active_swt, ref_swt)
func_ = mfilename;
result = 0;

if isempty(meg_info)
  fprintf('(%s)meg_info is a required parameter\n', func_);
  result = 1;
  return;
end

if ~isfield(meg_info, 'ExtraChannelInfo')
  fprintf('(%s)meg_info does not have ExtraChannelInfo field\n', func_);
  result = 1;
  return;
end

if isempty(ex_type)
  if isempty(ch_name)
    ex_type = 1; % all channel
  end
end

if ~strcmp('YOKOGAWA', vb_meginfo_get_device(meg_info))
  fprintf('(%s)return all the extra channels because of not YOKOGAWA\n', ...
    func_);
  ex_type = 1;
end

% priority
if ~isempty(ch_name) && ~iscell(ch_name)
    ch_name = {ch_name};
end

if isempty(active_swt)
  active_swt = false;
end

if isempty(ref_swt)
  ref_swt = false;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
