function	[data, fs] = vb_filter_data(data,parm)
% Bias correction, Filtering, Down sampling and Common reference
%  [data] = vb_filter_data(data,parm)
% --- Input
%  data : input signal [Nchannel x Tsample x Ntrial]
% --- preprocess parameter
% parm
%  .freq        : Sampling frequency [Hz]
%  --- optional fields
%  .bias_flg    : Bias correction flag  
%               = 0/1/2 : OFF/Bias/Linear) : Bias correction by all data
%               = [t1 t2]: Bias correction by time window [t1 t2] (msec)
%                 time is specified by [msec] from the begining of data
%  .cutfreq     : Cutoff frequency [Hz] for filtering
%               = [f1]    for 'low','high'
%               = [f1 f2] for 'band','stop'
%  .fsamp       : Down sampling frequency [Hz]
%                 lowpass or bandpass below Nyquist frequency is necessarily
%  .common_flg  : Common reference flag (= 1/0 : ON/OFF)
% --- If these fields are not given or empty,
%        corresponding process are not applied
% --- Type of filter specification
%  .filter_type : 'low','high','band','stop'
%     if this field is not given or empty, filtering is not done
%  .impulse_response : 
%                   = 1: Finite impluse filter (eegfilt)
%                   = 2: Butterworth filter
% --- following fields are Butterworth filter only
%  .order : filter order of Butterworth filter
%  .online = 1: online filter with time delay
%          = 0: 'filtfilt' is used for filter with no time delay 
%        if this is not given, 'filtfilt' is used
%
%  2012-9-19 Masa-aki Sato
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

[Nch ,T, Ntr] = size(data);

fs = parm.freq;

% Common reference
if	isfield(parm,'common_flg') && ~isempty(parm.common_flg) ...
	&& parm.common_flg==1
	
	bias = mean(data,1);
	data = data - repmat(bias, [Nch 1 1]);
end

% Bias correction
if  isfield(parm,'bias_flg') && ~isempty(parm.bias_flg) 
	if length(parm.bias_flg) == 2 ,
		% Bias correction by time window [t1 t2]
		twin = round(parm.bias_flg * fs * 0.001);
		twin = min(max(twin , 1) , T);
		bias = mean(data(:,twin(1):twin(2),:) , 2);
		data = data - repmat(bias, [1 T 1]);
	elseif parm.bias_flg > 0
		% Bias correction by all data
		for m=1:Ntr
			data(:,:,m) = vb_channel_bias_remove(data(:,:,m),parm.bias_flg);
		end
	end
end

% Filtering
if	isfield(parm,'cutfreq') && ~isempty(parm.cutfreq) && ...
	isfield(parm,'filter_type') && ~isempty(parm.filter_type),
	
	% filter_type
	filter_type = parm.filter_type;
	% cutoff frequency
	fcut  = parm.cutfreq; 
	Nfreq = length(fcut);
	
	if Nfreq > 2, error('Cutoff frequency is wrong'); end;
	
	if parm.impulse_response==1
		% FIR Lowpass filter
		switch	filter_type
		case	'low'
			data = eegfilt(data, fs, 0, fcut );
		case	'high'
			data = eegfilt(data, fs, fcut, 0 );
		case	'band'
			data = eegfilt(data, fs, fcut(1), fcut(2) );
		case	'stop'
			Norder = fix(3*(fs/(fcut(2)-fcut(1)))); 
		    [B, A] = fir1(Norder, fcut/(fs/2), 'stop');
			for m=1:Ntr
				for n=1:Nch
					data(n,:,m) = filtfilt(B,A, data(n,:,m) );
				end
			end
		end
	else
		% Butterworth filter
		Norder = parm.order; % filter order
		switch	filter_type
		case	'low'
			[B,A] = butter(Norder, fcut(1)/(fs/2) ,'low');
		case	'high'
			[B,A] = butter(Norder, fcut(1)/(fs/2) ,'high');
		case	'band'
			[B,A] = butter(Norder, fcut(1:2)/(fs/2) );
		case	'stop'
			[B,A] = butter(Norder, fcut(1:2)/(fs/2) , 'stop');
		end
		
		if isfield(parm,'online') && parm.online==1
			for m=1:Ntr
				for n=1:Nch
					data(n,:,m) = filter(B,A, data(n,:,m) );
				end
			end
		else
			for m=1:Ntr
				for n=1:Nch
					data(n,:,m) = filtfilt(B,A, data(n,:,m) );
				end
			end
		end
	end
end

% Down sampling
if isfield(parm,'fsamp') && ~isempty(parm.fsamp) && fs ~=parm.fsamp
	fsnew = parm.fsamp;
	%fprintf('Down sampling to %6.1f [Hz]\n',fsnew)
	
	data  = vb_convert_freq(data, fs, fsnew);
	fs = fsnew;
end
