function result_code = vb_job_eeg_brainamp(read_spec, verbose_swt)
% make EEG-MAT file from BrainAmp files.
%
% [usage]
%   result_code = vb_job_eeg_brainamp(read_spec, verbose_swt)
%
% [input]
%   read_spec  : <required> <<struct>> specification to read BrainAmp files.
%     .bamp_file    : <required> <<file>> brainamp file(.eeg) with valid path
%                                         see [note] below.
%     .pos_file     : <optional> <<file>> POS-MAT file with valid path
%     .device       : <optional> ['BRAINAMP'] device name
%     .output_dir   : <optional> output directory for EEG-MAT files ['.']
%     .eeg_file     : <optional> ['<date>.eeg.mat'] EEG-MAT file name
%     .bin_data_dir : <optional> currently not used.
%     .eeginfo_version : <optional> [] version of EEGinfo - unused yet
%
%   verbose_swt     : <optional> <<boolean>> [false]
%                   :   switch to output verbose message
% [output]
%   result_code : <integer>
%               :  1) no problem
% [note]
%   See also
%     vb_job_meg
%
%   * brain amp data consists of three files(.eeg/.vhdr/.vmrk). 
%     This function assumes that these files exist in a same directory
%     with the same name.
%
% [history]
%   2014-08-19 rhayashi initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)
tic;


%
% --- Previous check
%

% constants
vb_define_device;
func_ = mfilename;
result_code = 1; % no problem

if (nargin < 1 || 2 < nargin)
    error('(%s)Please check the usage of the function', func_);
end
if ~isfield(read_spec, 'bamp_file') || isempty(read_spec.bamp_file)
    error('(%s)cannot find bamp_file', func_);
end
if ~isfield(read_spec, 'pos_file')
    read_spec.pos_file = '';
end

if ~isfield(read_spec, 'device') || ...
   isempty(read_spec.device)
    read_spec.device = 'BRAINAMP';
end

%
% --- Main Procedure
%
if exist(read_spec.bamp_file, 'file') ~= 2
    error('%s)cannot find bamp_file: %s', func_, read_spec.bamp_file);
end

% Input/Output file
data_file   = read_spec.bamp_file;
pos_file    = read_spec.pos_file;
output_file = fullfile(read_spec.output_dir, read_spec.eeg_file);

[output_dir, f, e]  = vb_get_file_parts(output_file);
if exist(output_dir, 'dir') ~= 7
    vb_mkdir(output_dir);
end

% read header
sFile     = in_fopen_brainamp(data_file);

% get information
ch_list       = {sFile.channelmat.Channel.Name}';
Nch_all       = size(ch_list, 1); % This will contain the number of extra channel.
sample_freq   = sFile.prop.sfreq;
Nsamples      = sFile.header.nsamples;

% read data
sfid = fopen(sFile.filename, 'r', sFile.byteorder);
if sfid == -1, return; end
data2d = in_fread_brainamp(sFile, sfid); % ch x sample
fclose(sfid);


% Add marker(trigger) as channel data
for k=2:length(sFile.events)
    trig_start_list = sFile.events(k).samples;
    Ntrig           = length(trig_start_list);
    if Ntrig
        data2d = [data2d; zeros(1, Nsamples)];
        for j=1:Ntrig
            % trigger ON = 1
            data2d(end, trig_start_list(j)) = 1;
        end
        Nch_all = Nch_all + 1;
        ch_list{end+1} = ['EXT', num2str(k-1)];
    end
end

%
% --- read sensor location & make 
%
pos_to_data_ix   = [];  %  pos to data2d index
pos_valid_ix     = [];

if isempty(pos_file)
    % No POS-MAT file specified.
    pos_info.coord_type = '';
    pos_info.name       = {sFile.channelmat.Channel.Name}';
else
    pos_info = load(pos_file);
end

% search and create relationships.
% channel of measurement data vs pos(pos_ix, :)
for k=1:length(pos_info.name)
    channel = pos_info.name{k};
    ix = strmatch(channel, ch_list, 'exact');
    if isempty(ix)
        warning(['channel : ' channel '  not found in the measurement file.']);
    else
        pos_to_data_ix = [pos_to_data_ix; ix];
        pos_valid_ix = [pos_valid_ix; k];
    end
end

% find unmatched channel and treat them as extra channels.
ext_data_ix = setdiff([1:Nch_all]', pos_to_data_ix);
ext_ch_name = [];
if ~isempty(ext_data_ix)
    for k=1:length(ext_data_ix)
        ext_ch_name{k, 1} = ch_list{ext_data_ix(k)};
    end
end
ext_data = data2d(ext_data_ix, :);


%
% --- Formatting for VBMEG
%
if isfield(pos_info, 'pos')
    Coord = pos_info.pos(pos_valid_ix, :);
else
    Coord = [];
end

% Data info
s.Measurement = 'EEG';
s.EEGinfo.Measurement   = 'EEG';
s.EEGinfo.Device        = 'BRAINAMP';
s.EEGinfo.Nsample       = Nsamples;
s.EEGinfo.Nrepeat       = 1;
s.EEGinfo.ActiveTrial   = 1;
s.EEGinfo               = vb_eeginfo_set_pre_trigger(s.EEGinfo, 0);
s.EEGinfo.SampleFrequency = sample_freq;

% Data
s.EEGinfo.Trial.number = 1;
s.EEGinfo.Trial.sample = [1:Nsamples];
s.EEGinfo.Trial.Active = true;

% Channel info
s.EEGinfo.Nchannel      = length(pos_valid_ix);
s.EEGinfo.CoordType     = pos_info.coord_type;
s.EEGinfo.Coord         = Coord;
s.EEGinfo.ChannelID     = [1:s.EEGinfo.Nchannel]';
s.EEGinfo.ChannelName   = pos_info.name(pos_valid_ix);
s.EEGinfo.ActiveChannel = ones(s.EEGinfo.Nchannel, 1);

s.EEGinfo.ChannelInfo.Active = s.EEGinfo.ActiveChannel;
s.EEGinfo.ChannelInfo.Name   = s.EEGinfo.ChannelName;
s.EEGinfo.ChannelInfo.Type   = ones(s.EEGinfo.Nchannel, 1);
s.EEGinfo.ChannelInfo.ID     = s.EEGinfo.ChannelID;
s.EEGinfo.ChannelInfo.PhysicalUnit = repmat({'V'}, [s.EEGinfo.Nchannel, 1]);

Next_ch  = length(ext_data_ix);
if Next_ch
    s.EEGinfo.ExtraChannelInfo.Channel_active = ones(Next_ch, 1);
    s.EEGinfo.ExtraChannelInfo.Channel_name   = ext_ch_name;
    s.EEGinfo.ExtraChannelInfo.Channel_type   = ones(Next_ch, 1);
    s.EEGinfo.ExtraChannelInfo.Channel_id     = [s.EEGinfo.Nchannel+1:s.EEGinfo.Nchannel+Next_ch]';
end

% File
s.EEGinfo.File.BaseFile  = data_file;
s.EEGinfo.File.OutputDir = read_spec.output_dir;
s.EEGinfo.File.EEGFile   = read_spec.eeg_file;
s.EEGinfo.File.DataDir   = '';

% MRI_ID, Vcenter, Vradius, (device_info)TransInfo
s.EEGinfo = vb_info_add_posfile_info(s.EEGinfo, pos_file);

% EEG Data
s.eeg_data = data2d([pos_to_data_ix;ext_data_ix] , :);
s.EEGinfo.DataType = repmat({'float32'}, [Nch_all, 1]);


fprintf('--- now making EEG-MAT file(%s) ...\n', output_file);

% Save EEG-MAT
vb_save_struct(output_file, s);

fprintf(' done!\n');
fprintf('--- (%s) %.1f [sec]\n', func_, toc);
