function result_code = vb_job_eeg_biosemi(read_bdf_spec, verbose_swt)
% make EEG-MAT file from BDF file
%
% [usage]
%   result_code = vb_job_eeg_biosemi(read_bdf_spec, verbose_swt)
%
% [input]
%   read_bdf_spec   : <required> <<struct>> specification to read BDF file.
%                   : If this is empty ([]), pass through reading BDF.
%     .bdf_file     : <required> <<file>> BDF file with valid path
%     .pos_file     : <required> <<file>> POS-MAT file with valid path
%     .device       : <optional> ['BIOSEMI'] device name
%     .output_dir   : <optional> output directory for EEG-MAT files ['.']
%     .eeg_file     : <optional> ['<date>.eeg.mat'] EEG-MAT file name
%     .bin_data_dir : <optional> output directory for binary data files
%                   :  ['.\(name of eeg_file)_bin']
%                   :  this is relative path from output_dir
%  .eeginfo_version : <optional> [] version of EEGinfo - unused yet
%
%   verbose_swt     : <optional> <<boolean>> [false]
%                   :   switch to output verbose message
% [output]
%   result_code : <integer> be seldome used in practice
%               :  1) no problem
% [note]
%   See also
%     vb_job_meg
%
% [history]
%   2008-02-27 (Sako) initial version
%   2008-03-26 (Sako) modified data directories
%   2009-07-16 (Sako) added EEGinfo.ChannelInfo
%   2009-08-10 (Sako) modified the case that pos_file is not given
%   2009-12-25 (Sako) modified the unit of data when importing
%   2011-02-08 (Sako) adjusted file extension to modified definitions
%   2011-02-17 (Sako) changed defailt value of '.bin_data_dir'
%   2011-05-30 (Sako) modified according to the new data format
%   2011-08-18 (Sako) added setting a default value to the pretrigger field
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)
tic;

% ----- constants
vb_define_device;
func_ = mfilename;
result_code = 1; % no problem

% --- CHECK ARGUMENT --- %
if ~exist('read_bdf_spec', 'var'), read_bdf_spec = []; end;
if ~exist('verbose_swt', 'var'), verbose_swt = []; end;
[read_bdf_spec, VERBOSE] = inner_check_arguments(read_bdf_spec, verbose_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
% set local variables
[bdf_file, pos_file, ...
  figure_info, device, output_dir, eeg_file, data_dir, eeginfo_version] = ...
    inner_get_var_read_bdf_spec(read_bdf_spec);
  
% The eeg_file is defined from output_dir
% As data_dir is defined by the relative path from eeg_file,
% binary files will be created in 'output_dir/data_dir' directory in fact

% --- PREPARATION --- %
if ~isempty(figure_info)
  SensorPosition        = figure_info.SensorPosition;
  CoordType             = figure_info.CoordType;
  ChannelName_digitizer = figure_info.ChannelLabel;
else
  SensorPosition        = [];
  CoordType             = '';
  ChannelName_digitizer = [];
end

% ----- read BDF file
[ChannelName_hardware, RecordTime, Nch, SampleRate, data_type, ...
  org_physical_unit, DAT] = vb_bdffile_get_info(bdf_file);
  
% ----- match channels between digitizer label and hardware label
[eeg_name, eeg_idx, extra_name, extra_idx, coord_idx] = ...
    inner_match_channel_label(ChannelName_hardware, ChannelName_digitizer);
  
% ----- open files of each channel data
eeg_file_path = vb_get_file_parts(eeg_file);
act_data_dir = fullfile(eeg_file_path, data_dir);

all_eeg_ch = [eeg_name;extra_name];

[fid, data_type_list] = inner_open_files( ...
  all_eeg_ch, act_data_dir, data_type, VERBOSE);

% ----- make binary data files
ch_idx = [eeg_idx;extra_idx];
[new_physical_unit, new_eeg_idx, new_extra_idx] = ...
  inner_write_channel_data(fid, all_eeg_ch, ...
  DAT, RecordTime, data_type_list, org_physical_unit, ch_idx, VERBOSE);

% ----- close files
inner_close_files(fid, VERBOSE);

if VERBOSE
  fprintf('\n--- record time: %d [sec], sample rate:%d[Hz]\n', ...
    RecordTime, SampleRate);
end

% ----- make EEGinfo
EEGinfo = [];

% ------- MRI_ID, Vcenter, Vradius, (device_info)TransInfo
EEGinfo = vb_info_add_posfile_info(EEGinfo, pos_file);

EEGinfo.Measurement      = 'EEG';
EEGinfo.Device           = device;

% ------- channel
EEGinfo.ChannelID        = eeg_idx;
EEGinfo.ChannelName      = eeg_name;
EEGinfo.Nchannel         = length(eeg_name);

% ------- extra channel
n_extra_ch = length(extra_idx);
ExtraChannelInfo.Channel_active = ones(n_extra_ch,1);
ExtraChannelInfo.Channel_name   = extra_name;
ExtraChannelInfo.Channel_type   = ones(n_extra_ch,1);
ExtraChannelInfo.Channel_id     = extra_idx;

% ---------- physical unit - temporarily
ExtraChannelInfo.PhysicalUnit   = new_physical_unit(new_extra_idx);

EEGinfo.ExtraChannelInfo = ExtraChannelInfo;

EEGinfo.SampleFrequency  = SampleRate;
EEGinfo.Nsample          = RecordTime * SampleRate;
EEGinfo.DataType         = data_type_list;

EEGinfo = vb_eeginfo_set_sensor_position(EEGinfo, SensorPosition(coord_idx,:));
EEGinfo.CoordType        = CoordType;
% EEGinfo.Vcenter          = figure_info.Vcenter;
% EEGinfo.Vradius          = figure_info.Vradius;

% ----- file information
EEGinfo.File.BaseFile    = bdf_file;
EEGinfo.File.OutputDir   = output_dir;
[path,name,ext] = vb_get_file_parts(eeg_file);
EEGinfo.File.EEGFile     = [name ext];
EEGinfo.File.DataDir     = data_dir;

% ----- trial
Trial.number  = 1;
Trial.sample  = 1:EEGinfo.Nsample;
Trial.Active  = true;
EEGinfo.Trial = Trial;

% ----- set initial active channel flag
active_channel = ones(vb_info_get_Nchannel(EEGinfo), 1);
EEGinfo = vb_info_set_active_channel(EEGinfo, active_channel);
% --- EEGinfo.ChannelInfo.Active is set here

% ----- set initial active trial flag
% active_trial   = ones(vb_info_get_trial_number(EEGinfo), 1);
% EEGinfo = vb_info_set_active_trial(EEGinfo, active_trial);
EEGinfo = vb_info_adjust_trial(EEGinfo);

% ----- make ChannelInfo
EEGinfo.ChannelInfo.Type = ones(EEGinfo.Nchannel, 1);
EEGinfo.ChannelInfo.Name = EEGinfo.ChannelName;
EEGinfo.ChannelInfo.ID = EEGinfo.ChannelID;

% ---------- physical unit - temporarily
EEGinfo.ChannelInfo.PhysicalUnit = new_physical_unit(new_eeg_idx);

% ----- Pretrigger (default value)
EEGinfo = vb_eeginfo_set_pre_trigger(EEGinfo, 0);

% ----- DEVICE DEPENDENT DATA
EEGinfo = vb_eeginfo_set_reference(EEGinfo, EEGINFO_REFERENCE_COMMON);
EEGinfo = vb_eeginfo_set_rectime(EEGinfo, RecordTime);
EEGinfo = vb_eeginfo_set_header(EEGinfo, DAT.Head);
EEGinfo = vb_eeginfo_set_version(EEGinfo, eeginfo_version);

% ------- record history
history = ...
  sprintf('%s(''%s'', ''%s'', ''%s'')', func_, bdf_file, device, eeg_file);
EEGinfo = vb_eeginfo_add_history(EEGinfo, history);

% measurement information
fprintf('--- now making EEG-MAT file(%s) ...', eeg_file);
% if VERBOSE, fprintf('--- now making EEG-MAT file(%s) ...', eeg_file); end

Measurement = 'EEG';
vb_fsave(eeg_file, 'EEGinfo', 'Measurement');

fprintf(' done!\n');
fprintf('--- (%s) %.1f [sec]\n', func_, toc);
% if VERBOSE, fprintf(' done!\n'); end
% if VERBOSE, fprintf('--- (%s) %e [sec]\n', func_, toc); end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [read_bdf_spec, verbose_swt]= ...
  inner_check_arguments(read_bdf_spec, verbose_swt)
func_ = mfilename;
if isempty(read_bdf_spec)
  error('(%s)read_bdf_spec is a required parameter', func_);
end;

% ----- check fields
if ~isfield(read_bdf_spec, 'bdf_file')
  error('(%s)cannot find bdf_file in read_bdf_spec', func_);
end
if ~isfield(read_bdf_spec, 'pos_file')
  read_bdf_spec.pos_file = '';
end

if exist(read_bdf_spec.bdf_file, 'file') ~= 2
  error('(%s)cannot find bdf_file: %s', func_, read_bdf_spec.bdf_file);
end

if exist(read_bdf_spec.pos_file, 'file') ~= 2
  fprintf('--- (%s)read_bdf_spec.pos_file is empty\n', func_);
end

if ~isfield(read_bdf_spec, 'device') ...
    || isempty(read_bdf_spec.device)
  read_bdf_spec.device = 'BIOSEMI';
end

if ~isfield(read_bdf_spec, 'output_dir') ...
    || isempty(read_bdf_spec.output_dir)
  read_bdf_spec.output_dir = '';
end

if ~isfield(read_bdf_spec, 'eeg_file') ...
    || isempty(read_bdf_spec.eeg_file)
  read_bdf_spec.eeg_file = ...
    sprintf('%s.eeg.mat', datestr(now,'yyyymmdd_HHMMSS'));
end

% --- Also resolve directory of imported data
if ~isfield(read_bdf_spec, 'bin_data_dir') ...
    || isempty(read_bdf_spec.bin_data_dir)
  
  real_eeg_file = [read_bdf_spec.output_dir '/' read_bdf_spec.eeg_file];
  [out_dir, real_dir] = vb_device_make_data_dir(real_eeg_file);
  read_bdf_spec.bin_data_dir = out_dir;
else
  real_dir = [read_bdf_spec.output_dir '/' read_bdf_spec.bin_data_dir];
end

if exist(real_dir, 'dir') ~= 7
  vb_mkdir(real_dir);
end

if ~isfield(read_bdf_spec, 'eeginfo_version')
  read_bdf_spec.eeginfo_version = [];
end

if isempty(verbose_swt)
  verbose_swt = false;
end
return;
%
% --- end of inner_check_arguments()

% --- inner_get_var_read_bdf_spec()
%
function [bdf_file, pos_file, ...
  figure_info, device, output_dir, eeg_file, bin_data_dir, eeginfo_version] = ...
  inner_get_var_read_bdf_spec(read_bdf_spec)

% required fields
bdf_file = read_bdf_spec.bdf_file;
pos_file = read_bdf_spec.pos_file;
device = read_bdf_spec.device;
output_dir = read_bdf_spec.output_dir;
eeg_file = fullfile(output_dir, read_bdf_spec.eeg_file);
bin_data_dir = read_bdf_spec.bin_data_dir;
eeginfo_version = read_bdf_spec.eeginfo_version;

if ~isempty(pos_file)
  figure_info = vb_posfile_get_posinfo(pos_file);
else
  figure_info = [];
end
% ---=== figure_info ===--- %
% figure_info.SensorPosition;
% figure_info.CoordType;
% figure_info.ChannelLabel;
% figure_info.Vcenter;
% figure_info.Vradius;

return;
%
% --- end of inner_get_var_read_bdf_spec()


% --- inner_match_channel_label()
%
function [eeg_name, eeg_idx, extra_name, extra_idx, coord_idx] = ...
  inner_match_channel_label(ch_hw, ch_dg)
func_ = mfilename;

vb_define_device;

% --- if there is not pos-mat file
if isempty(ch_dg)
  % --- hard code extra channels
  ch_ex = {'Status'};

  % reset label size to [n_channel x 1]
  ch_hw = vb_util_arrange_list(ch_hw);
  ch_ex = vb_util_arrange_list(ch_ex);

  % --- get index of extra channels
  [idx_hw_ex] = vb_util_get_index(ch_hw, ch_ex);

  % ----- get labels of sensor channels and not sensor channels
  [ch_sensor, ch_sensor_idx, ch_extra, ch_extra_idx] = ...
    vb_util_clean_list(ch_hw, idx_hw_ex);

  % ----- set return values
  eeg_name = ch_sensor;
  eeg_idx  = ch_sensor_idx;
  extra_name = ch_extra;
  extra_idx  = ch_extra_idx;
  coord_idx  = [];

else
  % reset label size to [n_channel x 1]
  ch_hw = vb_util_arrange_list(ch_hw);
  ch_dg = vb_util_arrange_list(ch_dg);

  [idx_hw, idx_dg] = vb_util_get_index(ch_hw, ch_dg);

  % ----- get mask list
  [ex_ch_hw, ex_ch_hw_idx] = vb_util_clean_list(ch_hw, idx_hw);

  % ----- set return values
  eeg_name = ch_hw(idx_hw);
  eeg_idx  = idx_hw;
  extra_name = ex_ch_hw;
  extra_idx  = ex_ch_hw_idx;
  coord_idx  = idx_dg;
end
return;
%
% --- end of inner_match_channel_label()


% --- inner_open_files()
%
function [fid, data_type_list] = ...
  inner_open_files(channel_label, data_dir, data_type, VERBOSE)
func_ = mfilename;
vb_define_device;

% ----- make directory if necessary
if ~exist(data_dir, 'dir')
  if ~vb_mkdir(data_dir)
    error('(%s)cannot make directory: %s', func_, data_dir);
  end
end

Nch = size(channel_label,1);

DataFile = cell(Nch,1);
fid = zeros(Nch,1);
data_type_list = cell(Nch,1);

% support 'Status' channel
common_data_type = data_type;
status_data_type = STATUS_CH_DATA_TYPE;
status_channel_label = STATUS_CH_LABEL;

for ch = 1:Nch
  ch_name = channel_label{ch};
  
  DataFile{ch} = fullfile(data_dir, [ch_name, FILE_EXT_BIN_CH_EEG]);
  fid(ch) = fopen(DataFile{ch},'w');
  
  if fid(ch) < 0
    error('(%s)cannot open : %s', func_, DataFile{ch});
  end

  % data type
  if strcmp(ch_name, status_channel_label) == 1
    data_type_list{ch} = status_data_type;
  else
    data_type_list{ch} = common_data_type;
  end
  
  if VERBOSE, fprintf(' >>> open %s\n', DataFile{ch}); end
end
return;
%
% --- end of inner_open_files()

% --- inner_write_channel_data()
%
function [new_physical_unit, new_eeg_idx, new_extra_idx] = ...
  inner_write_channel_data(fid, ch_list, ...
  EDF_struct, RecordTime, data_type, org_physical_unit, ch_idx, VERBOSE)

do_mv2v = true;
vb_define_device;

if VERBOSE
  PLOT_EVERY = 30;
  fprintf('--- now reading and storing BDF data (plot every %d/%d [sec])', ...
    PLOT_EVERY, RecordTime);
end

Nch = length(fid);
ReadMode   = 0;

% --- these lists are expected to be the same order with ch_list
calib      = EDF_struct.Head.Cal(ch_idx);
org_offset = EDF_struct.Head.Off(ch_idx);
physical_unit = org_physical_unit(ch_idx);
new_physical_unit = cell(size(org_physical_unit));
new_eeg_idx = [];
new_extra_idx = [];

UNIT_MICRO_VOLT = 'uV';
UNIT_VOLT = 'V';
CF_MV2V = 10.^(-6);

n_ch = size(org_offset, 1);
offset = zeros(n_ch,1);

if do_mv2v
  % --- make new physical unit list and offset list
  for i_ch = 1:n_ch
    % --- physical_unit is the original list of physical unit
    if strcmp(physical_unit{i_ch}, UNIT_MICRO_VOLT)
      offset(i_ch) = org_offset(i_ch) * CF_MV2V;
      new_physical_unit{i_ch} = UNIT_VOLT;
    else
      offset(i_ch) = org_offset(i_ch);
      new_physical_unit{i_ch} = physical_unit{i_ch};
    end

    if strcmp(ch_list{i_ch}, STATUS_CH_LABEL)
      new_extra_idx = [new_extra_idx, i_ch];
    else
      new_eeg_idx = [new_eeg_idx, i_ch];
    end
  end
else
  offset = org_offset;
end

% --- arrange lists to [N_channel x ]
new_physical_unit = new_physical_unit';
new_extra_idx = new_extra_idx';
new_eeg_idx = new_eeg_idx';

for r = 1:RecordTime
  [DAT,DataBuf] = vb_readbdf(EDF_struct, r, ReadMode);
  
  for i_ch = 1:Nch
    cur_raw = DataBuf(:,i_ch);

    if strcmp(ch_list{i_ch}, STATUS_CH_LABEL)
      this_eeg = cur_raw;
    else
      if do_mv2v
        if strcmp(physical_unit(i_ch), UNIT_MICRO_VOLT)
          this_eeg = (cur_raw * CF_MV2V) * calib(i_ch) + offset(i_ch);
        else
          this_eeg = cur_raw * calib(i_ch) + offset(i_ch);
        end
      else
        this_eeg = cur_raw * calib(i_ch) + offset(i_ch);
      end
    end
    fwrite(fid(i_ch), this_eeg, data_type{i_ch}); 
  end
  if VERBOSE && rem(r,PLOT_EVERY) == 0, fprintf('.'); end
end
if VERBOSE, fprintf('\n'); end
return;
%
% --- end of inner_write_channel_data()

% --- inner_close_files()
%
function inner_close_files(fid, VERBOSE)
Nch = length(fid);
for ch=1:Nch
  fclose(fid(ch));
end
if VERBOSE, fprintf('--- closing files done ...\n'); end
return;
%
% --- end of inner_close_files()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
