function result = vb_eegfile_filter_ch_data(eeg_file, proc_spec, new_eeg_file)
% make new external channel data files after filtering
% [usage]
%   result = vb_eegfile_filter_ch_data(eeg_file, proc_spec, new_meg_file)
% [input]
%        eeg_file : <required> <<file>> EEG-MAT file (.eeg.mat)
%       proc_spec : <required> <<struct>> struct defined process specifications
%                 :       .parm   : <required> <<struct>>
%                 :               :  .filt_suffix
%                 :               :  .bias_flg
%                 :               :  .highpass
%                 :               :  .lowpass
%                 :               :  .highpass_online
%                 :               :  .lowpass_online
%                 :               :  .common_flg
%                 :               :  .fsamp
%                 :  .ch_type : channel type ['MEG'] - case-insensitive
%    new_eeg_file : <required> <<file>> filtered EEG-MAT file(.eeg.mat)
%                                  : This parameter must include the full extension.
%                                  :  ".eeg.mat"
%             :  .ch_type : channel type ['MEG'] - case-insensitive
%
% --- Meaning of preprocess parameter
%  .bias_flg    : Bias correction flag  (=0/1/2 : OFF/Bias/Linear)
%               = N > 5: Bias correction by first N sample
%  .highpass    : Highpass filter cutoff frequency [Hz]
%  .lowpass     : Lowpass  filter cutoff frequency [Hz]
%  .highpass_online > 0: online highpass filter is used (exponential)
%  .lowpass_online  > 0: online lowpass filter is used (order=lowpass_online)
%  .fsamp       : Down sampling frequency [Hz]
%  .common_flg  : Common reference flag (=1/0 : ON/OFF)
%   if these fields are empty, corresponding process is not done
% [output]
%      result : <<struct>> depends on current process
% [note]
%   @see vb_filter_raw_data.m
%   @see vb_megfile_filter_ch_data.m
% [history]
%   2009-07-17 (Sako) initial version
%   2009-08-20 (Sako) modified to complain if plural trial data
%   2016-11-25 (rhayashi) modified specification(new_eeg_file)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('eeg_file', 'var'), eeg_file = ''; end
if ~exist('proc_spec', 'var'), proc_spec = []; end
[eeg_file, proc_spec, new_eeg_file] = inner_check_arguments(eeg_file, proc_spec, new_eeg_file);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
result = [];
LINE_LEN = 15;

% --- re-store environment
[p_, f_, e_]   = vb_get_file_parts(new_eeg_file);

new_eeg_dir           = p_;           
new_eeg_filename      = [f_, e_];
bin_dir_name          = new_eeg_filename(1:strfind(new_eeg_filename, '.eeg.mat')-1);
proc_spec.new_bin_dir = ['./' bin_dir_name];

% --- process for each channel
if strcmp(proc_spec.ch_type, 'MEG') ~= 1
  error('(%s) not be supported channel_type : %s', ...
    mfilename, proc_spec.ch_type);
end

ch_labels = vb_eegfile_get_channel_label(eeg_file,[],1);
eeg_info  = vb_load_measurement_info(eeg_file);
ch_len    = size(ch_labels, 1);

% --- this function is for only single trial data
if vb_info_get_Nrepeat(eeg_info) > 1
  err_msg = ['this filtering function should be applied to ' ...
    'continuously measured data file without trial extraction.'];
  error('%s', err_msg);
end

% --- PRELIMINARILY PROCESS
proc_spec.parm.freq = vb_info_get_sampling_frequency(eeg_info);
    
% Create binary directory
new_bin_dir = fullfile(new_eeg_dir, proc_spec.new_bin_dir);
if exist(new_bin_dir, 'dir') ~= 7
  vb_mkdir(new_bin_dir);
end

precision = vb_eeginfo_get_datatype(eeg_info);

% --- PROCESS FOR EACH CHANNEL
%
for i_ch = 1:ch_len

  fprintf('[%s]', ch_labels{i_ch});
  if ~rem(i_ch, LINE_LEN), fprintf('\n'); end

  load_spec.ChannelName = {ch_labels{i_ch}};
  load_spec.ChannelSwitch = true; % ChannelName is to read
  load_spec.ChannelType = 'MEG';
  load_spec.ActiveChannel = 0; % all channels
  cur_data = vb_load_meg_data(eeg_file, load_spec);

% --- verbose message
%  fprintf('cur_data(%s) [%d x %d]\n', ...
%    ch_labels{i_ch}, size(cur_data,1), size(cur_data,2));
  
  filtered_data = vb_filter_raw_data(cur_data, proc_spec.parm);

  % --- re-stored filtered data
  new_file = sprintf('%s/%s.ch.eeg.dat', new_bin_dir, ch_labels{i_ch});
  
% --- verbose message
%   fprintf('%s/%s.ch.eeg.dat\n', new_bin_dir, ch_labels{i_ch});

  fid = fopen(new_file, 'wb');
  if fid == -1
    warning('(%s) cannot open file : %s\n', mfilename, new_file);
    continue;
  end
  fwrite(fid, filtered_data, precision{i_ch});
  fclose(fid);
end

n_sample = length(filtered_data);

% --- EXTRA CHANNELS and REFFERENCE CHANNELS
ch_labels_extra = vb_eegfile_get_channel_label(eeg_file, [], 2);
ch_len_extra = size(ch_labels_extra, 1);

for i_ch = 1:ch_len_extra

  load_spec.ChannelName = {ch_labels_extra{i_ch}};
  load_spec.ChannelSwitch = true; % ChannelName is to read
%   load_spec.ChannelType = 'EXT';
  load_spec.ChannelType = 'ALL';
  load_spec.ActiveChannel = 0; % all channels
  cur_data = vb_load_meg_data(eeg_file, load_spec);

  % Down sampling for external signal (No filtering)
  if isfield(proc_spec.parm, 'fsamp') && ...
    ~isempty(proc_spec.parm.fsamp) ...
  	&& proc_spec.parm.fsamp~=proc_spec.parm.freq,
	
    fprintf(' === EXTRA CHANNEL:[%s]', ch_labels_extra{i_ch});
 	fsamp = proc_spec.parm.freq;
  	fsnew = proc_spec.parm.fsamp;
  	T = size(cur_data,2);
  	t = round(1:fsamp/fsnew:T);
  	cur_data = cur_data(:,t);
  else
  	proc_spec.parm.fsamp = proc_spec.parm.freq;
  end
  
  % --- re-stored filtered data
  new_file = sprintf('%s/%s.ch.eeg.dat', new_bin_dir, ch_labels_extra{i_ch});
  fid = fopen(new_file, 'wb');
  if fid == -1
    warning('(%s) cannot open file : %s\n', mfilename, new_file);
    continue;
  end
  fwrite(fid, cur_data, precision{ch_len+i_ch});
  fclose(fid);
end


% --- POSTERIORI PROCESS
cur_eeg = load(eeg_file);
new_eeginfo = cur_eeg.EEGinfo;

% --- update fields
% channel info is not changed
new_eeginfo.Nsample  = n_sample;
new_eeginfo.SampleFrequency = proc_spec.parm.fsamp;
new_eeginfo.File.DataDir = proc_spec.new_bin_dir;
    
% --- undefined
new_eeginfo.FilterInfo = proc_spec.parm;

cur_eeg.EEGinfo = new_eeginfo;
eeg_data     = [];

fprintf('\n');
fprintf('save: %s\n',new_eeg_file);

save_cmd = sprintf('save ''%s'' -struct cur_eeg', new_eeg_file);
eval(save_cmd);
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [eeg_file, proc_spec, new_eeg_file] = inner_check_arguments(eeg_file, proc_spec, new_eeg_file)
func_ = mfilename;

% --- check meg_file
if isempty(eeg_file)
  error('(%s) eeg_file is a required parameter', func_);
end

if exist(eeg_file, 'file') ~= 2
  error('(%s) cannot find eeg_file : %s', func_, eeg_file);
end

if isempty(proc_spec)
  error('(%s) proc_spec is a required parameter', func_);
end

if ~isfield(proc_spec, 'parm')
  error('(%s) parm is a required field of proc_spec', func_);
end
if isempty(new_eeg_file)
  error('(%s) proc_spec is a required parameter', func_);
end
if ~strcmp(new_eeg_file(end-7:end), '.eeg.mat')
  error('(%s) specified new_meg_file should have ''.eeg.mat''', func_);
end

if ~isfield(proc_spec, 'ch_type')
  proc_spec.ch_type = 'MEG';
end
proc_spec.ch_type = upper(proc_spec.ch_type);
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
