function F = in_fread_brainamp(sFile, sfid, SamplesBounds)
% IN_FREAD_BRAINAMP:  Read a block of recordings from BrainVision BrainAmp .eeg file
%
% USAGE:  F = in_fread_brainamp(sFile, sfid, SamplesBounds=[])

% @=============================================================================
% This software is part of the Brainstorm software:
% http://neuroimage.usc.edu/brainstorm
% 
% Copyright (c)2000-2013 Brainstorm by the University of Southern California
% This software is distributed under the terms of the GNU General Public License
% as published by the Free Software Foundation. Further details on the GPL
% license can be found at http://www.gnu.org/copyleft/gpl.html.
% 
% FOR RESEARCH PURPOSES ONLY. THE SOFTWARE IS PROVIDED "AS IS," AND THE
% UNIVERSITY OF SOUTHERN CALIFORNIA AND ITS COLLABORATORS DO NOT MAKE ANY
% WARRANTY, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO WARRANTIES OF
% MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, NOR DO THEY ASSUME ANY
% LIABILITY OR RESPONSIBILITY FOR THE USE OF THIS SOFTWARE.
%
% For more information type "brainstorm license" at command prompt.
% =============================================================================@
%
% Authors: Francois Tadel, 2012-2013
% 2014-12-08 rhayashi Added scale calculation

% Parse inputs
if (nargin < 3) || isempty(SamplesBounds)
    SamplesBounds = sFile.prop.samples;
end

% BINARY and MULTIPLEXED files
if (strcmpi(sFile.header.DataFormat, 'BINARY') && strcmpi(sFile.header.DataOrientation, 'MULTIPLEXED'))
    nChan = sFile.header.NumberOfChannels;
    % Get start and length of block to read
    offsetData = SamplesBounds(1) * nChan * sFile.header.bytesize;
    nSamplesToRead = SamplesBounds(2) - SamplesBounds(1) + 1;
    % Position file at the beginning of the data block
    fseek(sfid, offsetData, 'bof');
    % Read all values at once
    F = fread(sfid, [nChan, nSamplesToRead], sFile.header.byteformat);
    % Convert from microVolts to Volts
    F = F * 1e-6;
    
% ASCII and VECTORIZED files
elseif (strcmpi(sFile.header.DataFormat, 'ASCII') && strcmpi(sFile.header.DataOrientation, 'VECTORIZED'))
    % Open file
    fid = fopen(sFile.filename, 'r');
    % Initialize data matrix
    F = zeros(sFile.header.NumberOfChannels, sFile.header.DataPoints);
    iChannel = 1;
    % Read the entire file line by line
    while(1)
        % Display message 
        % disp(sprintf('BRAINAMP> Reading channel #%d...', iChannel));
        % Reached the end of the file: exit the loop
        if feof(fid)
            break; 
        end;
        % Read one line
        strChan = strtrim(fgetl(fid));
        if isempty(strChan)
            continue;
        end
        % Find the first separator
        iSep = min([find(strChan == ' ',1), find(strChan == sprintf('\t'),1)]);
        % Replace "," with "." for the numbers
        strChan(strChan == ',') = '.';
        % Read the values
        F(iChannel,:) = sscanf(strChan(iSep+1:end), '%f')';
        iChannel = iChannel+1;
    end
    % Close file
    fclose(fid);
    % Select only the requested time points
    iTime = (SamplesBounds(1):SamplesBounds(2)) - sFile.prop.samples(1) + 1;
    F = F(:,iTime);
    % Convert from microVolts to Volts
    F = F * 1e-6;
end

% Scale data
for ch=1:sFile.header.NumberOfChannels
    F(ch, :) = F(ch, :) * sFile.channelmat.Channel(ch).scale;
end

