function sFile = in_fopen_brainamp(DataFile)
% IN_FOPEN_BRAINAMP: Open a BrainVision BrainAmp .eeg file.
%
% USAGE:  sFile = in_fopen_brainamp(DataFile)

% @=============================================================================
% This software is part of the Brainstorm software:
% http://neuroimage.usc.edu/brainstorm
% 
% Copyright (c)2000-2013 Brainstorm by the University of Southern California
% This software is distributed under the terms of the GNU General Public License
% as published by the Free Software Foundation. Further details on the GPL
% license can be found at http://www.gnu.org/copyleft/gpl.html.
% 
% FOR RESEARCH PURPOSES ONLY. THE SOFTWARE IS PROVIDED "AS IS," AND THE
% UNIVERSITY OF SOUTHERN CALIFORNIA AND ITS COLLABORATORS DO NOT MAKE ANY
% WARRANTY, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO WARRANTIES OF
% MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, NOR DO THEY ASSUME ANY
% LIABILITY OR RESPONSIBILITY FOR THE USE OF THIS SOFTWARE.
%
% For more information type "brainstorm license" at command prompt.
% =============================================================================@
%
% Authors: Guillaume Dumas, Francois Tadel, 2012-2013
% 2014-12-08 rhayashi Added channelmat.Channel(i).scale : scale factor
        
%% ===== GET FILES =====
% Build header and markers files names
% VHDR/AHDR File (header)
VhdrFile = [DataFile(1:end-4) '.vhdr'];
if ~file_exist(VhdrFile)
    VhdrFile = [DataFile(1:end-4) '.ahdr'];
    if ~file_exist(VhdrFile)
        error('Could not open VHDR header file.');
    else
        error(['File is saved in BrainVision V-Amp encrypted format.' 10 ...
               'Please, export it to "binary data format" before trying to read it in Brainstorm.']);
    end
end
% VMRK/AMRK File (markers)
VmrkFile = [DataFile(1:end-4) '.vmrk'];
if ~file_exist(VmrkFile)
    VmrkFile = [DataFile(1:end-4) '.amrk'];
    if ~file_exist(VmrkFile)
        disp('BRAINAMP> Warning: Could not open VMRK markers file.');
        VmrkFile = [];
    end
end


%% ===== READ HEADER =====
hdr.chnames = {};
hdr.chloc = [];
% Open and read file
fid = fopen(VhdrFile,'r');
curBlock = '';
% Read file line by line
while 1
    % Read one line
    newLine = fgetl(fid);
    if ~ischar(newLine)
        break;
    end
    % Empty lines and comment lines: skip
    if isempty(newLine) || ismember(newLine(1), {';', char(10), char(13)})
        continue;
    end
    % Read block names
    if (newLine(1) == '[')
        curBlock = newLine;
        curBlock(ismember(curBlock, '[] ')) = [];
        continue;
    end
    % Skip non-attribution lines
    if ~any(newLine == '=')
        continue;
    end
    % Split around the '='
    argLine = strtrim(str_split(newLine, '='));
    if (length(argLine) ~= 2) || (length(argLine{1}) < 2) || isempty(argLine{2})
        continue;
    end
    % Parameter
    if strcmpi(argLine{1}(1:2), 'Ch')
        iChan = str2num(argLine{1}(3:end));
        if ~isempty(iChan)
            if strcmpi(curBlock, 'ChannelInfos')
                hdr.chnames{iChan} = argLine{2};
            elseif strcmpi(curBlock, 'Coordinates')
                tmpLoc = str2num(argLine{2}); 
                if (length(tmpLoc) == 3) && (tmpLoc(1) == 1)
                    % Convert Spherical(degrees) => Spherical(radians) => Cartesian
                    TH  = (90 - tmpLoc(2)) ./ 180 * pi;
                    PHI = (90 + tmpLoc(3)) ./ 180 * pi;
                    [X,Y,Z] = sph2cart(PHI, TH, 1);
                    % Assign location
                    hdr.chloc(iChan,1:3) = [-X, -Y, Z+.5] .* .0875;
                end
            end
        end
    elseif ismember(argLine{1}, {'NumberOfChannels', 'SamplingInterval', 'DataPoints', 'SegmentDataPoints'})
        hdr.(argLine{1}) = str2num(argLine{2});
    else
        hdr.(file_standardize(argLine{1})) = argLine{2};
    end
end
% Close file
fclose(fid);


%% ===== REBUILD ACQ INFO =====
% BINARY and MULTIPLEXED files
if (strcmpi(hdr.DataFormat, 'BINARY') && strcmpi(hdr.DataOrientation, 'MULTIPLEXED'))
    % EEG file: get size
    dirInfo = dir(DataFile);
    % Get number of samples
    switch lower(hdr.BinaryFormat)
        case 'int_16';
            hdr.bytesize   = 2;
            hdr.byteformat = 'int16';
        case 'int_32';
            hdr.bytesize   = 4;
            hdr.byteformat = 'int32';
        case 'ieee_float_32';
            hdr.bytesize   = 4;
            hdr.byteformat = 'float32';
    end
    hdr.nsamples = dirInfo.bytes ./ (hdr.NumberOfChannels * hdr.bytesize);
% ASCII and VECTORIZED files
elseif (strcmpi(hdr.DataFormat, 'ASCII') && strcmpi(hdr.DataOrientation, 'VECTORIZED'))
    hdr.nsamples = hdr.DataPoints;
else
    error(['Only reading binary multiplexed or vectorize ASCII data format.' 10 'Please contact us if you would like to read other types of files in Brainstorm.']);
end



%% ===== CREATE BRAINSTORM SFILE STRUCTURE =====
% Initialize returned file structure
sFile = db_template('sfile');
% Add information read from header
sFile.byteorder  = 'l';
sFile.filename   = DataFile;
sFile.format     = 'EEG-BRAINAMP';
sFile.channelmat = [];
sFile.device     = 'BRAINAMP';
sFile.header     = hdr;
% Comment: short filename
[tmp__, sFile.comment, tmp__] = fileparts(DataFile);
% Consider that the sampling rate of the file is the sampling rate of the first signal
sFile.prop.sfreq   = 1e6 ./ hdr.SamplingInterval;
sFile.prop.samples = [0, hdr.nsamples - 1];
sFile.prop.times   = sFile.prop.samples ./ sFile.prop.sfreq;
sFile.prop.nAvg    = 1;
% No info on bad channels
sFile.channelflag = ones(hdr.NumberOfChannels, 1);


%% ===== CREATE EMPTY CHANNEL FILE =====
ChannelMat = db_template('channelmat');
ChannelMat.Comment = 'BrainAmp channels';
ChannelMat.Channel = repmat(db_template('channeldesc'), [1, hdr.NumberOfChannels]);
% For each channel
for i = 1:hdr.NumberOfChannels
    if ~isempty(hdr.chnames{i})
        read =  textscan(hdr.chnames{i}, '%s ', 'Delimiter', ',');
        chInfo = read{1};
        ChannelMat.Channel(i).Name = chInfo{1};
        scale = str2double(chInfo{3});
        if isempty(scale) || isnan(scale)
            scale = 1.0;
        end
        ChannelMat.Channel(i).scale = scale;
    else
        ChannelMat.Channel(i).Name = sprintf('E%d', i);
    end
    if ~isempty(hdr.chloc)
        ChannelMat.Channel(i).Loc = hdr.chloc(i,:)';
    else
        ChannelMat.Channel(i).Loc = [0; 0; 0];
    end
    ChannelMat.Channel(i).Type    = 'EEG';
    ChannelMat.Channel(i).Orient  = [];
    ChannelMat.Channel(i).Weight  = 1;
    ChannelMat.Channel(i).Comment = [];
end
% Return channel structure
sFile.channelmat = ChannelMat;
     

%% ===== READ EVENTS =====
if ~isempty(VmrkFile)
    newEvents = in_events_brainamp(sFile, VmrkFile);
    % Add each new event
    for iNew = 1:length(newEvents)
        if ~isempty(sFile.events)
            iEvt = find(strcmpi(newEvents(iNew).label, {sFile.events.label}));
        else
            iEvt = [];
        end
        % Make sure that the sample indices are round values
        newEvents(iNew).samples = round(newEvents(iNew).samples);
        newEvents(iNew).times   = newEvents(iNew).samples ./ sFile.prop.sfreq;
        % If event does not exist yet: add it at the end of the list
        if isempty(iEvt)
            if isempty(sFile.events)
                iEvt = 1;
                sFile.events = newEvents(iNew);
            else
                iEvt = length(sFile.events) + 1;
                sFile.events(iEvt) = newEvents(iNew);
            end
        % Event exists: merge occurrences
        else
            % Merge events occurrences
            sFile.events(iEvt).times      = [sFile.events(iEvt).times, newEvents(iNew).times];
            sFile.events(iEvt).samples    = [sFile.events(iEvt).samples, newEvents(iNew).samples];
            sFile.events(iEvt).epochs     = [sFile.events(iEvt).epochs, newEvents(iNew).epochs];
            sFile.events(iEvt).reactTimes = [sFile.events(iEvt).reactTimes, newEvents(iNew).reactTimes];
            % Sort by sample indices
            if (size(sFile.events(iEvt).samples, 2) > 1)
                [tmp__, iSort] = unique(sFile.events(iEvt).samples(1,:));
                sFile.events(iEvt).samples = sFile.events(iEvt).samples(:,iSort);
                sFile.events(iEvt).times   = sFile.events(iEvt).times(:,iSort);
                sFile.events(iEvt).epochs  = sFile.events(iEvt).epochs(iSort);
                if ~isempty(sFile.events(iEvt).reactTimes)
                    sFile.events(iEvt).reactTimes = sFile.events(iEvt).reactTimes(iSort);
                end
            end
        end
    end
end


function [ret] = file_exist(file)

if ~exist('file', 'var')
    error('specified file does not exist.');
end

if exist(file, 'file') == 2
   ret = true;
else
   ret = false;
end
