function [new_EEGinfo, exp_data, read_ch_info] = vb_read_ch_data_eeg( ...
  old_EEGinfo, to_read_swt, ch_list, time_from, time_to, verbose_swt)
% read experiment data from channel binary data file
% [usage]
%   [new_EEGinfo, exp_data, read_ch_info] = vb_read_ch_data_eeg( ...
%     old_EEGinfo, output_path, ...
%     to_read_swt, ch_list, time_from, time_to, verbose_swt)
% [input]
%    old_EEGinfo : <required> <<struct>> EEGinfo
%      : struct that is output from "vb_read_bdf_save_ch_bin"
%      : EEGinfo.Measurement           : measurement type 'EEG'
%      : EEGinfo.Device                : device name 'BIOSEMI'
%      : EEGinfo.File.namebase         : base name of channel data file
%      : EEGinfo.File.ext              : extension of channel data file
%      : EEGinfo.File.labeltype        : 'HARDWARE' or 'SOFTWARE'
%      : EEGinfo.Nchannel              : number of channel of original data
%      : EEGinfo.SampleFrequency       : sample rate of original data [Hz]
%      : EEGinfo.Nsample               : total number of sample
%      :                               :  SampleFrequency * RecordTime
%      : EEGinfo.DataType              : type of binary data (e.g. 'float32')
%      : EEGinfo.Coord                 : position coordinates of sensor
%      :                               : [n_sensor x 3]
%      : EEGinfo.CoodType              : coordinates' information
%
%      : EEGinfo.device_info           : device dependent data
%                                      :   TransInfo
%                                      :   Reference
%                                      :   RecordTime
%                                      :   Header
%                                      :   Version
%                                      :   History
%
%    to_read_swt : <optional> (true or false) purpose of "ch_list". [false]
%                :   true : ch_list is to read
%                :  false : ch_list is to omit
%        ch_list : <conditionally optional>
%                :   list of channel label that you want to get
%                :   if this "ch_list" is empty and ...
%                :     1) if "to_read_swt" is true ---> error
%                :     2) if "to_read_swt" is false --> read every channel
%      time_from : <optional> [msec]
%                :   beginning time that you want to get [0] (>=0)
%        time_to : <optional> [msec]
%                :   end of time that you want to get [end of data]
%                :   (>(time_from))
%    verbose_swt : <optional> switch to output verbose message [true]
% [output]
%    new_EEGinfo : new information struct data
%       exp_data : experiment data ([Nchannel x Nsample x Ntrial])
%                :   -- converted to phisical value by
%                :      [exp_data = eeg * EDF.Cal + EDF.Off]
%                :   -- (EDF = EEGinfo.header)
% [note]
%   @see vb_eeginfo_add_history
%   @see vb_util_arrange_list get_channel_idx
%   *** This function is DEPRECATED due to its incomplete check. ***
% [history]
%   2006-12-21 (Sako) initial version
%   2011-05-30 (Sako) modified according to the new data format
%   2011-07-29 (Sako) replaced searching index function
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% load common definitions for biosemi functions of vbmeg
vb_define_device;

% --- CHECK ARGUMENTS --- %
if ~exist('old_EEGinfo', 'var'), old_EEGinfo = []; end
if ~exist('to_read_swt', 'var'), to_read_swt = []; end
if ~exist('ch_list', 'var'),     ch_list = []; end
if ~exist('time_from', 'var'),   time_from = []; end
if ~exist('time_to', 'var'),     time_to = []; end
if ~exist('verbose_swt', 'var'), verbose_swt = []; end

[old_EEGinfo, to_read_swt, ch_list, time_from, time_to, VERBOSE,...
    sample_from, sample_to] = inner_check_argument( ...
  old_EEGinfo, to_read_swt, ch_list, time_from, time_to, verbose_swt);

% get local variables
[EDF, file_base, file_ext, Nch, SR, RT, data_type, data_size] = ...
  inner_get_local_var(old_EEGinfo);

% list of channel label as baseline
eeg_labels = vb_eeginfo_get_channel_label(old_EEGinfo);

% get list of label to read
channel_list = vb_util_get_labels_to_read(eeg_labels, ch_list, to_read_swt);
if isempty(channel_list), error('cannot make channel list'); end

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
func_ = mfilename;

% set channel_list [N x 1]
channel_list = vb_util_arrange_list(channel_list);

new_Nchannel = size(channel_list,1);
data_len = (sample_to + 1) - sample_from;

exp_data = zeros(new_Nchannel, data_len);

new_EEGinfo = old_EEGinfo;
new_idx = [];

common_data_type = data_type;
status_data_type = STATUS_CH_DATA_TYPE;
status_channel_label = STATUS_CH_LABEL;

for ch = 1:new_Nchannel
  file_path = sprintf('%s/%s%s', file_base, channel_list{ch}, file_ext);

  if exist(file_path, 'file') ~= 2
    error('(%s)cannot find file : %s', func_, file_path);
  end

  fid = fopen(file_path, 'r');
  if fid < 0
    error('(%s)cannot open : %s', func_, file_path); 
  end

  if strcmp(channel_list{ch}, status_channel_label) == 1
    status_ch = true;
  else
    status_ch = false;
  end
  
  % starting pointer
  p_begin = (sample_from-1) * data_size;
  fseek(fid, p_begin, 'bof');

  % data type
  if status_ch
    cur_data_type = status_data_type;
  else
    cur_data_type = common_data_type;
  end
  
%   this_eeg = (fread(fid, data_len, data_type))';
  this_eeg = (fread(fid, data_len, cur_data_type{ch}))';
  
  idx = vb_util_get_matching_label_index(eeg_labels, channel_list{ch});
  new_idx = [new_idx idx];
  
  % convert to phisical value
  if status_ch
    exp_data(ch,:) = this_eeg;
  else
    exp_data(ch,:) = this_eeg * EDF.Cal(idx) + EDF.Off(idx);
  end

  channel_info.label = channel_list{ch};
  channel_info.sample_begin = sample_from;
  channel_info.sample_end = sample_to;

  read_ch_info(ch) = channel_info;

  fclose(fid);

  if VERBOSE
    %{
    fprintf(' --  read channel: ''%s''\n', channel_list{ch});
    fprintf(' -- target sample: (%d -> %d)\n', sample_from, sample_to);
    fprintf(' ------------------------------------------------------\n');
    %}

    msg = sprintf(' --  read channel: ''%s''', channel_list{ch});
    msg = sprintf('%s\n -- target sample: (%d -> %d)', ...
      msg, sample_from, sample_to);
    msg = sprintf('%s\n ----------------------------------------', msg);
    vb_disp(msg);
  end
end

% update EEGinfo
new_EEGinfo = inner_update_EEGinfo(new_EEGinfo, ...
  new_idx, sample_from, sample_to, data_len, SR, channel_list, new_Nchannel);
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_argument()
% 
function [old_EEGinfo, to_read_swt, ch_list, time_from, time_to, VERBOSE, ...
    sample_from, sample_to] = inner_check_argument( ...
  old_EEGinfo, to_read_swt, ch_list, time_from, time_to, verbose_swt)

vb_define_device;

% required arguments
if isempty(old_EEGinfo)
  error('old_EEGinfo is a required parameter');
else
  % required fields
  if ~isfield(old_EEGinfo, 'File')
    error('old_EEGinfo must have %s field', 'FILE');
  end
  if ~isfield(old_EEGinfo, 'Device')
    error('old_EEGinfo must have %s field', 'Device');
  end
  if ~isfield(old_EEGinfo, 'Nchannel')
    error('old_EEGinfo must have %s field', 'Nchannel');
  end
  if ~isfield(old_EEGinfo, 'SampleFrequency')
    error('old_EEGinfo must have %s field', 'SampleFrequency');
  end
  if ~isfield(old_EEGinfo, 'Nsample')
    error('old_EEGinfo must have %s field', 'Nsample');
  end
  if ~isfield(old_EEGinfo, 'DataType')
    error('old_EEGinfo must have %s field', 'DataType');
  end

  % --- device dependent data
  if ~isfield(old_EEGinfo, 'device_info')
    error('old_EEGinfo must have %s field', 'device_info');
  else
    if ~isfield(old_EEGinfo.device_info, 'Header')
      error('old_EEGinfo.device_info must have %s field', 'Header');
    end
    if ~isfield(old_EEGinfo.device_info, 'RecordTime')
      error('old_EEGinfo.device_info must have %s field', 'RecordTime');
    end
  end
end

if isempty(to_read_swt)
  to_read_swt = false;
end

if isempty(time_from)
  sample_from = 1;
elseif ~isnumeric(time_from)
  error('[time_from] must be numeric');
end;

if isempty(time_to)
  sample_to = old_EEGinfo.Nsample;
elseif ~isnumeric(time_to)
  error('[time_to] must be numeric');
end;

SR = old_EEGinfo.SampleFrequency;

if ~exist('sample_from', 'var')
  sample_from = round(time_from * SR/1000);
end
if ~exist('sample_to', 'var')
  sample_to   = round(time_to * SR/1000);
end

if sample_from < 0 || old_EEGinfo.Nsample < sample_from
  error('time_from %d[msec] is out of range (Nsample:%d)', ...
    time_from, old_EEGinfo.Nsample);
end

if sample_to < sample_from || old_EEGinfo.Nsample < sample_to
  error('time_to %d[msec] is out of range (time_from:%d,Nsample:%d)', ...
    time_to, time_from, old_EEGinfo.Nsample);
end

if isempty(verbose_swt)
  verbose_swt = true;
end

if verbose_swt == true
  VERBOSE = true;
else
  VERBOSE = false;
end
% 
% --- end of inner_check_argument()

% --- inner_update_EEGinfo()
%
function EEGinfo = inner_update_EEGinfo(EEGinfo, ...
  new_idx, sample_from, sample_to, data_len, SR, ch_list, new_Nchannel);

vb_define_device;

EEGinfo.Nsample = (sample_to + 1) - sample_from;
EEGinfo.Nchannel = new_Nchannel;
old_SensorPosition = vb_eeginfo_get_sensor_position(EEGinfo);
pos_idx = find(size(old_SensorPosition,1) >= new_idx);
EEGinfo = vb_eeginfo_set_sensor_position( ...
  EEGinfo, old_SensorPosition(pos_idx,:));

EEGinfo = vb_eeginfo_set_rectime(EEGinfo, (data_len / SR));

% history
ch_str = [];
for n_ch = 1:size(ch_list,1)
  if isempty(ch_str)
    ch_str = sprintf('%s', ch_list{n_ch});
  else
    ch_str = sprintf('%s,%s', ch_str, ch_list{n_ch});
  end
end

history = ...
  sprintf('vb_read_eeg_ch_data(sample{%d, %d}, ch:%s)', ...
    sample_from, sample_to, ch_str);
EEGinfo = vb_eeginfo_add_history(EEGinfo, history);
%
% --- end of inner_update_EEGinfo()

% --- inner_get_local_var()
%
function [EDF, file_base, file_ext, Nch, SR, RT, data_type, data_size] = ...
  inner_get_local_var(old_EEGinfo)

% prepare EDF
EDF = vb_eeginfo_get_header(old_EEGinfo);
if ~isfield(EDF,'Cal'),  error('EDF must have %s field', 'Cal'); end
if ~isfield(EDF,'Off'),  error('EDF must have %s field', 'Off'); end

% get local variables from EEGinfo
f = old_EEGinfo.File;
file_base  = [f.OutputDir '/' f.DataDir];
file_ext   = '.ch.eeg.dat';
Nch        = old_EEGinfo.Nchannel;
SR         = old_EEGinfo.SampleFrequency;
RT         = vb_eeginfo_get_rectime(old_EEGinfo);
data_type  = old_EEGinfo.DataType;
data_size  = 4; % [byte] (float32)
%
% --- end of inner_get_local_var()

% --- END OF FILE --- %
