function [result] = vb_posfile_replace_label(src_file, dst_file, rpl_type)
% replace channel labels of posfile
% [usage]
%   [result] = vb_posfile_replace_label(src_file, dst_file, rpl_type)
% [input]
%   src_file : <required> POS-MAT file that will be converted
%   dst_file : <required> POS-MAT file that will be newly made
%   rpl_type : <optional> replace type
%            :  [0]) EEG from Fp1 series to A1 series (Biosemi 64 ch default)
%            :   1 ) EEG from A1 series to Fp1 series
% [output]
%     result : ==0) success
%            : ~=0) error occured
% [note]
%   support only the case of 64 channel EEG default, for now
% [history]
%   2010-05-07 (Sako) initial version
%   2012-11-14 (Sako) added conversion A1 to Fp1
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

func_ = mfilename;
result = 0; % success

% --- CHECK ARGUMENTS --- %
if ~exist('src_file', 'var'), src_file = ''; end
if ~exist('dst_file', 'var'), dst_file = ''; end
if ~exist('rpl_type', 'var'), rpl_type = []; end

[src_file, dst_file, rpl_type] = ...
  inner_check_arguments(src_file, dst_file, rpl_type);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%

% --- a sample of fields of POS-MAT file ------------------------------ *
%            mri_key: '2378239763b93c4c66ce9acde6d0ecb8'
%                pos: [64x3 double]
%               name: {64x1 cell}
%         coord_type: 'SPM_Right_m'
%     spherical_info: [1x1 struct]
%         trans_info: [1x1 struct]
%             header: [1x1 struct]
% ------------------------------------------------------------- *
pos = load(src_file);
if ~isfield(pos, 'name')
  fprintf('(%s) cannot find name field\n', func_);
  result = 1; % failure
  return;
end

switch rpl_type
  case 0
    new_labels = inner_sub_64_fp1_to_a1(pos.name);

  case 1
    new_labels = inner_sub_64_a1_to_fp1(pos.name);

  otherwise
    fprintf('(%s) unknown replace type : %d\n', func_, rpl_type);
    result = 1;
    return;
end

% overwrite labels
pos.name = new_labels;
vb_save_struct(dst_file, pos);
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [src_file, dst_file, rpl_type] = ...
  inner_check_arguments(src_file, dst_file, rpl_type)
func_ = mfilename;

if isempty(src_file)
  error('(%s) src_file is a required parameter', func_);
end

if exist(src_file, 'file') ~= 2
  error('(%s) cannot find src_file : %s\n', func_, src_file);
end

if isempty(dst_file)
  error('(%s) dst_file is a required parameter', func_);
end

if isempty(rpl_type)
  rpl_type = 0;
end
return;
%
% --- end of inner_check_arguments()


% --- inner_sub_fp1_to_a1()
%
function new_labels = inner_sub_64_fp1_to_a1(old_labels)

n_old = length(old_labels);
new_labels = cell(64,1);
i_new = 1;

for i_old = 1:n_old
  if strcmp(old_labels{i_old}, 'Fp1'), new_labels{i_new} = 'A1';
  elseif strcmp(old_labels{i_old}, 'Fpz'), new_labels{i_new} = 'B1';
  elseif strcmp(old_labels{i_old}, 'Fp2'), new_labels{i_new} = 'B2';

  elseif strcmp(old_labels{i_old}, 'AF7'), new_labels{i_new} = 'A2';
  elseif strcmp(old_labels{i_old}, 'AF3'), new_labels{i_new} = 'A3';
  elseif strcmp(old_labels{i_old}, 'AFz'), new_labels{i_new} = 'B5';
  elseif strcmp(old_labels{i_old}, 'AF4'), new_labels{i_new} = 'B4';
  elseif strcmp(old_labels{i_old}, 'AF8'), new_labels{i_new} = 'B3';
  
  elseif strcmp(old_labels{i_old}, 'F7' ), new_labels{i_new} = 'A7';
  elseif strcmp(old_labels{i_old}, 'F5' ), new_labels{i_new} = 'A6';
  elseif strcmp(old_labels{i_old}, 'F3' ), new_labels{i_new} = 'A5';
  elseif strcmp(old_labels{i_old}, 'F1' ), new_labels{i_new} = 'A4';
  elseif strcmp(old_labels{i_old}, 'Fz' ), new_labels{i_new} = 'B6';
  elseif strcmp(old_labels{i_old}, 'F2' ), new_labels{i_new} = 'B7';
  elseif strcmp(old_labels{i_old}, 'F4' ), new_labels{i_new} = 'B8';
  elseif strcmp(old_labels{i_old}, 'F6' ), new_labels{i_new} = 'B9';
  elseif strcmp(old_labels{i_old}, 'F8' ), new_labels{i_new} = 'B10';
  
  elseif strcmp(old_labels{i_old}, 'FT7'), new_labels{i_new} = 'A8';
  elseif strcmp(old_labels{i_old}, 'FC5'), new_labels{i_new} = 'A9';
  elseif strcmp(old_labels{i_old}, 'FC3'), new_labels{i_new} = 'A10';
  elseif strcmp(old_labels{i_old}, 'FC1'), new_labels{i_new} = 'A11';
  elseif strcmp(old_labels{i_old}, 'FCz'), new_labels{i_new} = 'B15';
  elseif strcmp(old_labels{i_old}, 'FC2'), new_labels{i_new} = 'B14';
  elseif strcmp(old_labels{i_old}, 'FC4'), new_labels{i_new} = 'B13';
  elseif strcmp(old_labels{i_old}, 'FC6'), new_labels{i_new} = 'B12';
  elseif strcmp(old_labels{i_old}, 'FT8'), new_labels{i_new} = 'B11';
  
  elseif strcmp(old_labels{i_old}, 'T7' ), new_labels{i_new} = 'A15';
  elseif strcmp(old_labels{i_old}, 'C5' ), new_labels{i_new} = 'A14';
  elseif strcmp(old_labels{i_old}, 'C3' ), new_labels{i_new} = 'A13';
  elseif strcmp(old_labels{i_old}, 'C1' ), new_labels{i_new} = 'A12';
  elseif strcmp(old_labels{i_old}, 'Cz' ), new_labels{i_new} = 'B16';
  elseif strcmp(old_labels{i_old}, 'C2' ), new_labels{i_new} = 'B17';
  elseif strcmp(old_labels{i_old}, 'C4' ), new_labels{i_new} = 'B18';
  elseif strcmp(old_labels{i_old}, 'C6' ), new_labels{i_new} = 'B19';
  elseif strcmp(old_labels{i_old}, 'T8' ), new_labels{i_new} = 'B20';
  
  elseif strcmp(old_labels{i_old}, 'TP7'), new_labels{i_new} = 'A16';
  elseif strcmp(old_labels{i_old}, 'CP5'), new_labels{i_new} = 'A17';
  elseif strcmp(old_labels{i_old}, 'CP3'), new_labels{i_new} = 'A18';
  elseif strcmp(old_labels{i_old}, 'CP1'), new_labels{i_new} = 'A19';
  elseif strcmp(old_labels{i_old}, 'CPz'), new_labels{i_new} = 'A32';
  elseif strcmp(old_labels{i_old}, 'CP2'), new_labels{i_new} = 'B24';
  elseif strcmp(old_labels{i_old}, 'CP4'), new_labels{i_new} = 'B23';
  elseif strcmp(old_labels{i_old}, 'CP6'), new_labels{i_new} = 'B22';
  elseif strcmp(old_labels{i_old}, 'TP8'), new_labels{i_new} = 'B21';
  
  elseif strcmp(old_labels{i_old}, 'P9' ), new_labels{i_new} = 'A24';
  elseif strcmp(old_labels{i_old}, 'P7' ), new_labels{i_new} = 'A23';
  elseif strcmp(old_labels{i_old}, 'P5' ), new_labels{i_new} = 'A22';
  elseif strcmp(old_labels{i_old}, 'P3' ), new_labels{i_new} = 'A21';
  elseif strcmp(old_labels{i_old}, 'P1' ), new_labels{i_new} = 'A20';
  elseif strcmp(old_labels{i_old}, 'Pz' ), new_labels{i_new} = 'A31';
  elseif strcmp(old_labels{i_old}, 'P2' ), new_labels{i_new} = 'B25';
  elseif strcmp(old_labels{i_old}, 'P4' ), new_labels{i_new} = 'B26';
  elseif strcmp(old_labels{i_old}, 'P6' ), new_labels{i_new} = 'B27';
  elseif strcmp(old_labels{i_old}, 'P8' ), new_labels{i_new} = 'B28';
  elseif strcmp(old_labels{i_old}, 'P10'), new_labels{i_new} = 'B29';
  
  elseif strcmp(old_labels{i_old}, 'PO7'), new_labels{i_new} = 'A25';
  elseif strcmp(old_labels{i_old}, 'PO3'), new_labels{i_new} = 'A26';
  elseif strcmp(old_labels{i_old}, 'POz'), new_labels{i_new} = 'A30';
  elseif strcmp(old_labels{i_old}, 'PO4'), new_labels{i_new} = 'B31';
  elseif strcmp(old_labels{i_old}, 'PO8'), new_labels{i_new} = 'B30';

  elseif strcmp(old_labels{i_old}, 'O1' ), new_labels{i_new} = 'A27';
  elseif strcmp(old_labels{i_old}, 'Oz' ), new_labels{i_new} = 'A29';
  elseif strcmp(old_labels{i_old}, 'O2' ), new_labels{i_new} = 'B32';

  elseif strcmp(old_labels{i_old}, 'Iz' ), new_labels{i_new} = 'A28';
  end
  
  i_new = i_new + 1;
end
return;
%
% --- end of inner_sub_fp1_to_a1()


% --- inner_sub_a1_to_fp1()
%
function new_labels = inner_sub_64_a1_to_fp1(old_labels)

n_old = length(old_labels);
new_labels = cell(64,1);
i_new = 1;

for i_old = 1:n_old
  if strcmp(old_labels{i_old}, 'A1'), new_labels{i_new} = 'Fp1';
  elseif strcmp(old_labels{i_old}, 'B1'), new_labels{i_new} = 'Fpz1';
  elseif strcmp(old_labels{i_old}, 'B2'), new_labels{i_new} = 'Fp2';

  elseif strcmp(old_labels{i_old}, 'A2'), new_labels{i_new} = 'AF7';
  elseif strcmp(old_labels{i_old}, 'A3'), new_labels{i_new} = 'AF3';
  elseif strcmp(old_labels{i_old}, 'B5'), new_labels{i_new} = 'AFz';
  elseif strcmp(old_labels{i_old}, 'B4'), new_labels{i_new} = 'AF4';
  elseif strcmp(old_labels{i_old}, 'B3'), new_labels{i_new} = 'AF8';
  
  elseif strcmp(old_labels{i_old}, 'A7' ), new_labels{i_new} = 'F7';
  elseif strcmp(old_labels{i_old}, 'A6' ), new_labels{i_new} = 'F5';
  elseif strcmp(old_labels{i_old}, 'A5' ), new_labels{i_new} = 'F3';
  elseif strcmp(old_labels{i_old}, 'A4' ), new_labels{i_new} = 'F1';
  elseif strcmp(old_labels{i_old}, 'B6' ), new_labels{i_new} = 'Fz';
  elseif strcmp(old_labels{i_old}, 'B7' ), new_labels{i_new} = 'F2';
  elseif strcmp(old_labels{i_old}, 'B8' ), new_labels{i_new} = 'F4';
  elseif strcmp(old_labels{i_old}, 'B9' ), new_labels{i_new} = 'F6';
  elseif strcmp(old_labels{i_old}, 'B10' ), new_labels{i_new} = 'F8';
  
  elseif strcmp(old_labels{i_old}, 'A8'), new_labels{i_new} = 'FT7';
  elseif strcmp(old_labels{i_old}, 'A9'), new_labels{i_new} = 'FC5';
  elseif strcmp(old_labels{i_old}, 'A10'), new_labels{i_new} = 'FC3';
  elseif strcmp(old_labels{i_old}, 'A11'), new_labels{i_new} = 'FC1';
  elseif strcmp(old_labels{i_old}, 'B15'), new_labels{i_new} = 'FCz';
  elseif strcmp(old_labels{i_old}, 'B14'), new_labels{i_new} = 'FC2';
  elseif strcmp(old_labels{i_old}, 'B13'), new_labels{i_new} = 'FC4';
  elseif strcmp(old_labels{i_old}, 'B12'), new_labels{i_new} = 'FC6';
  elseif strcmp(old_labels{i_old}, 'B11'), new_labels{i_new} = 'FT8';
  
  elseif strcmp(old_labels{i_old}, 'A15' ), new_labels{i_new} = 'T7';
  elseif strcmp(old_labels{i_old}, 'A14' ), new_labels{i_new} = 'C5';
  elseif strcmp(old_labels{i_old}, 'A13' ), new_labels{i_new} = 'C3';
  elseif strcmp(old_labels{i_old}, 'A12' ), new_labels{i_new} = 'C1';
  elseif strcmp(old_labels{i_old}, 'B16' ), new_labels{i_new} = 'Cz';
  elseif strcmp(old_labels{i_old}, 'B17' ), new_labels{i_new} = 'C2';
  elseif strcmp(old_labels{i_old}, 'B18' ), new_labels{i_new} = 'C4';
  elseif strcmp(old_labels{i_old}, 'B19' ), new_labels{i_new} = 'C6';
  elseif strcmp(old_labels{i_old}, 'B20' ), new_labels{i_new} = 'T8';
  
  elseif strcmp(old_labels{i_old}, 'A16'), new_labels{i_new} = 'TP7';
  elseif strcmp(old_labels{i_old}, 'A17'), new_labels{i_new} = 'CP5';
  elseif strcmp(old_labels{i_old}, 'A18'), new_labels{i_new} = 'CP3';
  elseif strcmp(old_labels{i_old}, 'A19'), new_labels{i_new} = 'CP1';
  elseif strcmp(old_labels{i_old}, 'A32'), new_labels{i_new} = 'CPz';
  elseif strcmp(old_labels{i_old}, 'B24'), new_labels{i_new} = 'CP2';
  elseif strcmp(old_labels{i_old}, 'B23'), new_labels{i_new} = 'CP4';
  elseif strcmp(old_labels{i_old}, 'B22'), new_labels{i_new} = 'CP6';
  elseif strcmp(old_labels{i_old}, 'B21'), new_labels{i_new} = 'TP8';
  
  elseif strcmp(old_labels{i_old}, 'A24' ), new_labels{i_new} = 'P9';
  elseif strcmp(old_labels{i_old}, 'A23' ), new_labels{i_new} = 'P7';
  elseif strcmp(old_labels{i_old}, 'A22' ), new_labels{i_new} = 'P5';
  elseif strcmp(old_labels{i_old}, 'A21' ), new_labels{i_new} = 'P3';
  elseif strcmp(old_labels{i_old}, 'A20' ), new_labels{i_new} = 'P1';
  elseif strcmp(old_labels{i_old}, 'A31' ), new_labels{i_new} = 'Pz';
  elseif strcmp(old_labels{i_old}, 'B25' ), new_labels{i_new} = 'P2';
  elseif strcmp(old_labels{i_old}, 'B26' ), new_labels{i_new} = 'P4';
  elseif strcmp(old_labels{i_old}, 'B27' ), new_labels{i_new} = 'P6';
  elseif strcmp(old_labels{i_old}, 'B28' ), new_labels{i_new} = 'P8';
  elseif strcmp(old_labels{i_old}, 'B29'), new_labels{i_new} = 'P10';
  
  elseif strcmp(old_labels{i_old}, 'A25'), new_labels{i_new} = 'PO7';
  elseif strcmp(old_labels{i_old}, 'A26'), new_labels{i_new} = 'PO3';
  elseif strcmp(old_labels{i_old}, 'A30'), new_labels{i_new} = 'POz';
  elseif strcmp(old_labels{i_old}, 'B31'), new_labels{i_new} = 'PO4';
  elseif strcmp(old_labels{i_old}, 'B30'), new_labels{i_new} = 'PO8';

  elseif strcmp(old_labels{i_old}, 'A27' ), new_labels{i_new} = 'O1';
  elseif strcmp(old_labels{i_old}, 'A29' ), new_labels{i_new} = 'Oz';
  elseif strcmp(old_labels{i_old}, 'B32' ), new_labels{i_new} = 'O2';

  elseif strcmp(old_labels{i_old}, 'A28' ), new_labels{i_new} = 'Iz';
  end
  
  i_new = i_new + 1;
end
return;
%
% --- end of inner_sub_a1_to_fp1()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
