function [state, guide_def] = vb_util_check_variable_in_matfile(mat_file, hot_var)
% check states of a variable of mat file
% [usage]
%   [state, guide_def] = vb_util_check_variable_in_matfile(mat_file, hot_var)
% [input]
%   mat_file : <required> <<file>> .mat file to be checked
%    hot_var : <required> name of variable which you want to check
% [output]
%      state : <<integer>> state code defind by bit sequence as follows:
%            :  1) (01) it exists
%            :  2) (10) it is not empty
%  guide_def : meaning of state
%            : (3) VALID ------- hot_var exist and is not empty
%            : (1) VACANT ------ hot_var exist but is empty
%            : (0) NONEXISTENT - hot_var does not exist in mat_file
% [note]
%   <example>
%     hot_var = 'a_var';
%     [state, const] = vb_util_check_variable_in_matfile(sample_mat, hot_var);
%     if state == const.VALID
%       fprintf('VALID\n');
%     elseif state == const.VACANT
%       fprintf('VACANT\n');
%     elseif state == const.NONEXISTENT
%       fprintf('NONEXISTENT\n');
%     else
%       fprintf('SOMETHING BAD STATE (%d)\n', state);
%     end
%   
%   @see vb_meg_tutorial(20001)
%   @see whos
% [history]
%   2010-02-10 (Sako) initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('mat_file', 'var'), mat_file = ''; end
if ~exist('hot_var', 'var'), hot_var = ''; end
[mat_file, hot_var] = inner_check_arguments(mat_file, hot_var);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
% --- constants
NONEXISTENT = 0;
   EXISTENT = 1;
  NOT_EMPTY = 2;

guide_def.NONEXISTENT = 0;
guide_def.VACANT = 1;
guide_def.VALID  = 3;

state = NONEXISTENT;

mat_head = whos('-file', mat_file);
n_head = size(mat_head,1);
for i_head = 1:n_head
  if strcmp(mat_head(i_head).name, hot_var)
    state = bitor(state, EXISTENT);
    if mat_head(i_head).bytes > 0
      state = bitor(state, NOT_EMPTY);
    end
    break;
  end
end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [mat_file, hot_var] = inner_check_arguments(mat_file, hot_var)
func_ = mfilename;
if isempty(mat_file)
  error('(%s) mat_file is a required parameter', func_);
end

if exist(mat_file, 'file') ~= 2
  error('(%s) cannot find %s', func_, mat_file);
end

if isempty(hot_var)
  error('(%s) hot_var is a required parameter', func_);
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
