function vb_concat_megfile(concat_parm)
% Concatenate MEG data sets. 
% Data sets must have the same number of channels and be recorded with
% the same sampling frequency. 
%
% --- Syntax
% vb_concat_megfile(concat_parm)
%
% --- Input
% concat_parm.megfile : New MEG data is saved with this file name. 
% concat_parm.ix_info : Recording information (active flag, sensor
%                       position, etc.) of the new MEG data is set to
%                       that of (ix_info)-th MEG data. 
% concat_parm.data{n} : Information of n-th MEG data. 
% data{n}.megfile: .meg.mat file concatenated.
% data{n}.twin   : Time window in sample number, not actual time. 
% data{n}.trial  : Sequence of trial numbers. 
%
% --- Input (optional)
% concat_parm.pretrigger: Pretrigger period of the new MEG data
%
% --- Example
% The following is an example to concatenates MEG data of 1-40 trials in
% 'meg1.meg.mat' and 41-80 trials in 'meg2.meg.mat'. The concatenated MEG
% data has the recording information of 'meg2.meg.mat'. 
%
% >> concat_parm.megfile = 'new_data.meg.mat';
% >> concat_parm.ix_info = 2;
% >> concat_parm.Pretrigger = 300;
% >> concat_parm.data{1}.megfile = 'meg1.meg.mat';
% >> concat_parm.data{1}.twin = [1 300];
% >> concat_parm.data{1}.trial = 1:40;
% >> concat_parm.data{2}.megfile = 'meg2.meg.mat';
% >> concat_parm.data{2}.twin = [1001 2000];
% >> concat_parm.data{2}.trial = 41:80; 
% >> vb_concat_megfile(concat_parm);
%
% --- History
% 2008-06-12 Taku Yoshioka
% 2008-06-27 Taku Yoshioka (pretrigger)
% 2009-09-01 (Sako) substituted vb_load_measurement_info for vb_load_meg_info
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

vb_struct2vars(concat_parm,{'data','ix_info','megfile'});
const = vb_define_const;

% Check consistency of data
Ndata = length(data);
MEGinfo = vb_load_measurement_info(data{1}.megfile);
Nrepeat = length(data{1}.trial);
Nchannel = MEGinfo.Nchannel;
SampleFreq = MEGinfo.SampleFreq;

for i=2:length(concat_parm.data)
  if length(data{i}.trial)~=Nrepeat, 
    error('Number of trials must be same.');
  end
  
  MEGinfo = vb_load_measurement_info(data{i}.megfile);
  if MEGinfo.SampleFreq~=SampleFreq, 
    error('Sampling frequency must be same.');
  end
  
  if MEGinfo.Nchannel~=Nchannel, 
    error('Channel number must be same.');
  end
end

% Sampling frame
ixx = (data{1}.twin(1)):(data{1}.twin(2));
ix_sample0{1} = 1:length(ixx);
ix_sample{1} = ix_sample0{1};
for i=2:Ndata
  ixx = (data{i}.twin(1)):(data{i}.twin(2));
  ix_sample0{i} = 1:length(ixx);
  ix_sample{i} = ix_sample0{i}+ix_sample{i-1}(end);
end

% Create concatenated data
for i=1:Ndata
  load_spec = [];
  load_spec.TrialNumber = data{i}.trial;
  load_spec.ActiveChannel = false;
  load_spec.ActiveTrial = false;

  % MEG channel
  load_spec.ChannelType = const.DATA_TYPE_MAIN;
  tmp = vb_load_meg_data(data{i}.megfile,load_spec);
  bexp(:,ix_sample{i},:) = tmp(:,ix_sample0{i},:);
  clear tmp;
  
  % External channel
  load_spec.ChannelType = const.DATA_TYPE_EXTRA;
  tmp = vb_load_meg_data(data{i}.megfile,load_spec);
  bexp_ext(:,ix_sample{i},:) = tmp(:,ix_sample0{i},:);

  % Reference channel
  load_spec.ChannelType = const.DATA_TYPE_REFERENCE;
  tmp = vb_load_meg_data(data{i}.megfile,load_spec);
  refmg(:,ix_sample{i},:) = tmp(:,ix_sample0{i},:);
  clear tmp;
end

% Trial information
for i=1:Nrepeat
  Trial(i).number = i;
  Trial(i).Sample = 1:size(bexp,2);
  Trial(i).Active = 1;
end

% Update MEGinfo
MEGinfo = vb_load_measurement_info(data{ix_info}.megfile);
MEGinfo.Nsample = size(bexp,2);
MEGinfo.Nrepeat = Nrepeat;
MEGinfo.Trial = Trial;
MEGinfo = vb_info_adjust_trial(MEGinfo);
if isfield(concat_parm,'Pretrigger'), 
  MEGinfo.Pretrigger = concat_parm.Pretrigger;
end

% Save data
load(data{ix_info}.megfile,'pick','Qpick','ref_pick','ref_Qpick', ...
     'CoordType','PositionFile','Measurement');
vb_fsave(megfile,'pick','Qpick','ref_pick','ref_Qpick', ...
         'CoordType','bexp','bexp_ext','refmg','PositionFile', ...
         'MEGinfo','Measurement');
