function [eeg_data] = vb_combine_eeg_files(in_files, out_file, ext_spec)
% combine plural .eeg.mat files and create a single .eeg.mat file
%
% [usage]
%   [eeg_data] = vb_combine_eeg_files(in_files, out_file, ext_spec)
%
% [input]
%   in_files : <required> <<cell array>> .eeg.mat files to be combined
%   out_file : <required> <<string>> combined file
%   ext_spec : <optional> <<struct>> 
%            :  .bin_data_dir : <optional> relative path from new_file
%            :    This field is valid only when #3 argument newfile is 
%            :  specified.
%            :  if this field is not given or is empty, sampling data
%            :  will be stored internally (eeg_data).
%
% [output]
%   eeg_data : combined data 
%
% [note]
%   See also :
%     vb_combine_meg_files
%   
% [history]
%   2012-11-20 (Sako) initial version

% --- CHECK ARGUMENTS --- %
if ~exist('in_files', 'var'), in_files = []; end
if ~exist('out_file', 'var'), out_file = ''; end
if ~exist('ext_spec', 'var'), ext_spec = []; end
[in_files, out_file, ext_spec] = ...
  inner_check_arguments(in_files, out_file, ext_spec);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
n_eeg = length(in_files);
if n_eeg == 0
  eeg_data = [];
  fprintf('(%s) in_files is empty\n', mfilename);
  return;
end

if n_eeg == 1
  eeg_data = vb_load_meg_data(in_files{1});
  fprintf('(%s) there is nothing to do\n', mfilename);
  return;
end

% --- check contents of in_files
chk_result = inner_is_combinable(in_files);

if ~chk_result
  eeg_data = [];
  return;
end

last_trial = 0;
n_repeat = 0;

% --- EEGinfo - basically adopt the first data
EEGinfo = vb_load_measurement_info(in_files{1});

active_trial = [];
trial = [];
base_file = [];


% --- concatenate data
load_spec.ChannelType = 'ALL';

for i_eeg = 1:n_eeg
  % --- eeg_data
  cur_eeg = vb_load_meg_data(in_files{i_eeg}, load_spec);
  cur_channel = size(cur_eeg, 1);
  cur_sample  = size(cur_eeg, 2);
  cur_trial   = size(cur_eeg, 3);

  start_tr = last_trial + 1;
  end_tr   = last_trial + cur_trial;
  eeg_data(:,:,start_tr:end_tr) = cur_eeg;
  last_trial = end_tr;

  cur_info = vb_load_measurement_info(in_files{i_eeg});
  n_repeat = n_repeat + cur_info.Nrepeat;
  
  if isfield(cur_info, 'Trial')
    trial_idx_new = start_tr : end_tr;
    for i = 1:cur_trial
      cur_info.Trial(i).number = trial_idx_new(i);
    end
    trial = [trial; cur_info.Trial];
  end
  
  if isfield(cur_info, 'ActiveTrial')
    active_trial = [active_trial; cur_info.ActiveTrial];
  end
  
  if isfield(cur_info, 'File')
    base_file = [base_file; cur_info.File.BaseFile];
  end
  
end

% --- re-arrange EEGinfo
EEGinfo.Nrepeat = n_repeat;
EEGinfo.Trial = trial;
EEGinfo.ActiveTrial = active_trial;

Measurement = EEGinfo.Measurement;

[ch_name, data_type, bin_data_dir, bin_data_dir_rel] = ...
  inner_solve_external_data(out_file, ext_spec, EEGinfo);

% ----- EEGinfo.File
EEGinfo.File.BaseFile = base_file;

[out_path, out_name] = vb_get_file_parts(out_file);
EEGinfo.File.OutputDir = out_path;
EEGinfo.File.EEGFile = out_name;
EEGinfo.File.DataDir = bin_data_dir_rel;

if isempty(bin_data_dir)
  
  save(out_file, 'eeg_data', 'Measurement', 'EEGinfo');
else
  save(out_file, 'Measurement', 'EEGinfo');

  vb_define_device;
  file_ext = FILE_EXT_BIN_CH_EEG;
  
  n_channel = EEGinfo.Nchannel;
  ch_name = EEGinfo.ChannelName;
  
  for i_ch = 1:n_channel

    datafile = sprintf('%s/%s.%s', bin_data_dir, ch_name{i_ch}, file_ext);
    cur_fid = fopen(datafile, 'wb');
    if cur_fid == -1
      error('(%s)cannot open file (%s)', mfilename, datafile);
    end

    fwrite(cur_fid, eeg_data(i_ch,:,:), data_type{i_ch});
    fclose(cur_fid);
    fprintf('--- %s\n', datafile);
  end
  
  % ----- Status channel
  n_data_type = length(data_type);
  if (n_data_type == size(eeg_data, 1)) && (n_data_type > n_channel)
    i_ch = i_ch + 1;
    datafile = sprintf('%s/%s.%s', bin_data_dir, STATUS_CH_LABEL, file_ext);
    cur_fid = fopen(datafile, 'wb');
    if cur_fid == -1
      error('(%s)cannot open file (%s)', mfilename, datafile);
    end

    fwrite(cur_fid, eeg_data(i_ch,:,:), data_type{i_ch});
    fclose(cur_fid);
    fprintf('--- %s\n', datafile);
  end
end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [in_files, out_file, ext_spec] = ...
  inner_check_arguments(in_files, out_file, ext_spec)
func_ = mfilename;

if isempty(in_files)
  error('(%s) in_fiels is a required parameter', func_);
end

if isempty(out_file)
  error('(%s) out_file is a required parameter', func_);
end

return;
%
% --- end of inner_check_arguments()

% --- inner_is_combinable()
%
function [chk_result] = inner_is_combinable(in_files)
func_ = mfilename;
chk_result = true;
n_file = length(in_files);

% --- read data
for i_file = 1:n_file
  cur_eeg = load(in_files{i_file}, 'Measurement', 'EEGinfo');
  
  % --- check variables of minimum format
  if ~isfield(cur_eeg, 'Measurement')
    fprintf('(%s) =FALSE= cannot find ''Measurement'' parameter\n', func_);
    chk_result = false;
    return;
  end
  measure{i_file} = cur_eeg.Measurement;
  
  if ~isfield(cur_eeg, 'EEGinfo')
    fprintf('(%s) =FALSE= cannot find ''EEGinfo'' parameter\n', func_);
    chk_result = false;
    return;
  end
  eeg_info(i_file) = cur_eeg.EEGinfo;
  
end  

% --- Measurement
for i_eeg = 1:n_file
  if ~isfield(eeg_info(i_eeg), 'Measurement')
    fprintf('(%s) EEGinfo(%d) does not have ''Measurement'' field\n', ...
      func_, i_eeg);
    chk_result = false;
    return;
  end
end
for i_eeg = 1:n_file
  if ~strcmpi(measure{i_eeg}, eeg_info(i_eeg).Measurement)
    fprintf('(%s) Measurement is different\n', func_);
    chk_result = false;
    return;
  end
end

for i_eeg = 2:n_file
  if ~strcmpi(measure{i_eeg-1}, measure{i_eeg})
    fprintf('(%s) Measurement is different\n', func_);
    chk_result = false;
    return;
  end
end

% --- EEGinfo - Promary variables - there or not there
for i_eeg = 1:n_file
  if ~isfield(eeg_info(i_eeg), 'Measurement')
    fprintf('(%s) EEGinfo(%d) does not have ''Measurement'' field\n', ...
      func_, i_eeg);
    chk_result = false;
    return;
  end
  if ~isfield(eeg_info(i_eeg), 'Device')
    fprintf('(%s) EEGinfo(%d) does not have ''Device'' field\n', ...
      func_, i_eeg);
    chk_result = false;
    return;
  end
  if ~isfield(eeg_info(i_eeg), 'Nchannel')
    fprintf('(%s) EEGinfo(%d) does not have ''Nchannel'' field\n', ...
      func_, i_eeg);
    chk_result = false;
    return;
  end
  if ~isfield(eeg_info(i_eeg), 'Nsample')
    fprintf('(%s) EEGinfo(%d) does not have ''Nsample'' field\n', ...
      func_, i_eeg);
    chk_result = false;
    return;
  end
  if ~isfield(eeg_info(i_eeg), 'Nrepeat')
    fprintf('(%s) EEGinfo(%d) does not have ''Nrepeat'' field\n', ...
      func_, i_eeg);
    chk_result = false;
    return;
  end
  if ~isfield(eeg_info(i_eeg), 'Pretrigger')
    fprintf('(%s) EEGinfo(%d) does not have ''Pretrigger'' field\n', ...
      func_, i_eeg);
    chk_result = false;
    return;
  end
  if ~isfield(eeg_info(i_eeg), 'SampleFrequency')
    fprintf('(%s) EEGinfo(%d) does not have ''SampleFrequency'' field\n', ...
      func_, i_eeg);
    chk_result = false;
    return;
  end
  if ~isfield(eeg_info(i_eeg), 'Coord')
    fprintf('(%s) EEGinfo(%d) does not have ''Coord'' field\n', ...
      func_, i_eeg);
    chk_result = false;
    return;
  end
end

% --- EEGinfo - each field of minimum format
% ----- EEGinfo.Measurement
for i_eeg = 2:n_file
  if ~strcmpi(eeg_info(i_eeg-1).Measurement, eeg_info(i_eeg).Measurement)
    fprintf('(%s) EEGinfo(%d).Measurement is different from EEGinfo(%d)\n', ...
      func_, i_eeg-1, i_eeg);
    chk_result = false;
    return;
  end
end

% ----- EEGinfo.Device
for i_eeg = 2:n_file
  if ~strcmpi(eeg_info(i_eeg-1).Device, eeg_info(i_eeg).Device)
    fprintf('(%s) EEGinfo(%d).Device is different from EEGinfo(%d)\n', ...
      func_, i_eeg-1, i_eeg);
    chk_result = false;
    return;
  end
end

% ----- EEGinfo.Nchannel
for i_eeg = 2:n_file
  if eeg_info(i_eeg-1).Nchannel ~= eeg_info(i_eeg).Nchannel
    fprintf('(%s) EEGinfo(%d).Nchannel is different from EEGinfo(%d)\n', ...
      func_, i_eeg-1, i_eeg);
    chk_result = false;
    return;
  end
end

% ----- EEGinfo.Nsample
for i_eeg = 2:n_file
  if eeg_info(i_eeg-1).Nsample ~= eeg_info(i_eeg).Nsample
    fprintf('(%s) EEGinfo(%d).Nsample is different from EEGinfo(%d)\n', ...
      func_, i_eeg-1, i_eeg);
    chk_result = false;
    return;
  end
end

% ----- EEGinfo.Pretrigger
for i_eeg = 2:n_file
  if eeg_info(i_eeg-1).Pretrigger ~= eeg_info(i_eeg).Pretrigger
    fprintf('(%s) EEGinfo(%d).Pretrigger is different from EEGinfo(%d)\n', ...
      func_, i_eeg-1, i_eeg);
    chk_result = false;
    return;
  end
end

% ----- EEGinfo.SampleFrequency
for i_eeg = 2:n_file
  if eeg_info(i_eeg-1).SampleFrequency ~= eeg_info(i_eeg).SampleFrequency
    fprintf( ...
      '(%s) EEGinfo(%d).SampleFrequency is different from EEGinfo(%d)\n', ...
      func_, i_eeg-1, i_eeg);
    chk_result = false;
    return;
  end
end

% ----- EEGinfo.Coord
for i_eeg = 2:n_file
  if ~isequal(eeg_info(i_eeg-1).Coord, eeg_info(i_eeg).Coord)
    fprintf('(%s) EEGinfo(%d).Coord is different from EEGinfo(%d)\n', ...
      func_, i_eeg-1, i_eeg);
    chk_result = false;
    return;
  end
end

return;
%
% --- end of inner_is_combinable()


% --- inner_solve_external_data()
%
function [ch_name, data_type, bin_data_dir, bin_data_dir_rel] = ...
  inner_solve_external_data(out_file, ext_spec, EEGinfo)

bin_data_dir = '';
bin_data_dir_rel = '';

ch_name = '';
data_type = '';

if isempty(out_file) || isempty(ext_spec)
  return;
end

if ~isfield(ext_spec, 'bin_data_dir') || isempty(ext_spec.bin_data_dir)
  return;
end

[out_path_base] = vb_get_file_parts(out_file);
if isempty(out_path_base)
  out_path_base = '.';
end

bin_data_dir = [out_path_base '/' ext_spec.bin_data_dir];

if ~isfield(EEGinfo, 'ChannelName') || isempty(EEGinfo.ChannelName)
  % --- EEGinfo.ChannelName is required if you store data externally
  bin_data_dir = [];
  return;
end
ch_name = EEGinfo.ChannelName;

if ~isfield(EEGinfo, 'DataType') || isempty(EEGinfo.DataType)
  bin_data_dir = [];
  return;
end
data_type = EEGinfo.DataType;

if exist(bin_data_dir, 'dir') ~= 7
  vb_mkdir(bin_data_dir);
end

bin_data_dir_rel = ext_spec.bin_data_dir;
return;
%
% --- end of inner_solve_external_data()

%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
