function [pick, Qpick, CoilWeight, Vcenter, result, channel_info] = ...
  vb_load_sensor(megfile, ch_type, ch_name, active_swt, method)
% Load sensor position, sensor direction, weight of MEG (meg.mat) and EEG (eeg.mat).
%
% This function supports three formats of megfile; Standard MEG format, Minimum MEG format and fileinfo format.
% By default, this function load only active sensors kept in MEGinfo.ActiveChannel
% in case of Standard MEG format or fileinfo.channel_id in case of fileinfo format.  
%
% [usage]
% 1. Load active sensors(Default) 
%   [pick, Qpick, CoilWeight, Vcenter, result, channel_info] = ...
%            vb_load_sensor(megfile);
% 2. Load all sensors including inactive sensors.
%   active_swt = false;
%   [pick, Qpick, CoilWeight, Vcenter, result, channel_info] = ...
%            vb_load_sensor(megfile, [], [], active_swt);
%
% [input]
%      megfile : <required> <<file>> MEG-MAT file or EEG-MAT file
%      ch_type : <optional> <<string>> channel type ['']
%              :  '' or 'MEG' : MEG channels of MEG
%              :        'REF' : REFFERENCE channels of MEG
%              :        'ALL' : MEG + REFFERENCE channels of MEG
%              :        'MAG' : magnetometer channels
%              :        'AXIAL'  : axial channels
%              :        'PLANAR' : planar channels
%              :  'MAG_X'     : Magnetometer OPM channel data toward X-axis (only QZFM device)
%              :  'MAG_Y'     : Magnetometer OPM channel data toward Y-axis (only QZFM device)
%              :  'MAG_Z'     : Magnetometer OPM channel data toward Z-axis (only QZFM device)
%              :  case-insensitive
%      ch_name : <optional> <<cell array>> channel name list [{''}(empty)]
%              :  If ch_type is specified, this is ignored.
%   active_swt : <optional> <<boolean>> switch to filter channel
%              :   if .ActiveChannel is valid [true]
%              :   if .ActiveChannel is invalid [false]
%              :    true) return only active channels
%              :   false) return all the channels
%       method : <optional> method of calculation
%              :  ['AVE'] : average
%              :   'TOP'  : use top data
%              :  this is valid for 'INIT' mode
% [output]
%         pick : n-th coil position  (Right-hand SPM coordinate) [m]
%              :  [Ncoil x 3]
%        Qpick : n-th coil direction (Right-hand SPM coordinate)
%              :  [Ncoil x 3]
%   CoilWeight : n-th coil weight for m-th channel [Nchannel x Ncoil]
%              : basis(channel,dipole) = CoilWeight * basis(coil,dipole)
%      Vcenter : Center of spherical head model [m] [1 x 3]
%       result : <<integer>> error infomation
%              :  In the case that meg_file is '.info.mat' file and method is
%              :  'AVE', this will become error struct.
%              :  fields are as follows:
%              :  These are errors between base data of each file and average.
%              :   .error_pick [n_ch x 3 x n_file]
%              :   .error_Qpick [n_ch x 3 x n_file] (if they are)
% channel_info : <<struct>> channel information of loaded data.
%              :  .Active [Nchannel x 1]
%              :  .Name   [Nchannel x 1]
%              :  .Type   [Nchannel x 1]
%              :  .ID     [Nchannel x 1]
%              : return empty when the loaded data contains reference sensor.
%                ('REF', 'ALL')
% [note]
%   if megfile is SBI data file or EEG-MAT file, REFFERENCE channel is empty
%
% [Active channel/Active trial information]
%   * Place
%    load(megfile, 'MEGinfo'); % EEG case, MEGinfo -> EEGinfo
%    MEGInfo.ActiveChannel : Active channel list (boolean array) [Nchannelx 1]
%           .ActiveTrial   : Active trial list   (boolean array) [Nrepeat x 1]
%   * Inactivate channel/trial
%     load_spec.ActiveChannel = false;
%     load_spec.ActiveTrial   = false;
%     [data] = vb_load_meg_data(meg_file, load_spec); % [Nchannel x Nsample x Ntrial]
%     MEGinfo.ActiveChannel = ones(size(data, 1), 1); % All channel active
%     MEGinfo.ActiveTrial   = ones(size(data, 3), 1); % All trial   active
%
%     % Inactivate channel/trial
%     MEGinfo.ActiveChannel(N) = 0; % Inactive data(N, :, :)
%     MEGinfo.ActiveTrial(M)   = 0; % Inactive data(:, :, M)
%     then save it. vb_save(megfile, 'MEGinfo');
%
% [see also]
%   vb_megfile_get_sensor_position_by_name
%   vb_megfile_get_sensor_position_by_type
%
% [history]
%   2006-11-01 (Sako) modified to be able to accept EEGinfo
%   2007-07-24 (Sako) added channel mode
%   2008-04-16 (Sako) added active_swt, result
%   2008-06-04 (Sako) changed default value of active_swt
%   2009-08-21 (????) supported 'INFO' as "Measurement" but 'mode' is ignored
%   2010-08-24 (Sako) added method argument
%   2011-06-02 (Sako) converted return values of vb_load_device to upper case
%   2011-07-26 (Sako) changed mode to ch_type, added ch_name as arguments
%   2017-03-06 (rhayashi) supported ch_type, ch_name for fileinfo format.
%   2017-04-15 (rhayashi) bugfix:ActiveChannel field work correctly.
%   2018-08-22 (Takeda) removed warning message about Vcenter and 
%                       supported NEUROMAG
%   2022-06-03 (k_suzuki) supported QZFM
%   2022-06-24 (k_suzuki) fixed bug when load Info file of QZFM device
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('megfile', 'var'), megfile = []; end
if ~exist('ch_type', 'var'), ch_type = []; end
if ~exist('ch_name', 'var'), ch_name = []; end
if ~exist('active_swt', 'var'), active_swt = []; end
if ~exist('method', 'var'), method = ''; end
[megfile, ch_type, ch_name, active_swt, method, result] = ...
  inner_check_arguments(megfile, ch_type, ch_name, active_swt, method);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
pick = [];
Qpick = [];
CoilWeight = [];
Vcenter = [];
channel_info = [];

if result ~= 0
  return;
end

func_ = mfilename;

measurement = vb_load_device(megfile);
Measurement = upper(measurement);

% return variables
% check measurement parameter
switch  Measurement
  case  'MEG'
    if ~isempty(ch_type)
      [pick, Qpick, CoilWeight, Vcenter, result, channel_info] = ...
        vb_megfile_get_sensor_position_by_type(megfile, ch_type, active_swt);

    elseif ~isempty(ch_name)
      [pick, Qpick, CoilWeight, Vcenter, result, channel_info] = ...
        vb_megfile_get_sensor_position_by_name(megfile, ch_name, active_swt);
      
    else
      error('(%s) Both ch_type and ch_name are invalid', mfilename);
    end

  case  'EEG'
    % for EEG
    if strcmp(ch_type, 'REF')
      % return empty
      return;
    else
      [pick, Vcenter, result, channel_info] = vb_load_eeg_sensor(megfile, active_swt);
      return;
    end
  
  case  'INFO'
    load(megfile, 'fileinfo');

    switch method
      case 'AVE'
        [pick, Qpick, CoilWeight, Vcenter, result] = ...
          inner_calc_average(fileinfo);
      case 'TOP'
        [pick, Qpick, CoilWeight, Vcenter, result] = ...
          vb_load_sensor(fileinfo.filename{1}, [], [], false);
      otherwise
        msg = sprintf('(%s) undefined method : %s --> average', ...
          'vb_load_sensor', method);
        vb_disp(msg);
        [pick, Qpick, CoilWeight, Vcenter, result] = ...
          inner_calc_average(fileinfo);
    end

    % Loaded sensor info
    info    = vb_load_measurement_info(megfile);
    ch_info = vb_info_get_channel_info(info);

    % ch_info.ID : is IDs for loaded information(pick, Qpick, CoilWeight).
    % To access the fileinfo.channel_id data, search and find index of ch_info.ID
    if isfield(fileinfo, 'channel_id')
        active_ix = vb_util_get_index(ch_info.ID, fileinfo.channel_id);
    else
        active_ix = find(fileinfo.ActiveChannel == 1);
    end
    
    % reset active information
    ch_info.Active = zeros(length(ch_info.ID), 1);
    ch_info.Active(active_ix) = true;

    if active_swt
        chan_info_ix = active_ix;
    else
        chan_info_ix = [1:length(ch_info.ID)]';
    end

    % specified channel type and name
    chan_info_type = ch_info.Type(chan_info_ix);
    chan_info_name = ch_info.Name(chan_info_ix);

    % 'chan_info_ix' modification by channel type
    ignore_ch_name = false; % alternative (both specified case: ch_type is used.)
    if strcmpi(ch_type, 'MAG')
        chan_info_ix = chan_info_ix(find(chan_info_type == 1)); % MAG = 1;
        ignore_ch_name = true;
    elseif strcmpi(ch_type, 'AXIAL')
        chan_info_ix = chan_info_ix(find(chan_info_type == 2)); % AXIAL = 2;
        ignore_ch_name = true;
    elseif strcmpi(ch_type, 'PLANAR')
        chan_info_ix = chan_info_ix(find(chan_info_type == 3)); % PLANAR = 3;
        ignore_ch_name = true;
    elseif strcmpi(ch_type, 'MAG_X')
        chan_info_ix = chan_info_ix(find(chan_info_type == 4)); % MAG_X = 4;
        ignore_ch_name = true;
    elseif strcmpi(ch_type, 'MAG_Y')
        chan_info_ix = chan_info_ix(find(chan_info_type == 5)); % MAG_Y = 5;
        ignore_ch_name = true;
    elseif strcmpi(ch_type, 'MAG_Z')
        chan_info_ix = chan_info_ix(find(chan_info_type == 6)); % MAG_Z = 6;
        ignore_ch_name = true;
    elseif strcmpi(ch_type, 'MEG')
        % fall through
        % NOTE:default ch_type is 'MEG' even if the data file is for EEG.
    else
        error('unknown channel type is supported. : %s', ch_type);
    end

    % 'chan_info_ix' modification by channel name
    if ~isempty(ch_name)  && ignore_ch_name == false
        ix = [];
        for k=1:length(ch_name)
            add_ix = strmatch(ch_name{k}, chan_info_name, 'exact');
            if isempty(add_ix)
                % display not found channel name.
                fprintf('%s ', ch_name{k});
            end
            ix = [ix; add_ix];
        end
        if length(ix) ~= length(ch_name)
            fprintf('data not loaded.\n');
        end
        chan_info_ix = chan_info_ix(ix);
    end

    % get sensor info through the 'chan_info_ix'.
    if isempty(CoilWeight)
        pick = pick(chan_info_ix,:);
    else
        weight  = CoilWeight(chan_info_ix,:);
        ix_coil = find( sum(abs(weight), 1) > 0);
        
        pick = pick(ix_coil,:);
        Qpick = Qpick(ix_coil,:);
        CoilWeight = weight(:,ix_coil);
    end

    % channel_info
    channel_info.Active = ch_info.Active(chan_info_ix);
    channel_info.Name   = ch_info.Name(chan_info_ix);
    channel_info.Type   = ch_info.Type(chan_info_ix);
    channel_info.ID     = ch_info.ID(chan_info_ix);

otherwise
    msg = sprintf('(%s)unknown Measurement : %s', func_, Measurement);
    vb_disp(msg);
    result = 2;
end

% --- solve empty center problem
if isempty(Vcenter) && ~isempty(pick)
    Vcenter = sum(pick, 1) ./ size(pick, 1);
end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [megfile, ch_type, ch_name, active_swt, method, result] = ...
  inner_check_arguments(megfile, ch_type, ch_name, active_swt, method)
func_ = mfilename;
result = 0; % success

if isempty(megfile)
  msg = sprintf('(%s)megfile is a required parameter', func_);
  vb_disp(msg);
  result = 1; % error code 1 means bad megfile
  return;
end

if exist(megfile, 'file') ~= 2
  msg = sprintf('(%s)cannot find megfile : %s', func_, megfile);
  vb_disp(msg);
  result = 1;
  return;
end

% --- ch_name must be a cell array
if ischar(ch_name)
  ch_name = {ch_name};
end

if iscellstr(ch_name) && isempty(ch_name{1})
  ch_name = [];
end

if isempty(ch_type)
  if isempty(ch_name)
    ch_type = 'MEG';
  end
end

% case-insensitive
ch_type = upper(ch_type);

if isempty(active_swt)
  if vb_msrmnt_active_channel_is_valid(megfile)
    active_swt = true;
  else
    active_swt = false;
  end
end

if isempty(method)
  method = 'AVE';
end
method = upper(method);
return;
%
% --- end of inner_check_arguments()

% --- inner_calc_average()
%
function [pick, Qpick, CoilWeight, Vcenter, result] = ...
          inner_calc_average(fileinfo)
%
n_file = length(fileinfo.filename);

pick = 0;
Qpick = 0;
CoilWeight = 0;
Vcenter = 0;

for i_file = 1:n_file
  cur_file = fileinfo.filename{i_file};
  [a, b, c, d] = vb_load_sensor(cur_file, [], [], false);
  
  pick = pick + a;
  Qpick = Qpick + b;
  CoilWeight = CoilWeight + c;
  Vcenter = Vcenter + d;
end

pick = pick / n_file;
Qpick = Qpick / n_file;
CoilWeight = CoilWeight / n_file;
Vcenter = Vcenter / n_file;

error_pick = zeros(size(pick,1), size(pick,2), n_file);
error_Qpick = zeros(size(Qpick,1), size(Qpick,2), n_file);

% --- error
for i_file = 1:n_file
  cur_file = fileinfo.filename{i_file};
  [a, b, c, d] = vb_load_sensor(cur_file, [], [], false);
  error_pick(:,:,i_file) = pick - a;
  error_Qpick(:,:,i_file) = Qpick - b;
end

result = struct;
result.error_pick = error_pick;
result.error_Qpick = error_Qpick;
return;
%
% --- end of inner_calc_average()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %
