function [Jall,ix_act] = vb_load_current_j_tr(currfile,trial,ix_area,verbose)
% load estimated current
%
% USAGE:
%  [Jall,ix_act] = vb_load_current_j_tr(currfile,trial,ix_area,verbose)
%
% --- INPUT:
%  currfile  - current file name
% --- Optional INPUT:
%  curr_type - current type 
%            = 0 : J-current : original vertex current [Default]
%            = 1 : Z-current : internal current 
%                              correspond to locally spread source
%     If currfile is old format, curr_type is neglected
%     and Jact is returned
%  trial   : trial number to load 
%            if empty, all trials are loaded
%  ix_area : (absolute) vertex index to load
%            if empty, all vertices are loaded
%  verbose : 0 (No print) , 1 ('.') , 2 ('session , trial')
% --- OUTPUT:
% ix_act  : Vertex index corresponding to current
% Jact    : J-current in focal region
%   Jact(Nact,Nsample,Ntrials)  
%     Nact     = Lact * Nvact,  Nvact = length(ix_act)
%     Nsample  : # of time sample, 
%     Ntrials  : # of trials in all session]
%
%   Jact( n + Nvact*(i-1), t, :) 
%    current at the vertex 'ix_act(n)', 'i-th' direction, time index 't'
%
%   Time sample in MEG data : t_meg = Tsample(t)
%   Time (ms)   in MEG data : t_ms  = (t_meg - Pretrigger)*(1000/SampleFreq)
%
%
% 2007-3-5 Masa-aki Sato
% 2008-7-9 Masa-aki Sato
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if ~exist('ix_area','var'), ix_area = []; end;
if ~exist('trial','var'),   trial = []; end;
if ~exist('verbose','var'), verbose=2; end;

load(currfile)

[Jinfo] = vb_load_current_info(currfile);

if ~isfield(Jinfo,'jactdir') 
	error(['No trial data directory in' currfile])
end

jactdir = [fileparts(currfile)  '/' Jinfo.jactdir];

if exist(jactdir,'dir')==0,
	error('No trial data directory')
end

if ~exist('Jact','var') 
	error(['No J-current in ' currfile])
end

[NJ, T] = size(Jact);

% Select vertex index 'ix_area' within the active current region
[ix_act, jx_act] = vb_current_area_info(Jinfo, ix_area);

Ntrial  = sum(Jinfo.Ntrial);

if isempty(trial), trial=1:Ntrial; end;

Ntrial = length(trial);
NJ = length(jx_act);

Jall = zeros(NJ, T, Ntrial);

session = [];
ntrial  = [];

% Session & Trial number list
for n=1:Jinfo.Nsession
	Ntry = Jinfo.Ntrial(n);
	session = [session, repmat(n,1,Ntry)];
	ntrial  = [ntrial, 1:Ntry];
end

% Trial loop 
for n=1:Ntrial
	nt = trial(n);
	
	if verbose==1,
		fprintf('.')
	elseif verbose==2,
		fprintf('Session: %04d , Trial: %04d\n',session(nt), ntrial(nt))
	end
	
	fname = sprintf('data_s%04dt%04d', session(nt), ntrial(nt));
	load([jactdir '/' fname], 'Jact');
	
	Jall(:,:,n) = Jact(jx_act,:);
end

if verbose==1,
	fprintf('\n')
end
