function [Jinfo] = vb_load_current_info(curr_file)
% Load information of cortical current file. 
%
% [syntax]
% Jinfo = vb_load_current_z(curr_file)
%
% [input]
% curr_file: <<string>> Cortical current file (.curr.mat). 
%
% [output]
% Jinfo: <<struct>> Information of cortical current.
% --- fields of Jinfo
%  version   : <<string>> Version of cortical current file.
%  curr_type : <<string>> 'Z-current'. It can be 'J-current' for VBMEG
%              version 0.8 or older. 
%  Wact      : <<float matrix>> Smoothing Gaussian filter, mapping from
%              Z-current to J-current. 
%  ix_act    : <<int vector>>: Vertex indices of Z-current.
%  ix_act_ex : <<int vector>>: Vertex indices of J-current.
%  Lact      : <<int>> Number of current direction at one vertex. 
%  Tsample   : <<int vector>> Time sample indices of the original MEG
%              data. length(Tsample) == size(Zact,2) == size(Jact,2). 
%  Tmsec     : <<float vector>> Time in msec. 
%  SampleFreq: <<float>> Sample frequency of cortical current, not
%              original M/EEG signal [Hz]. 
%  Pretrigger: <<int>> Time points of the length of the pretrigger
%              period of cortical current data. It is neither actual time
%              nor time points of the original M/EEG signal. 
%  Ntrial    : <<int>> Number of trials of estimated current. 
%  patch_norm: <<bool>> Cortical current is patch size normalized
%              (Am/m^2) or not (Am). 
% ---
%
% [history]
% 2010-12-06 taku-y
%  [enhancement] Cortical current file version 0.9 or later supported. 
% 2011-01-27 taku-y
%  [debug] Jinfo.patch_norm set. 
% 2011-08-23 taku-y
%  [debug] If Jinfo.patch_norm does not exist, bayes_parm.patch_norm is
%  used. 
% 2012-02-03 taku-y
%  [minor] Jinfo.Tix was added.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

load(curr_file);

if isfield(Jinfo,'version'), 
  v = Jinfo.version;
else
  v = '0.7';
end

if vb_version_cmp(v,'>','1.0-0.a.0'), 
  vb_struct2vars(Jinfo,{'version','curr_type','ix_act','ix_act_ex', ...
                      'Wact','Lact','Tsample','Tmsec','SampleFreq', ...
                      'Pretrigger','Ntrial','patch_norm','Tix'});
  clear Jinfo;
  Jinfo.version    = v;
  Jinfo.curr_type  = curr_type;
  Jinfo.ix_act     = ix_act;
  Jinfo.ix_act_ex  = ix_act_ex;
  Jinfo.Wact       = Wact;
  Jinfo.Lact       = Lact;
  Jinfo.Tsample    = Tsample;
  Jinfo.Tmsec      = Tmsec;
  Jinfo.SampleFreq = SampleFreq;
  Jinfo.Pretrigger = Pretrigger;
  Jinfo.Ntrial     = Ntrial;
  Jinfo.patch_norm = patch_norm;
  Jinfo.Tix        = Tix;
  
  if isempty(patch_norm) & isfield(bayes_parm,'patch_norm'), 
    Jinfo.patch_norm = bayes_parm.patch_norm;
  end
elseif vb_version_cmp(v,'>=','0.9-0.a.0'), 
  vb_struct2vars(Jinfo,{'version','curr_type','ix_act','ix_act_ex', ...
                      'Wact','Lact','Tsample','Tmsec','SampleFreq', ...
                      'Pretrigger','Ntrial','patch_norm'});
  clear Jinfo;
  Jinfo.version    = v;
  Jinfo.curr_type  = curr_type;
  Jinfo.ix_act     = ix_act;
  Jinfo.ix_act_ex  = ix_act_ex;
  Jinfo.Wact       = Wact;
  Jinfo.Lact       = Lact;
  Jinfo.Tsample    = Tsample;
  Jinfo.Tmsec      = Tmsec;
  Jinfo.SampleFreq = SampleFreq;
  Jinfo.Pretrigger = Pretrigger;
  Jinfo.Ntrial     = Ntrial;
  Jinfo.patch_norm = patch_norm;
  
  if isempty(patch_norm) & isfield(bayes_parm,'patch_norm'), 
    Jinfo.patch_norm = bayes_parm.patch_norm;
  end
else % otherwise
  if ~exist('bayes_parm','var'),
    bayes_parm = [];
  end;

  if ~exist('Jinfo','var')
    if exist('ix_act','var') 
      Jinfo.ix_act = ix_act;
    elseif exist('ix0','var') 
      Jinfo.ix_act = ix0;
    else
      Jinfo.ix_act = [];
    end;
	
    if exist('Lact','var')
      Jinfo.Lact = Lact;
    elseif exist('vb_parm','var')
      Jinfo.Lact = vb_parm.Norient;
    else
      Jinfo.Lact = 1;
    end;
    
    if exist('Tsample','var')
      Jinfo.Tsample = Tsample;
    else
      if ~isempty(Jact)
        T = size(Jact,2);
      elseif ~isempty(Jbck)
        T = size(Jbck,2);
      else
        error(['There is no Jact in ' curr_file])
      end;
      
      if ~exist('Tstart','var') & exist('bayes_parm','var')
        Tstart  = bayes_parm.twin_meg(1);
      else
        Tstart = 1; 
      end
      if ~exist('Tend','var') & exist('bayes_parm','var')
        Tend = bayes_parm.twin_meg(2);
      else
        Tend = Tstart + T - 1; 
      end;
      
      step = (Tend - Tstart + 1)/T;
      Jinfo.Tsample = Tstart:step:Tend;
    end;
  end;

  if exist('JactInfo','var')
    Jinfo.jactdir  = JactInfo.jactdir;
    Jinfo.Ntrial   = JactInfo.Ntrial  ;
    Jinfo.Nsession = JactInfo.Nsession;
  end;

  if ~isfield(Jinfo,'SampleFreq')
    if exist('MEGinfo','var')
      Jinfo.SampleFreq = MEGinfo.SampleFreq;
    else
      MEGinfo = [];
      Jinfo.SampleFreq = [];
    end;
  end;
  
  if ~isfield(Jinfo,'Pretrigger')
    if exist('MEGinfo','var')
      Jinfo.Pretrigger = vb_meginfo_get_pre_trigger(MEGinfo);
    else
      MEGinfo = [];
      Jinfo.Pretrigger  = [];
    end;
  end;
  
  Jinfo.Tmsec = time_msec( Jinfo.Tsample, Jinfo);

  if exist('Zact','var') 
    Jinfo.curr_type = 'Z-current';
  elseif exist('Jact','var') 
    Jinfo.curr_type = 'J-current';
  elseif exist('J','var')
    Jinfo.curr_type = 'J-current';
  else
    Jinfo.curr_type = '';
  end;
end;

return;

%
% --- Inner function
%
function	Tms = time_msec( t, para )
%   Time (ms)  : Tms  = (t - Pretrigger)*(1000/SampleFreq)

if isempty(para.Pretrigger), para.Pretrigger = 0; end;
if isempty(para.SampleFreq), para.SampleFreq = 1000; end;

Tms = (t - para.Pretrigger)*(1000/para.SampleFreq);
