function [ch_info] = vb_load_channel_info(datafile, ch_class, ch_user)
% return channel information
% [usage]
%   [ch_info] = vb_load_channel_info(datafile, ch_class, ch_user)
% [input]
%   datafile : <required> <<file>> meg.mat or eeg.mat file
%   ch_class : <optional> channel class to load ['MEG(EEG)']
%            :  *** Except for YOKOGAWA, this parameter is invalid. ***
%            :  You can demand "channel class" by two ways.
%            :  1) <<string>> the defined key words
%            :     'MEG'|'EEG'|'EXTRA'|'REFERENCE'|'ALL'|'MAG'|'AXIAL'|'PLANAR'|
%                  'MAG_X'|'MAG_Y'|'MAG_Z'|'USER'
%            :  2) <<numeric>> channel types defined by device
%    ch_user : <conditionally required> channel list you want, Name or ID.
%            :  {n_channel x 1}(Name) or [n_channel x 1](ID)
%            :  If ch_class is 'USER', this is required.
%            :  If ch_class is not 'USER', this is ignored.
%            :  is numerical list ------> ID list
%            :  is not numerical list --> Name list
% [output]
%    ch_info : <<struct>> the fields are as follows:
%            :  .Active : list of active flag   [Nchannel x 1 boolean]
%            :  .Name   : list of channel label {Nchannel x 1}
%            :  .ID     : list of channel index [Nchannel x 1]
%            :  .Type   : list of channel type  [Nchannel x 1]
%
% [note]
%   @see vb_info_make_channel_info.m
% [history]
%   2009-07-27 (Sako) initial version
%   2009-08-19 (Sako) modified to make data without ChannelInfo field
%   2009-10-13 (Sako) modified help
%   2009-10-14 (Sako) added 'USER' type
%   2011-06-01 (Sako) converted return values of vb_load_device to upper case
%   2018-08-28 (rhayashi) ch_class 'MAG' is supported.
%   2022-06-03 (k_suzuki) ch_class 'MAG_X', 'MAG_Y', 'MAG_Z' are supported
%   2022-06-20 (k_suzuki) corrected a bug when measurement of datafile is 'INFO'
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('datafile', 'var'), datafile = ''; end
if ~exist('ch_class', 'var'), ch_class = []; end
if ~exist('ch_user', 'var'), ch_user = []; end
[datafile, ch_class, ch_user] = ...
  inner_check_arguments(datafile, ch_class, ch_user);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
info = vb_load_measurement_info(datafile);
if ~isfield(info, 'ChannelInfo')
  measurement = vb_load_device(datafile);
  info.ChannelInfo = vb_info_make_channel_info(info, measurement);
end

if isempty(ch_class)
  % only info of ChannelInfo
  ch_info = vb_info_get_channel_info(info, 0);
  return;
end

% --- with extracting conditions
measurement = vb_load_device(datafile);

% If Measurement of datafile is 'INFO', then use Measurement of 1st file in INFO file
if strcmp(measurement, 'INFO')
    M = load(datafile, 'fileinfo');
    measurement = vb_load_device(M.fileinfo.filename{1});
end

if ischar(ch_class)
  ch_info = inner_extract_channel_by_name(info, measurement, ch_class, ch_user);
else
  ch_info = inner_extract_channel_by_type(info, measurement, ch_class);
end
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [datafile, ch_class, ch_user] = ...
  inner_check_arguments(datafile, ch_class, ch_user)
func_ = mfilename;

if isempty(datafile)
  error('(%s) datafile is a required parameter', func_);
end

if exist(datafile, 'file') ~= 2
  error('(%s) cannot find datafile : %s', func_, datafile);
end

measurement = vb_load_device(datafile);
const = vb_define_const(measurement);
if isempty(ch_class)
  % require no action
elseif strcmp(ch_class, const.DATA_TYPE_USER)
  if isempty(ch_user)
    error('(%s) ch_user is a required parameter when ch_type is ''%s''', ...
      func_, const.DATA_TYPE_USER);
  end
end
return;
%
% --- end of inner_check_arguments()


% --- inner_extract_channel_by_name()
%
function ch_info = ...
  inner_extract_channel_by_name(info, measurement, ch_class, ch_user)
const = vb_define_const(measurement);

Measurement = upper(measurement);

if strcmp(Measurement, 'MEG')
  switch ch_class
    case const.DATA_TYPE_ALL
      target_ch = vb_meginfo_get_channel_label_whole(info);

    case const.DATA_TYPE_MAIN
      target_ch = vb_meginfo_get_channel_label_meg(info);

    case const.DATA_TYPE_EXTRA
      target_ch = vb_meginfo_get_channel_label_extra(info);

    case const.DATA_TYPE_REFERENCE
      target_ch = vb_meginfo_get_channel_label_refmg(info);

    case const.DATA_TYPE_MAG
      target_ch = vb_info_get_channel_label_by_type(info, 1);
    
    case const.DATA_TYPE_AXIAL
      target_ch = vb_info_get_channel_label_by_type(info, 2);
    
    case const.DATA_TYPE_PLANAR
      target_ch = vb_info_get_channel_label_by_type(info, 3);
      
    case const.DATA_TYPE_MAG_X
      target_ch = vb_info_get_channel_label_by_type(info, 4);
    case const.DATA_TYPE_MAG_Y
      target_ch = vb_info_get_channel_label_by_type(info, 5);
    case const.DATA_TYPE_MAG_Z
      target_ch = vb_info_get_channel_label_by_type(info, 6);
    
    case const.DATA_TYPE_USER
      target_ch = inner_get_channel_label_user(info, ch_user);
      
    otherwise
      warning('(%s) unknown ch_class : %s\n', mfilename, ch_class);
      ch_info = [];
      return;
  end
  
elseif strcmp(Measurement, 'EEG')
  switch ch_class
    case const.DATA_TYPE_ALL
      target_ch = [vb_eeginfo_get_channel_label(info); ...
        vb_eeginfo_get_channel_label_extra(info)];

    case const.DATA_TYPE_MAIN
      target_ch = vb_eeginfo_get_channel_label(info);

    case const.DATA_TYPE_EXTRA
      target_ch = vb_eeginfo_get_channel_label_extra(info);

    case const.DATA_TYPE_USER
      target_ch = inner_get_channel_label_user(info, ch_user);
      
    otherwise
      warning('(%s) unknown ch_class : %s\n', mfilename, ch_class);
      ch_info = [];
      return;
  end
else
  warning('(%s) unknown measurement : %s\n', mfilename, measurement);
  ch_info = [];
  return;
end

% --- get all channels info
org_ch_info = vb_info_get_channel_info(info, 2);
[idx] = vb_util_get_index(org_ch_info.Name, target_ch);

ch_info.Active = org_ch_info.Active(idx,:);
ch_info.Name   = org_ch_info.Name(idx,:);
ch_info.ID     = org_ch_info.ID(idx,:);
ch_info.Type   = org_ch_info.Type(idx,:);
return;
%
% --- end of inner_extract_channel_by_name()


% --- inner_extract_channel_by_type()
%
function ch_info = inner_extract_channel_by_type(info, measurement, ch_class)
target_ch = vb_info_get_channel_label_by_type(info, ch_class);

if isempty(target_ch)
  warning('(%s) cannot find channels of type : %d\n', mfilename, ch_class);
  ch_info = [];
  return;
end

% --- get all channels info
org_ch_info = vb_info_get_channel_info(info, 2);
[idx] = vb_util_get_index(org_ch_info.Name, target_ch);

ch_info.Active = org_ch_info.Active(idx,:);
ch_info.Name   = org_ch_info.Name(idx,:);
ch_info.ID     = org_ch_info.ID(idx,:);
ch_info.Type   = org_ch_info.Type(idx,:);
return;
%
% --- end of inner_extract_channel_by_type()

% --- inner_get_channel_label_user()
%
function target_ch = inner_get_channel_label_user(info, ch_user)
ch_info = vb_info_get_channel_info(info, 2);

if isnumeric(ch_user)
  [idx_ch] = vb_util_get_index(ch_info.ID, ch_user);
else
  [idx_ch] = vb_util_get_index(ch_info.Name, ch_user);
end
target_ch = ch_info.Name(idx_ch, :);
return;
%
% --- end of inner_get_channel_label_user()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
