function [N,C] = vb_fs_load_normal(ascfile,xfmfile)
% Load surface normal data created by FreeSurfer and 'mris_convert' 
% (.curv.asc file). Vertex number is reduced. Inverse talairach 
% transformation is applied. 
%
% --- Syntax
% [N,C] = vb_fs_load_surface(ascfile,xfmfile)
%
% --- Input
% ascfile: include cortical structure created by FreeSurfer
% xfmfile: Linear transformation file (Talairach transformation)
% N: Unit normal vector
% C: Curvature
%
% - This program supports linear transformation only (.xfm). 
%
% --- History
% 2006-01-30 Taku Yoshioka
% 2008-10-16 Taku Yoshioka
%   Input variable 'xfmfile' can be empty. 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Load normal data
n = load('-ascii',ascfile); % inner function
N = n(:,2:4);
C = n(:,5);
ix = find(C==0);
C(ix) = 1;

if nargin>=2 & ~isempty(xfmfile), 
  % Load .xfm file
  xfm = load_xfm(xfmfile);

  % Inverse transformation
  xfm_inv = inv(xfm);
  Next = [N'; zeros(1,size(N,1))];
  N = (xfm_inv*Next)';
  N = N(:,1:3).*repmat(sign(C),[1 3]);
end;

% Normalization
Nnorm = sqrt(sum(N.^2,2)); 
N = N./repmat(Nnorm,[1 3]);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Inner functions
%
function xfm = load_xfm(xfmfile)

% Open file
fid = fopen(xfmfile,'r');

% Search line starts with 'Linear_Transform'
chk = 0;
while chk==0, 
  str = fgetl(fid);
  if strcmp(str,'Linear_Transform ='), chk = 1; end; 
end

% Read three lines (transformation matrix)
xfm = zeros(4,4); 
for i=1:3, xfm(i,:) = str2num(fgetl(fid)); end;
xfm(4,:) = [0 0 0 1];

fclose(fid); 

