function	[XY, Pindx, FLR, VLR, XYLR, IDLR] = ...
					vb_inflate_to_flat_func(V,F,Xdim,xcenter)
% Project vertex point into 2D-plane
% XY	: Projected spherical coordinate 
% Pindx : Partition index (L/R & U/D) for each vertex
% Pindx(j) = [ 1 2 3 4 ]  :  { LU;  LD;  RU;  RD } (Left/Right & Up/Down)
%       
% FLR   = { FLU;  FLD;  FRU;  FRD }; % Face index for each part
% VLR   = { VLU;  VLD;  VRU;  VRD }; % Vertex coordinate for each part
% XYLR  = {XYLU; XYLD; XYRU; XYRD }; % Projected coordinate for each part
% IDLR  = { ILU;  ILD;  IRU;  IRD }; % Vertex index for L/R & U/D
%
% V     : Inflate brain coordinate
% F3L   : Face index for left brain
% F3R   : Face index for right brain
% Nleft : Number of point in the left brain
% Xdim  : Projection direction
%
% Ver 1.0 written by M. Sato  2003-3-15
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

F3R   = F.F3R;
F3L   = F.F3L;
Nleft = F.NdipoleL;

% Where the center of sphere is set
if nargin<4, xcenter = 0.1; end;

Xrate   = xcenter*[ 1  -1  1 -1 ];

% Set Projection direction
switch	Xdim,
case	1,
	prj_mode = { 'x' ; '-x' ;  'x' ; '-x'};
case	2,
	prj_mode = { 'y' ; '-y' ;  'y' ; '-y'};
case	3,
	prj_mode = { 'z' ; '-z' ;  'z' ; '-z'};
end;

Npoint = size(V,1);
Nright = Npoint - Nleft;

XY	   = zeros(Npoint,3); % Projected spherical coordinate 
Pindx  = zeros(Npoint,1); % Partition index (L/R & U/D) for each vertex
%      = [ 1 2 3 4 ]    : { LU;  LD;  RU;  RD } (Left/Right & Up/Down)

VL	   = V(1:Nleft,:);
VR	   = V((Nleft+1):Npoint,:);

if Nleft==0,
	VLmean = zeros(1,3);
else
	VLmean = sum(VL)/Nleft;
end;

if Nright==0,
	VRmean = zeros(1,3);
else
	VRmean = sum(VR)/Nright;
end;

% Set origin to the center of gravity
VL	   = VL	- repmat(VLmean ,[ Nleft 1]);
VR	   = VR	- repmat(VRmean ,[ Nright 1]);

% Select Up/Down(Front/Back) part
LU	 = find( VL(:,Xdim) >= 0 );
LD	 = find( VL(:,Xdim) <  0 );
RU	 = find( VR(:,Xdim) >= 0 );
RD	 = find( VR(:,Xdim) <  0 );

VLU  = VL(LU,:);
VLD  = VL(LD,:);
VRU  = VR(RU,:);
VRD  = VR(RD,:);

RU	 = RU+Nleft;
RD	 = RD+Nleft;

IDLR = { LU;  LD;  RU;  RD }; % Index for L/R & U/D
VLR  = {VLU; VLD; VRU; VRD }; % Vertex coordinate for each part
XYLR = cell(4,1);             % Projected coordinate for each part

% Transform coordinate into spherical coordinate in each part
for j=1:4,
	VLRj   = VLR{j};
	IDj    = IDLR{j};
	NID    = size(VLRj,1);
	
	% Set center of sphere inside of selected part
	Xzero  = (max(VLRj(:,Xdim)) -min(VLRj(:,Xdim)))*Xrate(j);

	VLRj(:,Xdim) = VLRj(:,Xdim) - repmat(Xzero,NID,1);
	
	% Spherical coordinate projection
	[ Xj , Yj ]  = vb_cart2theta_phi( VLRj , prj_mode{j} );

	VLR{j}	   = VLRj;
	Pindx(IDj) = j;
	XY(IDj,1)  = Xj;
	XY(IDj,2)  = Yj;
	XYLR{j}    = [Xj , Yj];
end;

% Select face inside of each projrcted part
FLU = [];
FLD = [];
FRU = [];
FRD = [];

Lface=length(F3L);
Rface=length(F3R);

for n=1:Lface,
	ix=F3L(n,:);
	pix=Pindx(ix);
	
	if (pix(1)==1)&(pix(2)==1)&(pix(3)==1),
		FLU = [ FLU; ix];
	end;
	
	if (pix(1)==2)&(pix(2)==2)&(pix(3)==2),
		FLD = [ FLD; ix];
	end;
	
end;

for n=1:Rface,
	ix=F3R(n,:);
	pix=Pindx(ix);
	
	if (pix(1)==3)&(pix(2)==3)&(pix(3)==3),
		FRU = [ FRU; ix];
	end;
	
	if (pix(1)==4)&(pix(2)==4)&(pix(3)==4),
		FRD = [ FRD; ix];
	end;
	
end;

FLR = { FLU; FLD; FRU; FRD }; % Face index for each part
