function vb_import_fsannot_file(brain_file,atlas_id,antl_file,antr_file, prefix)
% Import FreeSurfer annotation file. 
%
% [syntax]
% vb_import_fsannot_file(brain_file,area_file,antl_file,antr_file)
%
% [input]
% brain_file: <<string>> Cortical surface model file (.brain.mat).
% atlas_id  : <<string>> Cortical area file (.area.mat)
% antl_file : <<string>> FS annotation file for lh.
% antr_file : <<string>> FS annotation file for rh.
% prefix    : <<int>>    optional
%                        (deafult) = 0 : none is added to the area/act name.
%                                  = 1 : lh_/rh_ is added to the area/act name.
%                                  
% [output]
% Cortical areas are stored into the cortical area file. These areas are
% associated with labels defined with FS. 
%
% [example]
% >> brain_file = '3D.brain.mat';
% >> atlas_id   = 'Destrieux';
% >> antl_file  = 'lh.aparc.a2009s.annot';
% >> antr_file  = 'rh.aparc.a2009s.annot';
% >> vb_import_fsannot_file(brain_file,atlas_id,antl_file,antr_file)
%
% [history]
% 2011-06-17 taku-y
% 2012-02-20 taku-y
%  [debug] Due to using information in fsannot file without modification,
%          color value could be out of range from 0.0 to 1.0. taken
%          directly from fsannot file. This bug has been fixed. 
% 2019-10-07 rhayashi
%          atlas_id is added to make both area file and act file.
% 2021-3-01 rhayashi
%          prefix argument is added.


%%%%%%%%%%%%%%%%%%%
% filename prepare
EXT_brain = '.brain.mat';
brain_id  = brain_file(1:findstr(brain_file,EXT_brain)-1);

save_areafile  = [brain_id '_' atlas_id '.area.mat'];
save_atlasfile = [brain_id '_' atlas_id '.act.mat'];

if exist(save_atlasfile,'file')
	delete(save_atlasfile);
end
if exist(save_areafile,'file')
	delete(save_areafile);
end

if ~exist('prefix', 'var')
    prefix = 0;
end
if prefix
    prefix_l = 'lh_';
    prefix_r = 'rh_';
else
    prefix_l = '';
    prefix_r = '';
end

%%%%%%%%%%%%%%%%%
% start importing

% load .brain.mat
[Nvertex, tmp2, BV_index, Vinfo] = vb_load_cortex_info(brain_file, 'subj');
Ncortex = length([Vinfo.cortexL; Vinfo.cortexR]);

% load annotation file for lh
[v,l,c] = read_annotation(antl_file);

% make label data that of VBMEG brain model for lh
[tmp,ix1,ix2] = intersect(BV_index.Left,v+1);
l = l(ix2);
l(ix1) = l;
Ileft0 = length(v);
Ileft = length(BV_index.Left);

%
% make area for lh
%


for i=1:c.numEntries
  ix = find(l==c.table(i,5));
  if ~isempty(strfind(c.struct_names{i}, 'FreeSurfer_Defined_Medial_Wall')) || ...
      strcmpi(c.struct_names{i}, 'Unknown') || ...
      isempty(ix)
      continue;
  end
  Area.key = [prefix_l c.struct_names{i}];
  Area.Iextract = ix;
  Area.clr = c.table(i,1:3);
  
  % here it is assumed that the maximal color value is 255.
  % 2012-02-20 taku-y
  Area.clr = (Area.clr)./255;
  
  vb_add_area(save_areafile,Area);
end

% load annotation file for rh
[v,l,c] = read_annotation(antr_file);

% make label data that of VBMEG brain model for lh
[tmp,ix1,ix2] = intersect(BV_index.Right,v+Ileft0+1);
l = l(ix2);
l(ix1) = l;

%
% make area for rh
%
for i=1:c.numEntries
  ix = find(l==c.table(i,5))+Ileft;
  if ~isempty(strfind(c.struct_names{i}, 'FreeSurfer_Defined_Medial_Wal')) || ...
      strcmpi(c.struct_names{i}, 'Unknown') || ...
      isempty(ix)
      continue;
  end
  Area.key = [prefix_r c.struct_names{i}];
  Area.Iextract = ix;
  Area.clr = c.table(i,1:3);
  
  % here it is assumed that the maximal color value is 255.
  % 2012-02-20 taku-y
  Area.clr = (Area.clr)./255;

  vb_add_area(save_areafile,Area);
end

%
% make act file(.act.mat)
%
keyset = vb_get_keyset_area(save_areafile);
load(save_areafile,'Area');

act = struct;
act.key   = atlas_id;
act.xxP   = zeros(Nvertex, 1);
act.label = 1:length(keyset);
act.label_name = keyset;
for k=1:length(act.label)
    act.xxP(Area{k}.Iextract) = act.label(k); % set area number to xxP
end
vb_add_act(save_atlasfile, act, [], false);

fprintf('Save the area file as "%s" \n', save_areafile);
fprintf('Save the  act file as "%s"  \n', save_atlasfile);

fprintf('# of all vertex     = %d\n',Nvertex);
fprintf('# of cortex points  = %d\n',Ncortex);
fprintf('# of areas          = %d\n',length(keyset));



return;