function megmat_file = neuromag_to_megmat(data_fiffile, mri_fiffile, ...
                        output_prefix, face_file)
% make MEG-MAT file from neuromag fiffile.
% [USAGE]
%    neuromag_to_megmat(data_fiffile, mri_fiffile, ...
%                       output_prefix, face_file, sensor_mode)
% [IN]
%      data_fiffile : Data fiffile.
%       mri_fiffile : MRI fiffile.
%     output_prefix : output filename prefix.
%                     (ex. '/home/RH/RH' => /home/RH/RH.meg.mat)
% [optional input]
%         face_file : If you want to calculate by using spherical model,
%                     center position and radius of spherical model are 
%                     necessary.
%                     A face file which is made by positioning tool has them.
%                     For face file, please refer to the manual of the
%                     positioning tool
% [OUT]
%    meg.mat file with output_prefix is saved
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('data_fiffile', 'var')
    error('data_fiffile is a required parameter.');
end
if ~exist('mri_fiffile', 'var')
    error('mri_fiffile is a required parameter.');
end
if ~exist('output_prefix', 'var')
    error('output_prefix is a required parameter.');
end
if ~exist('face_file', 'var')
    face_file = [];
end
if exist(data_fiffile, 'file') ~= 2
    error('data_fiffile:%s is not found.', data_fiffile);
end
if exist(mri_fiffile, 'file') ~= 2
    error('mri_fiffile:%s is not found.', mri_fiffile);
end

[f_path] = vb_get_file_parts(output_prefix);
if exist(f_path, 'dir') ~= 7
    error('output_prefix directory:%s is invalid.', f_path);
end
if ~exist('sensor_mode','var'), sensor_mode = 'All'; end

%
% --- Main Procedure
%


%
% --- Load Header
%
fprintf('--- now loading neuromag header.\n');
[MEGinfo_list, sensor_info] = neuromag_load_meg_header(data_fiffile, ...
                                                       mri_fiffile);%, ...
%                                                       analyzefile);

%
% --- Extra channel info
%
[extra_info] = neuromag_load_extra_info(data_fiffile);

%
% --- Make sphere model
%
if ~isempty(face_file)

	if exist(face_file,'file') ~= 2
    warning('face_file:%s is not found.\n', face_file);
%     error('face_file:%s is not found.', face_file);
  end
  
  [state, guide_def] = ...
    vb_util_check_variable_in_matfile(face_file, 'spherical_head');

  if state == guide_def.VALID
    load(face_file, 'spherical_head');
    Vcenter = spherical_head.Vcenter;
    Vradius = spherical_head.Vradius;
  else
    [state, guide_def] = ...
      vb_util_check_variable_in_matfile(face_file, 'surf_face');
    if state == guide_def.VALID
      % --- calculate center position and radius of spherical model
      % --- Upper face is used for fitting sphere model 
      %	--- ix = find(V(:,3) > (h * Zmax + (1-h) * Zmin));
      h = 0.4;
    	load(face_file, 'surf_face')
	
    	V = surf_face.V_reduce;
    	Zmax = max(V(:,3));
    	Zmin = min(V(:,3));
    	ix = find(V(:,3) > (h * Zmax + (1-h) * Zmin));
	
    	% Fit face to sphere 
    	[Vcenter,Vradius] = vb_center_sphere(V(ix,:));
    else
    	Vcenter = [];
    	Vradius = [];
    end
  end
    
else
	Vcenter = [];
	Vradius = [];
end

pick   = sensor_info.pick;
Qpick  = sensor_info.Qpick;
weight = sensor_info.Weight;

Ninfo = length(MEGinfo_list);

%
% --- Make MEG-MAT file(s).
%
fprintf('--- now loading neuromag data.\n');
for k=1:Ninfo
  % -- Make file name
  if Ninfo == 1
    megmat_file{1} = [output_prefix, '.meg.mat'];
  else
    megmat_file{k} = [output_prefix, '_', num2str(k), '.meg.mat'];
  end
	
	% Set MEGinfo
  MEGinfo = MEGinfo_list{k};
    
  session_no = MEGinfo.device_info.index_in_fiffile;

  % load data 
  [bexp, ch_info, sfreq] = neuromag_load_meg_data_SSP(data_fiffile, session_no);
  [bexp_ext, extra_info] = neuromag_load_extra_data(data_fiffile, session_no);

  if ~isempty(Vcenter)
	  MEGinfo.Vcenter = Vcenter;
	  MEGinfo.Vradius = Vradius;
  end
  
  MEGinfo.Nsample    = size(bexp, 2);
  MEGinfo.SampleFreq = sfreq;

  % ----- MEGinfo.Trial
  Trial(1).number = 1;
  Trial(1).sample = 1:MEGinfo.Nsample;
  Trial(1).Active = true;
  MEGinfo.Trial = Trial;
	
  MEGinfo.sensor_weight = weight;
  MEGinfo.ExtraChannelInfo = extra_info;

  MEGinfo.device   = 'NEUROMAG';
  CoordType   = 'SPM_Right_m';
    
  Measurement = 'MEG';

  MEGinfo.Measurement   = Measurement;
  MEGinfo.ActiveChannel = MEGinfo.ChannelInfo.Active;
  MEGinfo.ActiveTrial   = MEGinfo.Trial.Active;

  
  
  % save MEG-MAT file
  vb_fsave(megmat_file{k}, 'MEGinfo', 'Measurement', 'CoordType', ...
  	 'pick', 'Qpick', 'bexp', 'bexp_ext');
  fprintf('%s was created.\n', megmat_file{k});
end
fprintf('--- conversion were finished.\n');
