function [ch_info,pos_info] = neuromag_load_ch_info(fiffile)
% load channel information from data-fiffile
% [USAGE]
%    [ch_info] = neuromag_load_ch_info(<fiffile>);
% [IN]
%    fiffile : neuromag data fiffile.
% [OUT]
%    ch_info : includes channel information and transform matrix
%             .Nch      [double]
%                 : The number of channels.
%             .channel_ix   [Nch x 1 double]
%                 : MEG channel name list.
%             .sensort_type [Nch x 1 double]
%                 : sensor type list
%             .coil_type    {Nch x 1 cell}
%                 : coil type string list
%                   ex. coil_type{1} = 'VV_PLANAR_T1'
%             .trans_coil2head   : {Nch x 1}
%                 : transformation matrix (COIL  ==> HEAD)
%                   trans_coil2head{n} = [4x4 double]
%             .trans_coil2device : {Nch x 1}
%                 : transformation matrix list (COIL  ==> DEVICE)
%                   trans_coil2device{n} = [4x4 double]
%             .trans_device2head : [4 x 4 double]
%                 : transformation matrix (DEVICE  ==> HEAD)
%
%    pos_info : fitting points in head coordinate
%             .fiducial : fiducial points (LPA, Nasion, RPA)
%             .hpi : points used to fitting with MEG
%             .head_point : head points used to fitting with MRI
%             .eeg : EEG sensor position
%
%  [rHead 1]   = [rCoil 1] * ch_info.trans_coil2head{n}'
%  [rDevice 1] = [rCoil 1] * ch_info.trans_coil2device{n}'
%  [rHead 1]   = [rDevise 1] * ch_info.trans_device2head{n}'
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)
%
% 2012/11/15 M.Fukushima
% * modified to import Henson dataset

%
% --- Previous check
%
if ~exist('fiffile', 'var') || exist(fiffile, 'file') ~=2
    error('fiffile is invalid.');
end

%
% --- Get Channel Info
%

% NUMBER : Channel ID numbers     [n x 1]
%   NAME : Channel name matrix    [n x m]
%   NAME(n,:) is a strings like  'MEG 0111'
%      T : The transformation between 'COIL' and either
%          'HEAD' or 'DEVICE' (@see MEGMODEL)
%          {n x 1} : T{n} = [4x4 double]
%   GRAD :
%        : 0 for magnetometers, 1 for planar gradiometers [n x 1]
%   TYPE : Names of the coil types {n x 1} like 'VV_PLANAR_T1'
%  NOISE : Weighting factor for the coil types [n x 1]

info = fiff_read_meas_info(fiffile);

% Get info
ix = strmatch('MEG', info.ch_names);
NAME = [];
T  = cell(0);     % coil information in head coordinate
T2 = cell(0); % coil information in device coordinate
GRAD = [];
TYPE = [];
NUMBER = [];

for k=1:length(ix)
    NUMBER   = [NUMBER; info.chs(k).logno];
    NAME     = [NAME; info.ch_names{ix(k)}];
    T{k, 1}  = info.dev_head_t.trans * info.chs(k).coil_trans;
    T2{k, 1} = info.chs(k).coil_trans;
    GRAD     = [GRAD; neuromag_get_sensor_kind(info.chs(k).coil_type)];
    TYPE     = [TYPE; info.chs(k).coil_type];
end

%[NOISE]           = chaninfo('noise');

% Get fitting point info in head coordinate
X     = [info.dig.r];
xtype = [info.dig.kind]';

% make structure of ch_info
ch_info.Nch         = length(ix);
ch_info.sensor_type = GRAD;
ch_info.coil_type   = TYPE; % ch_type; % TYPE;
ch_info.trans_coil2device = T2;
ch_info.trans_coil2head = T;
ch_info.trans_device2head  = info.dev_head_t(1).trans;
ch_info.channel_name = NAME;
%ch_info.noise = NOISE;

% Nch = length(NUMBER);
Nch = ch_info.Nch;
ch_info.channel_ix  = cell(Nch,1);
for n=1:Nch
	ch_info.channel_ix{n}  = deblank(num2str(NUMBER(n)));
end

% Set fitting point info
% xtype: (1=cardinal, 2=HPI, 3=EEG, 4=extra)
ix = find(xtype == 1);
pos_info.fiducial = X(:,ix)';
ix = find(xtype == 2);
pos_info.hpi = X(:,ix)';
ix = find(xtype == 3);
pos_info.eeg = X(:,ix)';
ix = find(xtype == 4);
pos_info.head_point = X(:,ix)';
