function [AVWVIEW] = vb_mri_point_select_display_layer(AVWVIEW, cut, ...
                                                    slice, space_name)
% plots element data on vb_mri_point_select slice.
% This function is called from mri_point_select.
% If you want to show the data, please reffer to Example.
%
% [USAGE]
%    [AVWVIEW] = vb_mri_point_select_display_layer(<AVWVIEW>, <cut>, <slice>);
% [IN]
%       AVWVIEW : <<struct>> vb_mri_point_select data
%           cut : update axes.
%                 'x' : plot sagittal axes
%                 'y' : plot coronal axes
%                 'z' : plot axial_axes
%         slice : slice number to plot
%    space_name : element will be plotted to this space.
%                 'VOXEL' : voxel coordinate space
%                 'MM'    : mm coordinate space
%                 'M'     : m coordinate space
% [OUT]
%    AVWVIEW : <<struct>> updated vb_mri_point_select data.
%
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('AVWVIEW', 'var'), error('AVWVIEW is a required parameter.'); end
if ~exist('cut', 'var'), error('cut is a required parameter.'); end
if ~exist('slice', 'var'), error('slice is a required parameter.'); end

if ~isfield(AVWVIEW, 'display_factor'), return; end

%
% --- Main Procedure
%
global vbmeg_inst;
const = vbmeg_inst.const;

display_factor = AVWVIEW.display_factor;
Mtype = vb_display_factor_get_element_color(display_factor);

Vdim  = size(AVWVIEW.avw.img);
Vsize = AVWVIEW.pixdim;

fig = figure(AVWVIEW.gui);

% Get current Axes
previous_focus = get(fig, 'CurrentAxes');

% Change current Axes
switch(cut)
    case 'x'
        set(fig, 'CurrentAxes', AVWVIEW.handles.sagittal_axes);
    case 'y'
        set(fig, 'CurrentAxes', AVWVIEW.handles.coronal_axes);
    case 'z'
        set(fig, 'CurrentAxes', AVWVIEW.handles.axial_axes);
end

% Previous display delete
AVWVIEW = delete_previous_display(AVWVIEW, cut);

% Display layer
Nelements = vb_display_factor_get_length(display_factor);
for k=1:Nelements
    element = vb_display_factor_get_element(display_factor, k);

    %
    % --- Marker setting
    %
    Mark = Mtype{k};
    color = [];
    % separate Color from Marker type
    for k=1:length(Mark)
        if isletter(Mark(k)), color = strcat(color, Mark(k)); end
    end
    
    switch(element.class_name)
        case 'surf_element'
            %
            % --- Plot surface data
            %
            coord_type = vb_display_element_get_coord_type(element);
            if strcmp(coord_type, const.COORDINATE_ANALYZE_RIGHT_MM)
                Vana = vb_analyze_mm_to_analyze(...
                            vb_surf_element_get_vertex(element), Vdim, Vsize);
            else
                error('unknown coordinate data');
            end
            hold on;

            % get points on specified slice to plot
            LineWidth = 1;
            Msize = LineWidth;

            [X, Y, X1, Y1, X2, Y2, X3, Y3] = vb_get_cross_section(...
                                                Vana, ...
                                                vb_surf_element_get_face(element), ...
                                                                slice, cut, 15);
            % plot points
            if strcmp(space_name, 'MM') || strcmp(space_name, 'M')
                if cut == 'x'
                    % Sagittal(plotX == sliceY, plotY == sliceZ)
                    X = X * Vsize(2);
                    Y = Y * Vsize(3);
                    X1 = X1 * Vsize(2);
                    Y1 = Y1 * Vsize(3);
                    X2 = X2 * Vsize(2);
                    Y2 = Y2 * Vsize(3);
                    X3 = X3 * Vsize(2);
                    Y3 = Y3 * Vsize(3);
                elseif cut == 'y'
                    % Coronal(plotX == sliceX, plotY == sliceZ)
                    X = X * Vsize(1);
                    Y = Y * Vsize(3);
                    X1 = X1 * Vsize(1);
                    Y1 = Y1 * Vsize(3);
                    X2 = X2 * Vsize(1);
                    Y2 = Y2 * Vsize(3);
                    X3 = X3 * Vsize(1);
                    Y3 = Y3 * Vsize(3);
                elseif cut == 'z'
                    % Axial(plotX == sliceX, plotY == sliceY);
                    X = X * Vsize(1);
                    Y = Y * Vsize(2);
                    X1 = X1 * Vsize(1);
                    Y1 = Y1 * Vsize(2);
                    X2 = X2 * Vsize(1);
                    Y2 = Y2 * Vsize(2);
                    X3 = X3 * Vsize(1);
                    Y3 = Y3 * Vsize(2);
                end
                if strcmp(space_name, 'M')
                    X = X / 1000;
                    Y = Y / 1000;
                end
            end
            h = plot(X, Y, Mark, 'Markersize', Msize, 'LineWidth', LineWidth);
            if ~isempty(X1), h = [h; plot(X1, Y1, [color, '.'], 'Markersize', Msize)]; end
            if ~isempty(X2), h = [h; plot(X2, Y2, [color, '.'], 'Markersize', Msize)]; end
            if ~isempty(X3), h = [h; plot(X3, Y3, [color, '.'], 'Markersize', Msize)]; end

        case 'mask_image_element'
            hold on;
            %
            % --- Plot mask image data
            %

            % get points on specified slice to plot
            [X,Y] = vb_get_3d_mask_point(...
                            vb_mask_image_element_get_voxel_data(element), ...
                            slice, cut, Vdim);
            if strcmp(space_name, 'MM') || strcmp(space_name, 'M')
                if cut == 'x'
                    % Sagittal(plotX == sliceY, plotY == sliceZ)
                    X = X * Vsize(2);
                    Y = Y * Vsize(3);
                elseif cut == 'y'
                    % Coronal(plotX == sliceX, plotY == sliceZ)
                    X = X * Vsize(1);
                    Y = Y * Vsize(3);
                elseif cut == 'z'
                    % Axial(plotX == sliceX, plotY == sliceY);
                    X = X * Vsize(1);
                    Y = Y * Vsize(2);
                end
                if strcmp(space_name, 'M')
                    X = X / 1000;
                    Y = Y / 1000;
                end
            end
            % plot points
            h = plot(X, Y, [color, '.'], 'Markersize',5);
    end

    % save plot component handles
    AVWVIEW = add_current_display(AVWVIEW, cut, h);
end

set(fig, 'CurrentAxes', previous_focus);

if nargout ~= 1
    error('function caller should receive AVWVIEW');
end

function AVWVIEW = delete_previous_display(AVWVIEW, cut)
% delete plotted components
% [IN]
%    AVWVIEW : <<struct>> AVWVIEW
%        cut : specify which component on axes
%              'x' : plot sagittal axes
%              'y' : plot coronal axes
%              'z' : plot axial_axes
% [OUT]
%    AVWVIEW : updated AVWVIEW
%
switch(cut)
    case 'x'
        if ~isfield(AVWVIEW, 'sagittal_layer')
            return;
        else
            layer_cell = AVWVIEW.sagittal_layer;
            AVWVIEW = rmfield(AVWVIEW, 'sagittal_layer');
        end
    case 'y'
        if ~isfield(AVWVIEW, 'coronal_layer')
            return;
        else
            layer_cell = AVWVIEW.coronal_layer;
            AVWVIEW = rmfield(AVWVIEW, 'coronal_layer');
        end
    case 'z'
        if ~isfield(AVWVIEW, 'axial_layer')
            return;
        else
            layer_cell = AVWVIEW.axial_layer;
            AVWVIEW = rmfield(AVWVIEW, 'axial_layer');
        end
end

Nlayer = length(layer_cell);
for k=1:Nlayer
    delete(layer_cell{k});
end

if nargout ~= 1
    error('function caller should receive AVWVIEW.');
end

function AVWVIEW = add_current_display(AVWVIEW, cut, h)
% add plotted components to AVWVIEW
% [IN]
%    AVWVIEW : <<struct>> AVWVIEW
%        cut : specify which component on axes
%              'x' : plot sagittal axes
%              'y' : plot coronal axes
%              'z' : plot axial_axes
%          h : added component handles
% [OUT]
%    AVWVIEW : updated AVWVIEW
%
switch(cut)
    case 'x'
        if ~isfield(AVWVIEW, 'sagittal_layer')
            AVWVIEW.sagittal_layer = cell(0);
        end
        Nlayer = length(AVWVIEW.sagittal_layer);
        AVWVIEW.sagittal_layer{Nlayer+1} = h;
    case 'y'
        if ~isfield(AVWVIEW, 'coronal_layer')
            AVWVIEW.coronal_layer = cell(0);
        end
        Nlayer = length(AVWVIEW.coronal_layer);
        AVWVIEW.coronal_layer{Nlayer+1} = h;
    case 'z'
        if ~isfield(AVWVIEW, 'axial_layer')
            AVWVIEW.axial_layer = cell(0);
        end
        Nlayer = length(AVWVIEW.axial_layer);
        AVWVIEW.axial_layer{Nlayer+1} = h;
end

if nargout ~= 1
    error('function caller should receive AVWVIEW.');
end
