function [Jest, F, Info] = wmn_solution_fixreg(B, G, X, w, lambda, option)
% Weighted L2 Minimum Norm solution for the multiple regularization parameters
% Require huge memory when length(lambda) is large.
%  
% Solve the following minimization problem  
%   min {E(J) = (B-GJ)'X(B-GJ) + lam*J'WJ}  
% with optimization of 'lam' by ABIC or GCV. 
% When 'option.lapacian' exists, the following minimization problem
%   min {E(J) = (B-GJ)'X(B-GJ) + lam*J'LWLJ}  
% is solved ('LORETA' solution).
%  
% [Syntax] 
%   > [lambda, sigma2, ABIC, Jest, F, Info] = wmn_solution_fixreg(B, G, Ceps, W, lambda, option)
%
% [Example]
% >> option.criteria = 'ABIC'
% >> option.flag_current = 1;
% >> [lambda, sigma2, ABIC, Jest, F] = wmn_solution(B, G, Ceps, W, lambda, option)
%
% returns optimal parameters, estimates of J, and the inverse filter using ABIC 
% as the statisitcal criteria for optimization.
%
% [Output]
%   Jest : estimated J   
%   F    : inverse filter (mapping from B to J) 
%   Info : information of the optimization 
%        1st column : lambda
%        2nd column : criteria (ABIC or GCV)
%        3rd column : observation noise variance (SIG2)
%
% [Input]
%   B : EEG/MEG observation of a single trial (Nch*Nt matrix)
%   G : lead field matrix  (Nch * Nj matrix)
%   X : observation weight matrix (inverse of observation noise)
%   w : the diagonal elements of prior weight matrix W (should be diagonal, because of memory problem) 
%  lambda : the regularization parameters 
%
% [Optional Input]
%   option.criteria : criteria for the estimation of reguralization parameter 
%   option.laplacian : if exist, W is replaced by LWL (L is lapalcian)
%   option.flag_plot : if 1, the values of 'criteria' (ABIC or GCV) are displayed with respect to 'lambda'  
%   option.Npoints : the number of grid points in lambda axis to be explored (default:100) 
%   option.search_margin : parameter to define range of lambda to be searched (default:5).
%
% 2016/05/02 O.Yamashita
% * fixed regularization paremeters 
% * return Jest for the multiple regurlarization parameters


[Nch, Nt] = size(B);
[Nj] = size(G,2);


% default 
criteria = 'ABIC';
flag_plot = 1;
L = speye(Nj);
Npoints = 100;   % Number of points on the curve.
search_margin = 5;  % in log scale

%% Error check 
if exist('option') == 1  & isfield(option, 'criteria')
    criteria = option.criteria;
end
if exist('option') == 1  & isfield(option, 'flag_plot')
    flag_plot = option.flag_plot ;
end
if exist('option') == 1  & isfield(option, 'laplacian')
    L = option.laplacian;
    L = L / max(L(:));
end
if exist('option') == 1  & isfield(option, 'Npoints')
    Npoints = option.Npoints;
end
if exist('option') == 1  & isfield(option, 'search_margin')
    search_margin = option.search_margin;
end

% pre-whitening 
if isempty(X)
    Xsqrt = eye(Nch);
else
    Xsqrt = sqrtm(X);
end
if isempty(w)
    wsqrt = ones(Nj,1);
else
    wsqrt = sqrt(w);
end

Btil = Xsqrt * B; 
Gtil = Xsqrt * G / L * spdiags(1./wsqrt, 0, Nj, Nj) ;

[U s V] = csvd1(Gtil);

% Sets grid points
if isempty(lambda)
    minloglx = log(s(end)) - search_margin;
    maxloglx = log(s(1)) + search_margin;
    bin = (maxloglx-minloglx)/(Npoints-1);
    loglx_points = [minloglx : bin : maxloglx];
else
    loglx_points = log(lambda);   % fixed lambda
end

Npoints = length(loglx_points); 
SIG2 = zeros(Npoints,1);
CRITERIA = zeros(Npoints,1);

UtB = U'*Btil;
BB = UtB * UtB';
s2=s.*s; 

% compute Jest and F for all the given regularization parameters
Jest = zeros(Nj,Nt,length(lambda));
F    = zeros(Nj,Nch,length(lambda));

switch criteria
    case 'ABIC'
        %% calculate ABIC
        for i = 1 : Npoints
          lam = exp(loglx_points(i));         
          l2=lam;
          r = l2./(s2+l2); 
          sig2 = sum(sum(diag(r).*BB))/Nch/Nt; %tr(diag(r)*BB)
          ABIC1=Nch*Nt*log(sig2);
          ABIC2= -Nt*sum(log(r));
          CRITERIA(i)=ABIC1+ABIC2;
          SIG2(i) = sig2;
          
          % Jest and inverse filters
          rs = s./(s2+l2); 
          Jtil = V *diag(rs)*UtB;
          Jest(:,:,i) = L\ Jtil ./ wsqrt(:,ones(1,Nt));  %Jest = L\ spdiags(1./wsqrt, 0, Nj, Nj) * Jtil;
          if nargout >=2, F(:,:,i) = L\spdiags(1./wsqrt, 0, Nj, Nj)*V*diag(rs)*U'*Xsqrt; end
        end
        
    case 'GCV'
        %% calculate GCV
        for i = 1 : Npoints
          lam = exp(loglx_points(i));         
          l2=lam;
          r = l2./(s2+l2);
          nume = sum(sum(diag(r.^2).*BB));
          denom = (sum(r))^2;
          GCV = nume/denom;
          sig2 = sum(sum(diag(r).*BB))/Nch/Nt; %tr(diag(r)*BB)
          CRITERIA(i) = GCV;
          SIG2(i) = sig2;
          
          % Jest and inverse filters
          rs = s./(s2+l2); 
          Jtil = V *diag(rs)*UtB;
          Jest(:,:,i) = L\ Jtil ./ wsqrt(:,ones(1,Nt));  %Jest = L\ spdiags(1./wsqrt, 0, Nj, Nj) * Jtil;
          if nargout >= 2, F(:,:,i) = L\spdiags(1./wsqrt, 0, Nj, Nj)*V*diag(rs)*U'*Xsqrt; end
          
        end
        
    otherwise
        disp('No such a method');
end


Info = [lambda(:) CRITERIA(:) SIG2(:)];


if flag_plot
    figure
    [AX,H1,H2] = plotyy(loglx_points, CRITERIA, loglx_points, SIG2);
    xlabel('Log of lambda');
    set(get(AX(1),'Ylabel'),'String','ABIC or GCV')
    set(get(AX(2),'Ylabel'),'String','Observation Noise Variance');
    hold on;
    plot([loglx_points(ind) loglx_points(ind)], [min(get(AX(1),'ytick')) max(get(AX(1),'ytick'))], 'k--');
end
    


